/*
 * cpu-debug.cpp
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// cpu-debugger
// by Matt Ownby
// designed to interface with MAME cpu cores
// some of the code in here was inspired by and/or taken from Marat Fayzullin's z80 debugger

// This should be undefined by default.
// The MAME z80 debugger can be compiled into daphne, but binaries must not be distributed due to GPL
// and MAME licensing conflicts.  In other words, if you want to use this, you have to compile it yourself.
// MAME might be switching over to GPL though, which would make things a lot easier in
// the future.
//#define USE_MAME_Z80_DEBUGGER

#ifdef CPU_DEBUG

#ifdef USE_MAME_Z80_DEBUGGER
unsigned DasmZ80( char *buffer, unsigned pc );
#endif

#include <stdio.h>
#include <ctype.h>
#include "mamewrap.h"
#include "../io/conout.h"
#include "../io/conin.h"
#include "../daphne.h"
#include "../daphne.h"
#include "../game/game.h"
#include "../ldp-out/ldp.h"
#include "cpu.h"
#include "cpu-debug.h"
#include "../cpu/generic_z80.h"

unsigned char g_cpu_trace = 0;	// whether we are stopping at each instruction
UINT16	g_breakpoint = 0;	// address to break at
unsigned char g_break = 0;	// whether to break at breakpoint or not

// function used to set the value of g_cpu_trace to avoid extern statements
void set_cpu_trace(unsigned char value)
{
	g_cpu_trace = value;
}

// I called this MAME_Debug so that I could test mame cpu cores with daphne (obviously I can't ship mame cpu cores with
// daphne due to licensing issues)
void MAME_Debug(void)
{	
	// if we are in trace mode OR if we've got our desired breakpoint
	if (g_cpu_trace || (g_break && (Z80_GET_PC == g_breakpoint)))
	{
		debug_prompt();	// give them a prompt
	}
}

// called by debugger to give us the opportunity to translate a hex address into a label
// (for easier reading)
// Actually, this is a rather brilliant idea, props go to Juergen or whoever thought it up
// 'what' is either 0 or 1, not sure the difference
// size corresponds to enumerations 
// acc (access) also corresponds to enumerations
const char *set_ea_info( int what, unsigned address, int size, int acc )
{
	static char addrstr[160] = { 0 };	// must be static so it isn't de-allocated upon returning
	char *name = NULL;

	switch(acc)
	{
		case EA_NONE:
		case EA_VALUE:
			sprintf(addrstr, "%x", address);
			break;
		case EA_ABS_PC:
		case EA_MEM_RD:
		case EA_MEM_WR:
			name = g_game->get_address_name ( (Uint16) address);
			// if memory location has a name, print that too
			if (name)
			{
				sprintf(addrstr, "%s ($%x)", name, address);
			}
			else
			{
				sprintf(addrstr, "$%x", address);
			}
			break;
		case EA_REL_PC:
			address += size;	// relative offset, address address
			name = g_game->get_address_name ( (Uint16) address);

			if (name)
			{
				sprintf(addrstr, "%s ($%x) (PC + %d)", name, address, size);
			}
			else
			{
				sprintf(addrstr, "$%x (PC + %d)", address, size);
			}
			break;
		default:
			sprintf(addrstr, "Unknown: acc %d, size %d, address %u, what %d", acc, size, address, what);
			break;
	}

	return(addrstr);
}

void debug_prompt()
{
	char s[81] = { 0 };
	UINT8 done = 0;
	unsigned int addr = Z80_GET_PC;	// this function relies on addr being initialized to PC
	
	g_break = 0;	// once they get  here, don't break anymore
	
	// they have to issue a proper command to get out of the prompt
	while (!done && !get_quitflag())
	{
		newline();
		print_cpu_context();	// show registers and stuff
		
		sprintf(s, "[%04x Command,'?']-> ", Z80_GET_PC);
		outstr(s);
		con_getline(s, 80);
		switch(toupper(s[0]))
		{
		case 'S':
			// this might help with debugging *shrug*
			g_ldp->pre_step_forward();
			printline("Stepping forward one frame...");
			break;
		case 'C':	// continue execution
				g_cpu_trace = 0;
				flush_cpu_timers(); // if we don't do this, interrupts will be too fast for a sec
				done = 1;
				break;
			case 'D':	// disassemble

				// if they entered in an address to disassemble at ...
				if (strlen(s) > 1)
				{
					addr = (unsigned int) strtol(&s[1], NULL, 16);	// convert base 16 text to a number
				}
				// else if they entered no parameters, disassemble from PC

				debug_disassemble(addr);
				break;
			case 'I':	// break at end of interrupt
				printline("This feature not implemented yet =]");
				break;
			case '=':	// set a new breakpoint
				// if they entered in an address to disassemble at ...
				if (strlen(s) > 1)
				{
					g_breakpoint = (UINT16) strtol(&s[1], NULL, 16);	// convert base 16 text to a number
					g_break = 1;
					g_cpu_trace = 0;
					flush_cpu_timers();	// if we don't do this, interrupts will be too fast for a second
					done = 1;
				}
				// else if they entered no parameters, disassemble from PC
				else
				{
					printline("You must specify an address to break at.");
				}
				break;
			case 'A':
				if (strlen(s) > 1)
				{
					addr = (unsigned int) strtol(&s[1], NULL, 16);
				}

				Z80_SET_AF((Uint16) addr);
				break;
			case 'M':	// memory dump

				if (strlen(s) > 1)
				{
					addr = (unsigned int) strtol(&s[1], NULL, 16);
				}

				print_memory_dump(addr);
				break;
			case 0:	// carriage return
				g_cpu_trace = 1;
				done = 1;
				break;
			case 'Q':	// quit emulator
				set_quitflag();
				break;
			case '?':	// get menu
				debug_menu();
				break;
			default:
				printline("Unknown command, press ? to get a menu");
				break;
		}
	} // end while
}

void debug_menu()
{
	newline();
	printline("CPU Debugger Commands");
	printline("---------------------");
	printline("<CR>     : break at next instruction");
	printline("c        : continue without breaking");
	printline("i        : break at end of interrupt");
	printline("d <addr> : disassembly at address");
	printline("m <addr> : memory dump at address");
	printline("= <addr> : break at address");
	printline("q        : quit emulator");
	printline("?        : this menu");
}

// print a disassembly starting from addr
void debug_disassemble(unsigned int addr)
{
	char s[160] = { 0 };
	int line = 0;
	char *name = NULL;

	// print 14 lines because that's how many our console can show at a time
	while (line < 14)
	{
		name = g_game->get_address_name( (Uint16) addr); // check to see if this address has a name
		// if so, it's a label name, so print it
		if (name)
		{
			outstr(name);
			printline(":");
			line++;
		}
		sprintf(s, "%04x: ", addr);
		outstr(s);
#ifdef USE_MAME_Z80_DEBUGGER
		addr += DasmZ80(s, addr);
#endif
		printline(s);
		line++;
	}

}

// prints cpu registers and flags (keep it on 2 lines)
void print_cpu_context()
{

	char tmpstr[160] = { 0 };
	static char all_flags[9] = "SZ5H3PNC";
	char flags[10] = { 0 };
	unsigned char i = (unsigned char) (Z80_GET_AF & 0xFF);
	unsigned char j = 0;
	char nextinstr[160];

#ifdef USE_MAME_Z80_DEBUGGER
	DasmZ80( nextinstr, Z80_GET_PC );
#endif

	for(; j < 8 ; j++ , i = (unsigned char) (i << 1))
	{
		if (i & 0x80)
		{
			flags[j] = all_flags[j];
		}
		else
		{
			flags[j] = '.';
		}
	}

	sprintf(tmpstr, 
    "AF:%04X HL:%04X DE:%04X BC:%04X PC:%04X SP:%04X IX:%04X IY:%04X",
    Z80_GET_AF,
    Z80_GET_HL,
    Z80_GET_DE,
    Z80_GET_BC,
	Z80_GET_PC,
	Z80_GET_SP,
	Z80_GET_IX,
	Z80_GET_IY
	); 
	printline(tmpstr);

	sprintf(tmpstr,
    "AT PC: [%02X - %s]   FLAGS: [%s]  IFF1: %x IFF2: ? IM: ?",
    Z80_GET_PC,
	nextinstr,
	flags,
	Z80_GET_IFF1
	);
	printline(tmpstr);
}

// print a memory dump (duh)
void print_memory_dump(unsigned int addr)
{
	int i = 0, j = 0;	// temporary indices
	char tmpstr[160] = { 0 };
	const char lines = 14;	// # of lines in the memory dump

	newline();

	for(j=0 ; j < lines ; j++)
	{
		sprintf(tmpstr, "%04X: ", addr);
		outstr(tmpstr);

		for(i=0; i < 16; i++,addr++)
		{
			sprintf(tmpstr, "%02X ", cpu_readmem16(addr));
			outstr(tmpstr);
		}

		outstr(" | ");

		addr-=16;
		for(i=0;i < 16;i++,addr++)
		{
			outchr(isprint(cpu_readmem16(addr))? cpu_readmem16(addr):'.');
		}

		newline();
	}
}

#endif
// end #ifdef CPU_DEBUG
