/*
 * cpu.cpp
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// cpu
// by Matt Ownby
// Designed to do all of the universal CPU functions needed for the emulator

#include "cpu.h"
#include <stdio.h>	// for stderr
#include <string.h>	// for memcpy
#include "../daphne.h"
#include "../game/game.h"
#include "../timer/timer.h"
#include "../io/input.h"
#include "../io/conout.h"
#include "6809infc.h"
#include "nes6502.h"
#include "nes_6502.h"
#include "mamewrap.h"
#include "generic_z80.h"
#include "cop.h"

struct cpudef *g_head = NULL;	// pointer to the first cpu in our linked list of cpu's
unsigned char g_cpu_count = 0;	// how many cpu's have been added
bool g_cpu_initialized[CPU_COUNT] = { false };	// whether cpu core has been initialized
Uint32 g_cpu_timer = 0;	// used to make cpu's run at the right speed
Uint32 g_expected_elapsed_ms = 0;	// how many ms we expect to have elapsed since last cpu execution loop
Uint8 g_active_cpu = 0;	// which cpu is currently active

// Uncomment this if you want to test the CPUs to make sure they are running at the proper speed
//#define CPU_DIAG 1

#ifdef CPU_DIAG
// how many cpu's cpu diag can support (this has nothing to do with how many cpu's daphne can support)
#define CPU_DIAG_CPUCOUNT 10
	static unsigned int cd_cycle_count[CPU_DIAG_CPUCOUNT];	// for speed test
	static unsigned int cd_old_time[CPU_DIAG_CPUCOUNT] = { 0 }; // " " "
	static unsigned int cd_irq_count[CPU_DIAG_CPUCOUNT][MAX_IRQS] = { { 0 } };
	static unsigned int cd_nmi_count[CPU_DIAG_CPUCOUNT] = { 0 };
	static double cd_compounded_mhz[CPU_DIAG_CPUCOUNT] = { 0.0 };	// sum of all mhz ratings so we can get an average
	static int cd_report_count[CPU_DIAG_CPUCOUNT] = { 0 };	// how many reports have been given
#endif

//////////////////////////////////////////////////////////////////////////////////

// adds a cpu to our linked list.  The data is copied, so you can clobber the original data after this call.
void add_cpu (struct cpudef *candidate)
{
	struct cpudef *cur = NULL;
	
	// if this is the first cpu to be added to the list
	if (!g_head)
	{
		g_head = new struct cpudef;	// allocate a new cpu, assume allocation is successful
		cur = g_head;	// point to the new cpu so we can populate it with info
	}
	// else we have to move to the end of the list
	else
	{
		cur = g_head;

		// go to the last cpu in the list
		while (cur->next_cpu)
		{
			cur = cur->next_cpu;
		}

		cur->next_cpu = new struct cpudef;	// allocate a new cpu at the end of our list
		cur = cur->next_cpu;	// point to the new cpu so we can populate it with info
	}

	// now we must copy over the relevant info
	memcpy(cur, candidate, sizeof(struct cpudef));	// copy entire thing over
	cur->id = g_cpu_count;
	g_cpu_count++;

	// now we must assign the appropriate callbacks
	switch (cur->type)
	{
	case CPU_Z80:
#ifdef USE_M80
		cur->init_callback = m80_reset;
		cur->shutdown_callback = NULL;
		cur->setmemory_callback = m80_set_opcode_base;
		cur->execute_callback = m80_execute;
		cur->getcontext_callback = m80_get_context;
		cur->setcontext_callback = m80_set_context;
		cur->setpc_callback = m80_set_pc;
		cur->elapsedcycles_callback = m80_get_cycles_executed;
		cur->reset_callback = m80_reset;
		m80_set_irq_callback(generic_m80_irq_callback); // this needs to be done here to allow games to override
		m80_set_nmi_callback(generic_m80_nmi_callback); // " " "
#endif
#ifdef USE_Z80
		cur->init_callback = z80_reset;
		cur->shutdown_callback = z80_exit;
		cur->setmemory_callback = mw_z80_set_mem;
		cur->execute_callback = z80_execute;
		cur->getcontext_callback = z80_get_context;
		cur->setcontext_callback = z80_set_context;
		cur->setpc_callback = z80_set_pc;
		cur->reset_callback = z80_reset;
		cur->elapsedcycles_callback = generic_cpu_elapsedcycles_stub;	// FIXME : write this function
		z80_set_irq_callback(generic_z80_irq_callback); // this needs to be called here to allow games to override
#endif
		break;
	case CPU_M6809:
		cur->init_callback = initialize_m6809;
		cur->shutdown_callback = NULL;
		cur->setmemory_callback = m6809_set_memory;
		cur->execute_callback = mc6809_StepExec;
		cur->getcontext_callback = NULL;
		cur->setcontext_callback = NULL;
		cur->setpc_callback = NULL;
		cur->reset_callback = m6809_reset;
		cur->elapsedcycles_callback = generic_cpu_elapsedcycles_stub;
		break;
	case CPU_M6502:
		cur->init_callback = generic_6502_init;
		cur->shutdown_callback = generic_6502_shutdown;
		cur->setmemory_callback = generic_6502_setmemory;
		cur->execute_callback = nes6502_execute;
		cur->getcontext_callback = generic_6502_getcontext;
		cur->setcontext_callback = generic_6502_setcontext;
		cur->setpc_callback = NULL;
		cur->reset_callback = generic_6502_reset;
		cur->elapsedcycles_callback = generic_cpu_elapsedcycles_stub;
		break;
	case CPU_COP421:
		cur->init_callback = cop421_reset;
		cur->shutdown_callback = NULL;
		cur->setmemory_callback = cop421_setmemory;
		cur->execute_callback = cop421_execute;
		cur->getcontext_callback = NULL;
		cur->setcontext_callback = NULL;
		cur->setpc_callback = NULL;
		cur->reset_callback = cop421_reset;
		cur->elapsedcycles_callback = generic_cpu_elapsedcycles_stub;
		break;
	default:
		fprintf(stderr, "FATAL ERROR : unknown cpu added\n");
		break;
	}
}

// de-allocate all cpu's that have been allocated
void del_all_cpus()
{
	struct cpudef *cur = g_head;
	struct cpudef *tmp = NULL;
	
	// while we have cpu's left to delete
	while (cur)
	{
		tmp = cur;
		cur = cur->next_cpu;
		delete tmp;	// de-allocate
	}
	g_head = NULL;
	g_cpu_count = 0;

}

// initializes all cpus
void cpu_init()
{
	struct cpudef *cur = g_head;
	
	while (cur)
	{
		g_active_cpu = cur->id;
#ifdef CPU_DIAG
		cd_old_time[g_active_cpu] = refresh_ms_time();
#endif

		cur->cycle_adjustment = 0;
		cur->cycles_per_ms = cur->hz / 1000.0;
		cur->cycles_per_nmi = cur->cycles_per_ms * cur->nmi_period;
		cur->nmi_cycle_count = 0;
		for (int i = 0; i < MAX_IRQS; i++)
		{
			cur->cycles_per_irq[i] = cur->cycles_per_ms * cur->irq_period[i];
			cur->irq_cycle_count[i] = 0;
		}
		cur->total_cycles_executed = 0;

		// if the cpu core has not been initialized yet, then do so .. it should only be done once per cpu core
		if (!g_cpu_initialized[cur->type])
		{
			(cur->init_callback)();	// initialize the cpu
			g_cpu_initialized[cur->type] = true;
		}
		(cur->setmemory_callback)(cur->mem);	// set where the memory is located
		
		// if we have an unexpected PC to begin on, set our PC
		if (cur->initial_pc != 0)
		{
			(cur->setpc_callback)(cur->initial_pc);	// set the initial program counter for the cpu
		}

		// if we are required to copy the cpu context, then get the info now
		if (cur->must_copy_context)
		{
			unsigned int context_size = (cur->getcontext_callback)(cur->context);

			// sanity check
			if (context_size > MAX_CONTEXT_SIZE)
			{
				fprintf(stderr, "FATAL ERROR : Increase MAX_CONTEXT_SIZE to at least %u and recompile\n", context_size);
				set_quitflag();
			}
		}
		
		cur = cur->next_cpu;	// advance to the next cpu

	} // end while
}

// shutdown all cpu's
void cpu_shutdown()
{
	struct cpudef *cur = g_head;
	
	// go through each cpu and shut it down
	while (cur)
	{
		g_active_cpu = cur->id;
		// if we have a shutdown callback defined
		if (cur->shutdown_callback)
		{
			// we only want to shutdown the cpu core once
			if (g_cpu_initialized[cur->type] == true)
			{
				(cur->shutdown_callback)();	// shutdown the cpu
				g_cpu_initialized[cur->type] = false;
			}
		}
		cur = cur->next_cpu; // move to the next cpu entry
	}
	
	del_all_cpus();
}

// executes all cpu cores "simultaneously".  this function only returns when the game exits
void cpu_execute()
{
	int i = 0;
	Uint32 last_inputcheck = 0; //time we last polled for input events

	flush_cpu_timers();				// so we we don't get flooded by NMI's/IRQ's when we first start

	// loop until the quit flag is set which means the user wants to quit the program
	while (!get_quitflag())
	{
		struct cpudef *cpu = g_head;
		unsigned int actual_elapsed_ms = 0;
		bool nmi_asserted = false;
		Uint32 elapsed_cycles = 0;
		double cycles_to_execute = 0.0;	// how many cycles to execute this time around

		// go through each cpu and execute 1 ms worth of cycles
		while (cpu)
		{
			// if we are required to copy the cpu context, then set the context for the current cpu
			if (cpu->must_copy_context)
			{
				(cpu->setcontext_callback)(cpu->context);	// restore registers
				(cpu->setmemory_callback)(cpu->mem);	// restore memory we're working with
			}
			g_active_cpu = cpu->id;
			
			nmi_asserted = false;

			cycles_to_execute = cpu->cycle_adjustment + cpu->cycles_per_ms;
			
			// if we have a meaningful # of cycles to execute this time around, then do so
			if (cycles_to_execute > 0.0)
			{
				elapsed_cycles = (cpu->execute_callback)((Uint32) cycles_to_execute);
				// execute ~1 ms worth of cycle

				cpu->cycle_adjustment = cpu->cycles_per_ms - elapsed_cycles;
				// calculate adjustment for next iteration because we probably won't hit our goal exactly
			
				cpu->total_cycles_executed += elapsed_cycles;	// always track how many cycles have elapsed
			}
			// else if we executed too many cycles last time, then we have to just kill time
			else
			{
				elapsed_cycles = 0;
				cpu->cycle_adjustment += cpu->cycles_per_ms;
			}

#ifdef CPU_DIAG
			cd_cycle_count[g_active_cpu] += elapsed_cycles;
#endif

			// add elapsed cycles to the interrupt timers
			for (i = 0; i < MAX_IRQS; i++)
			{
				cpu->irq_cycle_count[i] += elapsed_cycles;
			}
			cpu->nmi_cycle_count += elapsed_cycles;

			// NOW WE CHECK TO SEE IF IT'S TIME TO DO AN NMI
			// if NMI's are enabled
			// FIXME : find a faster way than comparing a double
			if (cpu->nmi_period > 0.0)
			{
				// if we need to do an NMI
				if (cpu->nmi_cycle_count > cpu->cycles_per_nmi)
				{
					cpu->nmi_cycle_count -= cpu->cycles_per_nmi;
					g_game->do_nmi();
					nmi_asserted = true;
#ifdef CPU_DIAG
					cd_nmi_count[cpu->id]++;
#endif
				}
			}

			// NOW WE CHECK TO SEE IF IT'S TIME TO DO AN IRQ

			// go through each IRQ
			for (i = 0; i < MAX_IRQS; i++)
			{
				// if irq period is 0.0 it means there is no IRQ
				// FIXME: find faster way than comparing a double
				if (cpu->irq_period[i] > 0.0)
				{
					if (cpu->irq_cycle_count[i] > cpu->cycles_per_irq[i])
					{
						// don't do an NMI and an IRQ at the same time just to be safe
						if (!nmi_asserted)
						{
							cpu->irq_cycle_count[i] -= cpu->cycles_per_irq[i];
							g_game->do_irq(i);
#ifdef CPU_DIAG
							cd_irq_count[cpu->id][i]++;
#endif
							break;	// break out of for loop because we only want to assert 1 IRQ per loop
						}
					}
				} // end if
			} // end for loop
			// END CHECK FOR IRQ


			// this chunk of code tests to make sure the CPU is running
			// at the proper speed.  It should be undef'd unless we are debugging cpu stuff

#ifdef CPU_DIAG
			char s[160] = { 0 };

#define CPU_DIAG_ACCURACY 24000000
// the bigger the #, the more accurate the result

			// if it's time to print some statistics
			if (cd_cycle_count[g_active_cpu] >= CPU_DIAG_ACCURACY)
			{
				Uint32 elapsed_ms = elapsed_ms_time(cd_old_time[g_active_cpu]);
				double cur_mhz = ((double) cd_cycle_count[g_active_cpu] / (double) elapsed_ms) * 0.001;
				cd_compounded_mhz[g_active_cpu] += cur_mhz;
				cd_report_count[g_active_cpu]++;

				sprintf(s,"CPU #%d : cycles = %d, time = %d ms, MHz = %f, avg MHz = %f",
					g_active_cpu,
					cd_cycle_count[g_active_cpu], elapsed_ms,
					cur_mhz, (cd_compounded_mhz[g_active_cpu] / cd_report_count[g_active_cpu]));
				printline(s);
				sprintf(s, "         NMI's = %d ", cd_nmi_count[g_active_cpu]);
				cd_nmi_count[g_active_cpu] = 0;
				outstr(s);
				for (int irqi = 0; irqi < MAX_IRQS; irqi++)
				{
					sprintf(s, "IRQ%d's = %d ", irqi, cd_irq_count[g_active_cpu][irqi]);
					outstr(s);
					cd_irq_count[g_active_cpu][irqi] = 0;
				}
				newline();
				cd_old_time[g_active_cpu] += elapsed_ms;
				cd_cycle_count[g_active_cpu] -= CPU_DIAG_ACCURACY;
			}
#endif




			// if we are required to copy the cpu context, then preserve the context for the next time around
			if (cpu->must_copy_context)
			{
				(cpu->getcontext_callback)(cpu->context);	// preserve registers
			}

			cpu = cpu->next_cpu; // go to the next cpu

		} // end while looping through each cpu

		// BEGIN FORCING EMULATOR TO RUN AT PROPER SPEED

		// we have executed 1 ms worth of cpu cycles before this point, so slow down if 1 ms has not passed
		g_expected_elapsed_ms++;
		actual_elapsed_ms = elapsed_ms_time(g_cpu_timer);

		// if not enough time has elapsed, slow down
		while (g_expected_elapsed_ms > actual_elapsed_ms)
		{
				SDL_Delay(1);
//			SDL_Delay(0);
			actual_elapsed_ms = elapsed_ms_time(g_cpu_timer);
		}

		// END FORCING CPU TO RUN AT PROPER SPEED

		// limit checks for input events to every 16 ms
		//(this is really expensive in Windows for some reason)
		actual_elapsed_ms = elapsed_ms_time(last_inputcheck);
		if (actual_elapsed_ms > 16)
		{
			last_inputcheck = refresh_ms_time();
			SDL_check_input();	// check for input events (keyboard, joystick, etc)
		}	
	} // end while quitflag is not true
}

// sets the PC on all cpu's to their initial PC values.
// in the future this might reset the context too, but for now let's see if this is sufficient
// to reboot all our games
void cpu_reset()
{
	struct cpudef *cpu = g_head;
	
	// reset each cpu
	while (cpu)
	{
		// set the context if we need to
		if (cpu->must_copy_context)
		{
			(cpu->setcontext_callback)(cpu->context);	// restore registers
			(cpu->setmemory_callback)(cpu->mem);	// restore memory we're working with
		}
		
		(cpu->reset_callback)();
		
		// save the context if we need to
		if (cpu->must_copy_context)
		{
			(cpu->getcontext_callback)(cpu->context);	// preserve registers
		}

		cpu = cpu->next_cpu;
	}
}

// flushes all cpu timers used by emulator
// This makes it so if a laserdisc seek takes a long time, a flood of
//   interrupts aren't generated after the seek is completed.
// This should be called after any laserdisc command that might not be
//   instantaneous, or if the emulator was paused and is no longer
void flush_cpu_timers()
{
	struct cpudef *cpu = g_head;

	g_expected_elapsed_ms = 0;
	
	// clear each cpu
	while (cpu)
	{
		for (int i = 0; i < MAX_IRQS; i++)
		{
			cpu->irq_cycle_count[i] = 0;
		}
		cpu->nmi_cycle_count = 0;
		g_cpu_timer = refresh_ms_time();	// so the cpu doesn't run too quickly when we first start
		cpu->cycle_adjustment = 0;
		cpu->total_cycles_executed = 0;		
		cpu = cpu->next_cpu;
	}
}

// returns the timer used by all cpu's to run at the proper speed
// WARNING: this timer is reset by flush_cpu_timers
Uint32 get_cpu_timer()
{
	return g_cpu_timer;
}

// returns the total # of cycles that have elapsed 
// This is very useful in determining how much "time" has elapsed for time critical things like controlling the PR-8210
// laserdisc player
// WARNING : flush_cpu_timers will reset the total_cycles_executed so you must always check
// for this by making sure latest result is greater than previous result
// Failure to check for this will result in some very puzzling and frustrating bugs
Uint64 get_total_cycles_executed(Uint8 id)
{
	Uint64 result = 0;	
	struct cpudef *cpu = get_cpu_struct(id);
	
	if (cpu)
	{
		result = cpu->total_cycles_executed + (cpu->elapsedcycles_callback)();
	}
	return result;
}

// returns the pointer to the cpu structure using the cpu id as input
// returns NULL if the cpu doesn't exist
struct cpudef *get_cpu_struct(Uint8 id)
{
	struct cpudef *result = NULL;
	struct cpudef *cpu = g_head;
	
	while (cpu)
	{
		if (cpu->id == id)
		{
			result = cpu;
			break;
		}
		cpu = cpu->next_cpu;
	}
	
	return result;
}

// returns the current active cpu.  First cpu is 0
unsigned char cpu_getactivecpu()
{
	return(g_active_cpu);
}

// returns the location of the memory for the indicated cpu
// returns NULL of the indicated cpu has no memory (ie if the cpu does not exist)
Uint8 *get_cpu_mem(Uint8 id)
{
	Uint8 *result = NULL;
	struct cpudef *cpustruct = get_cpu_struct(id);

	// if the cpu exists, then we can return its memory	
	if (cpustruct)
	{
		result = cpustruct->mem;
	}
	// else the cpu does not exist in which case we return null

	return result;
}

// returns the Hz of the CPU indicated, or 0 if the cpu does not exist
Uint32 get_cpu_hz(Uint8 id)
{
	Uint32 result = 0;
	struct cpudef *cpustruct = get_cpu_struct(id);

	// if the cpu exists, then we can return its memory	
	if (cpustruct)
	{
		result = cpustruct->hz;
	}

	return result;
}

// returns cycles/ms of CPU indicated or 0.0 if cpu does not exist
double get_cpu_cycles_per_ms(Uint8 id)
{
	double result = 0.0;
	struct cpudef *cpustruct = get_cpu_struct(id);

	// if the cpu exists, then we can return its memory	
	if (cpustruct)
	{
		result = cpustruct->cycles_per_ms;
	}
	return result;
}

void cpu_change_nmi(Uint8 id, double new_period)
{
	struct cpudef *cpu = get_cpu_struct(id);
	
	if (cpu)
	{
		cpu->nmi_period = new_period;
		cpu->cycles_per_nmi = (Uint32) (cpu->cycles_per_ms * cpu->nmi_period);
	}
	else
	{
		fprintf(stderr, "ERROR : Attempted to change nmi period for cpu %d which does not exist\n", id);
	}
}

void cpu_change_irq(Uint8 id, unsigned int which_irq, double new_period)
{
	struct cpudef *cpu = get_cpu_struct(id);
	
	if (cpu)
	{
		if (which_irq < MAX_IRQS)
		{
			cpu->irq_period[which_irq] = new_period;
			cpu->cycles_per_irq[which_irq] = (Uint32) (cpu->cycles_per_ms * cpu->irq_period[which_irq]);
		}
		else
		{
			fprintf(stderr, "ERROR : Attempted to change an IRQ %d which is out of range!\n", which_irq);
		}
	}
	else
	{
		fprintf(stderr, "ERROR : Attempted to change irq period for cpu %d which does not exist\n", id);
	}
}

//////////////////////////////////////////////////////////////////////////////////

static NES_6502* g_6502 = NULL;

// the glue between daphne and the 6502 core we're using
void generic_6502_init()
{
	g_6502 = new NES_6502();

	generic_6502_setmemory(get_cpu_mem(g_active_cpu));

	NES_6502::Reset();
}

void generic_6502_shutdown()
{
	delete g_6502;
	g_6502 = NULL;
}

void generic_6502_reset()
{
	NES_6502::Reset();
}

void generic_6502_setmemory(Uint8 *buf)
{
    NES_6502::Context context;

    memset((void*)&context, 0x00, sizeof(context));
    g_6502->GetContext(&context);
			
    context.mem_page[0] = &buf[0x0000];
    context.mem_page[1] = &buf[0x2000];
    context.mem_page[2] = &buf[0x4000];
    context.mem_page[3] = &buf[0x6000];
    context.mem_page[4] = &buf[0x8000];
    context.mem_page[5] = &buf[0xa000];
    context.mem_page[6] = &buf[0xc000];
    context.mem_page[7] = &buf[0xe000];

    g_6502->SetContext(&context);

}

Uint32 generic_6502_getcontext(void *context_buf)
{
	g_6502->GetContext((NES_6502::Context *) context_buf);
	return sizeof(NES_6502::Context);
}

void generic_6502_setcontext(void *context_buf)
{
	g_6502->SetContext( (NES_6502::Context *) context_buf);
}

////////////////////////////////////////

// just a stub if the cpu core cannot return current # of elapsed cycles
// this always returns 0
Uint32 generic_cpu_elapsedcycles_stub()
{
	return 0;
}
