/*
 * cpu.h
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef CPU_H
#define CPU_H

#include <SDL.h>	// for Uint definitions

#define CPU_MEM_SIZE	0x10000	/* 64k of cpu memory */
#define MAX_CONTEXT_SIZE	100	/* max # of bytes that a cpu context can have */
#define MAX_IRQS	2	/* how many IRQs we will support per CPU */

struct cpudef;

// structure that defines parameters for each cpu daphne uses
struct cpudef
{
	// these may all be modified externally
	int type;	// which kind of cpu it is
	Uint32 hz;	// how many cycles per second the cpu is to run
	Uint16 initial_pc;	// the initial program counter for the cpu
	bool must_copy_context;	// whether THIS core will be used to emulate two or more cpu's OF THIS SAME TYPE
	double nmi_period;	// how often the NMI ticks
	double irq_period[MAX_IRQS];	// how often the IRQs tick
	Uint8 *mem;	// where the cpu's memory begins

	// these should not be modified externally
	Uint8 id;	// which we are adding
	void (*init_callback)();	// callback to initialize the cpu
	void (*shutdown_callback)();	// callback to shutdown the cpu
	void (*setmemory_callback)(Uint8 *);	// the callback to set the location of the RAM/ROM for the CPU core
	Uint32 (*execute_callback)(Uint32);	// callback to execute cycles for this particular cpu
	Uint32 (*getcontext_callback)(void *);	// callback to get a cpu's context
	void (*setcontext_callback)(void *);	// callback to set a cpu's context
	void (*setpc_callback)(Uint32);	// callback to set the program counter
	Uint32 (*elapsedcycles_callback)();	// callback to get the # of elapsed cycles
	void (*reset_callback)();	// callback to get the # of elapsed cycles
	double cycle_adjustment;	// how many extra cycles were executed last time that we don't want to execute this time
	double cycles_per_ms;	// how many cycles per ms (related to hz)
	double cycles_per_nmi;	// how many cycles per nmi (related to nmi_period)
	double cycles_per_irq[MAX_IRQS];	// how many cycles per irq (related to irq_period)
	double nmi_cycle_count;	// used to keep track of when we need to do an NMI
	double irq_cycle_count[MAX_IRQS];	// used to keep track of when we need to do an IRQ
	Uint64 total_cycles_executed;	// any cycles we've tracked so far
	Uint8 context[MAX_CONTEXT_SIZE];	// the cpu's context (in case we were forced to copy it out)
	struct cpudef *next_cpu;	// pointer to the next cpu in this linked list
};

void add_cpu(struct cpudef *);	// add a new cpu
void del_all_cpus();	// delete all cpus that have been added (for shutting down daphne)
void cpu_init();	// initialize one cpu
void cpu_shutdown();	// shutdown all cpus
void cpu_execute();
void cpu_reset();
void flush_cpu_timers();
Uint32 get_cpu_timer();
Uint64 get_total_cycles_executed(Uint8 cpu_id);
struct cpudef * get_cpu_struct(Uint8 cpu_id);
unsigned char cpu_getactivecpu();
Uint8 *get_cpu_mem(Uint8 cpu_id);
Uint32 get_cpu_hz(Uint8 cpu_id);
double get_cpu_cycles_per_ms(Uint8 id);
void cpu_change_nmi(Uint8 cpu_id, double new_period);
void cpu_change_irq(Uint8 cpu_id, unsigned int which_irq, double new_period);
void generic_6502_init();
void generic_6502_shutdown();
void generic_6502_reset();
void generic_6502_setmemory(Uint8 *buf);
Uint32 generic_6502_getcontext(void *context_buf);
void generic_6502_setcontext(void *context_buf);
Uint32 generic_cpu_elapsedcycles_stub();

#endif
