/*
 * cobraconv.cpp
 *
 * Copyright (C) 2003 Warren Ondras
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// cobraconv.cpp
// by Warren Ondras, based on bega.cpp by Mark Broadhead
//
// Cobra Command - Conversion hardware
// uses two 6502's (the second is for sound) that communicate through an am2950
// one ay-3-8910's for sound
//
// The conversion runs either the Pioneer LD-V1000 or Pioneer PR8210 (the 8210
// version has an extra pcb on top which includes another 6502 and rom). 
// It seems that the conversion board doesn't have any color ram, but this is 
// unverified

#include <string.h>
#include "cobraconv.h"
#include "../daphne.h"
#include "../cpu/cpu.h"
#include "../cpu/nes6502.h"
#include "../io/conout.h"
#include "../ldp-in/ldv1000.h"
#include "../ldp-out/ldp.h"
#include "../video/palette.h"
#include "../video/video.h"

////////////////

cobraconv::cobraconv()
{
	struct cpudef cpu;
	
	m_shortgamename = "cobraconv";
	memset(&cpu, 0, sizeof(struct cpudef));
	memset(banks, 0xFF, 4);	// fill banks with 0xFF's
	
//	m_game_type = GAME_BEGA;
	m_disc_fps = 29.97;
	
	m_video_overlay_width = COBRACONV_OVERLAY_W;
	m_video_overlay_height = COBRACONV_OVERLAY_H;
	m_palette_color_count = COBRACONV_COLOR_COUNT;
	m_video_row_offset = -16;

	cpu.type = CPU_M6502;
	cpu.hz = 2500000; // unverified	
	cpu.irq_period[0] = 16.6666; // no periodic interrupt, we are just using this for vblank (60hz)
	cpu.nmi_period = 0; // no periodic nmi
	cpu.initial_pc = 0;
	cpu.must_copy_context = false;	// set this to true when you add multiple 6502's
	cpu.mem = m_cpumem;
	add_cpu(&cpu);	// add 6502 cpu

	//ldp_status = 0x00;

	vblank = false;

	//just to test some video stuff (remove me)
	m_nvram_begin = &m_cpumem[0x0000];
	m_nvram_size = 0xFFFF;

	//m_game_issues = "This game doesn't wook at all yet.";

	// NOTE : this must be static
	const static struct rom_def roms[] =
	{
		// main cpu roms
		{ "bd00", NULL, &m_cpumem[0xe000], 0x2000, 0x8d9ad777 },
		{ "bd01", NULL, &m_cpumem[0xc000], 0x2000, 0x1b4db507 },
		{ "bd02", NULL, &m_cpumem[0xa000], 0x2000, 0x3d802707 },
		{ "bd03", NULL, &m_cpumem[0x8000], 0x2000, 0xf1b9df77 },
		
		// roms for the character/sprite generator (conv board only has one)
		{ "bd06", NULL, &character2[0x0000], 0x2000, 0xb1340125 },
		{ "bd05", NULL, &character2[0x2000], 0x2000, 0x98412178 },
		{ "bd04", NULL, &character2[0x4000], 0x2000, 0x33013cc2 },

		// color lookup prom
		{ "vd0-c.bpr", NULL, &color_prom[0x0000], 0x0020, 0x2c27aa0 },

		// unused video timing prom(?), misc. PLD
		{ "vd0-t.bpr", NULL, &miscprom[0x0000], 0x0020, 0x78449942 },
		{ "lp4-2.pld", NULL, &miscprom[0x0100], 0x022A, 0x4bc65ab0 },

		{ NULL }
	};

	m_rom_list = roms;
	
}


// clocks screen updates and vblank timing
void cobraconv::do_irq(unsigned int which_irq)
{
	video_blit();
	vblank = true;
	nes6502_irq();
}

void cobraconv::do_nmi()
{
	// not timed; used for coin input detection
	// nes6502_nmi();
}

Uint8 cobraconv::cpu_mem_read(Uint16 addr)
{
	static unsigned int loopcount = 0;
	char s[81] = {0};

	Uint8 result = m_cpumem[addr];

	// main ram
	if (addr >= 0x0000 && addr <= 0x0fff)
	{
	}
	
	// main rom
	else if (addr >= 0x4000 && addr <= 0xffff)
	{
	}
	
	// video ram
	else if (addr >= 0x2000 && addr <= 0x3fff)
	{
	}

	//bit 7 - vblank
	//bit 0 - tilt input
	//bits 5 and 6  - ld strobes
	//bits 2 and 3 - coin inputs
	//
	// bits 5, 6 and 7 need some kind of bit sequence to keep the game running
	// (test mode wooks without it)
	else if (addr == 0x1001) 
	{
		if (vblank)
		{
			loopcount = 0;
			vblank = false;
			banks[3] &= 0x1F;
			banks[3] |= 0x80;  //vblank on, ld off
		}
		
		// just toggle the vblank and LD bits for now
		// FIXME:  need to figure out how these bits really work
		if (loopcount < 4) banks[3] ^= 0xE0; 
		
		loopcount++;
 
		result = banks[3];
	}  

	// control panel
	else if (addr == 0x1000)
	{
		result = banks[0];
	}

	// dipswitch 1
	else if (addr == 0x1002)
	{
		result = banks[1];
	}

	// dipswitch 2
	else if (addr == 0x1003)
	{
		result = banks[2];
	}

	// laserdisc player interface
	else if (addr == 0x1004)
	{
		result = read_ldv1000();
	}

	else
	{
		sprintf(s, "Unmapped read from %x", addr);
		printline(s);
	}

	return result;
}

void cobraconv::cpu_mem_write(Uint16 addr, Uint8 value)
{
	char s[81] = {0};

	// main ram
	if (addr >= 0x0000 && addr <= 0x0fff)
	{
	}
	
	// video ram
	else if (addr >= 0x2000 && addr <= 0x3fff)
	//  how much video ram does cobraconv have?
	//  and does it have palette ram?  from 0x3000-3fff?
	//	else if (addr >= 0x2000 && addr <= 0x2fff)
	{
		if (value != m_cpumem[addr]) 
		{
//			sprintf(s, "Video write to %x with value %x", addr, value);
//			printline(s);
			m_video_overlay_needs_update = true;
		}
	}

	// 0x1001 bit 1 is LD command strobe? - it occurs before ld commands are sent
	//        other bits for int/ext sync? other?
	else if (addr == 0x1001)  
	{
	}

	// 0x1002 coin counters and blockers?
	// bit 6 toggles on NMI
	else if (addr == 0x1002)  
	{
	}
	
	// 0x1005 is sound -- sends a sound number to second CPU, sends 0 to mute
	// (should be easy to hack with samples for now, until we have a real sound core/mixer)
	else if (addr == 0x1005)  
	{
	}

	else if (addr == 0x1003)  
	{
		// ?
	}

	//LD player data bus
	else if (addr == 0x1004)  
	{
		write_ldv1000(value);
	}
	

	// main rom
	else if (addr >= 0x4000 && addr <= 0xffff)
	{
		sprintf(s, "Error! write to main rom at %x", addr);
		printline(s);
	}
	
	else
	{
		sprintf(s, "Unmapped write to %x with value %x", addr, value);
		printline(s);
	}

	m_cpumem[addr] = value;
}

void cobraconv::palette_calculate()
{
	SDL_Color temp_color;
		
	//Convert palette rom into a useable palette
	for (int i = 0; i < COBRACONV_COLOR_COUNT; i++)
	{
		Uint8 bit0,bit1,bit2;	

		/* red component */
		bit0 = static_cast<Uint8>((color_prom[i] >> 0) & 0x01);
		bit1 = static_cast<Uint8>((color_prom[i] >> 1) & 0x01);
		bit2 = static_cast<Uint8>((color_prom[i] >> 2) & 0x01); 
		temp_color.r = static_cast<Uint8>((0x21 * bit0) + (0x47 * bit1) + (0x97 * bit2));
		
		/* green component */
		bit0 = static_cast<Uint8>((color_prom[i] >> 3) & 0x01);
		bit1 = static_cast<Uint8>((color_prom[i] >> 4) & 0x01);
		bit2 = static_cast<Uint8>((color_prom[i] >> 5) & 0x01);
		temp_color.g = static_cast<Uint8>((0x21 * bit0) + (0x47 * bit1) + (0x97 * bit2));
			
		/* blue component */
		bit0 = static_cast<Uint8>(0);
		bit1 = static_cast<Uint8>((color_prom[i] >> 6) & 0x01);
		bit2 = static_cast<Uint8>((color_prom[i] >> 7) & 0x01);
		temp_color.b = static_cast<Uint8>((0x21 * bit0) + (0x47 * bit1) + (0x97 * bit2));
		
		////apply gamma correction to make colors brighter overall
		//Corrected value = 255 * (uncorrected value / 255) ^ (1.0 / gamma)
		//temp_color.r = (Uint8) (255 * pow((static_cast<double>(temp_color.r)) / 255, 1/COBRACONV_GAMMA));
		//temp_color.g = (Uint8) (255 * pow((static_cast<double>(temp_color.g)) / 255, 1/COBRACONV_GAMMA));
		//temp_color.b = (Uint8) (255 * pow((static_cast<double>(temp_color.b)) / 255, 1/COBRACONV_GAMMA));

		palette_set_color(i, temp_color);
	}

	// set the transparent color number
	palette_set_transparent_number(0);

}

void cobraconv::video_repaint()
{
/*	if (palette_updated)
	{
		recalc_palette();
		m_video_overlay_needs_update = true;
		palette_updated = false;
	}  */

	//fast screen clear
	SDL_FillRect(m_video_overlay[m_active_video_overlay], NULL, 0);

	// draw sprites first(?)
	draw_sprites(0x2800, character2);
	//draw_sprites(0x2be0, character2);

	// draw tiles first
	for (int charx = 0; charx < 32; charx++)
	{
		// don't draw the first (or last?) lines of tiles (this is where the sprite data is)
		for (int chary = 1; chary < 31; chary++)
		//for (int chary = 1; chary < 32; chary++)
		{
			// draw 8x8 tiles from tile/sprite generator 2
			int current_character = m_cpumem[chary * 32 + charx + 0x2800] + 256 * (m_cpumem[chary * 32 + charx + 0x2c00] & 0x03);
			draw_8x8(current_character, 
				character2, 
				charx*8, chary*8, 
				0, 0, 
				0); // this isn't the correct color... i'm not sure where color comes from right now
					
			// draw 8x8 tiles from tile/sprite generator 1
			current_character = m_cpumem[chary * 32 + charx + 0x2000] + 256 * (m_cpumem[chary * 32 + charx + 0x2400] & 0x03);
			draw_8x8(current_character, 
				character2, 
				//charx*8, chary*8,  // x/y swapped vs Bega's Battle hardware
				chary*8, charx*8, 
				0, 0, 
				0); // this isn't the correct color... i'm not sure where color comes from right now
		}
	}
}

// used to set dip switch values
bool cobraconv::set_bank(unsigned char which_bank, unsigned char value)
{
	bool result = true;
	
	switch (which_bank)
	{
	case 0:	// bank A
		banks[1] = (unsigned char) (value ^ 0xFF);	// dip switches are active low
		break;
	case 1:	// bank B
		banks[2] = (unsigned char) (value ^ 0xFF);	// switches are active low
		break;
	default:
		printline("ERROR: Bank specified is out of range!");
		result = false;
		break;
	}
	
	return result;
}
// this gets called when the user presses a key or moves the joystick
void cobraconv::input_enable(Uint8 move)
{
	switch (move)
	{
	case SWITCH_UP:
		banks[0] &= ~0x02; 
		break;
	case SWITCH_LEFT:
		banks[0] &= ~0x08; 
		break;
	case SWITCH_RIGHT:
		banks[0] &= ~0x04; 
		break;
	case SWITCH_DOWN:
		banks[0] &= ~0x01; 
		break;
	case SWITCH_START1:
		banks[0] &= ~0x40; 
		break;
	case SWITCH_START2:
		banks[0] &= ~0x80; 
		break;
	case SWITCH_BUTTON1: 
		banks[0] &= ~0x10; 
		break;
	case SWITCH_BUTTON2: 
		banks[0] &= ~0x20; 
		break;
	case SWITCH_BUTTON3: 
//		banks[3] &= ~0x10; 
		break;
	case SWITCH_COIN1: 
		banks[3] &= ~0x04; 
		nes6502_nmi();
		break;
	case SWITCH_COIN2: 
		banks[3] &= ~0x08; 
		nes6502_nmi();
		break;
	case SWITCH_SERVICE: 
//		banks[3] &= ~0x08; 
		break;
	case SWITCH_TEST: 
		banks[3] &= ~0x01;  //TILT
		break;
	default:
//		printline("Error, bug in move enable");
		break;
	}
}  

// this gets called when the user releases a key or moves the joystick back to center position
void cobraconv::input_disable(Uint8 move)
{
	switch (move)
	{
	case SWITCH_UP:
		banks[0] |= 0x02; 
		break;
	case SWITCH_LEFT:
		banks[0] |= 0x08; 
		break;
	case SWITCH_RIGHT:
		banks[0] |= 0x04; 
		break;
	case SWITCH_DOWN:
		banks[0] |= 0x01; 
		break;
	case SWITCH_START1: 
		banks[0] |= 0x40; 
		break;
	case SWITCH_START2: 
		banks[0] |= 0x80; 
		break;
	case SWITCH_BUTTON1: 
		banks[0] |= 0x10; 
		break;
	case SWITCH_BUTTON2: 
		banks[0] |= 0x20; 
		break;
	case SWITCH_BUTTON3: 
//		banks[3] |= 0x10; 
		break;
	case SWITCH_COIN1: 
		banks[3] |= 0x04; 
		break;
	case SWITCH_COIN2: 
		banks[3] |= 0x08; 
		break;
	case SWITCH_SERVICE: 
//		banks[3] |= 0x08; 
		break;
	case SWITCH_TEST: 
		banks[3] |= 0x01;
		break;
	default:
//		printline("Error, bug in move enable");
		break;
	}
}

void cobraconv::draw_8x8(int character_number, Uint8 *character_set, int xcoord, int ycoord,
					int xflip, int yflip, int color)
{
	Uint8 pixel[8] = {0};
	
	for (int y = 0; y < 8; y++)
	{
		Uint8 byte1 = character_set[character_number*8+y];
		Uint8 byte2 = character_set[character_number*8+y+0x2000];
		Uint8 byte3 = character_set[character_number*8+y+0x4000];

		pixel[0] = static_cast<Uint8>(((byte1 & 0x01) << 2) | ((byte2 & 0x01) << 1) | ((byte3 & 0x01) << 0));
		pixel[1] = static_cast<Uint8>(((byte1 & 0x02) << 1) | ((byte2 & 0x02) << 0) | ((byte3 & 0x02) >> 1));
		pixel[2] = static_cast<Uint8>(((byte1 & 0x04) << 0) | ((byte2 & 0x04) >> 1) | ((byte3 & 0x04) >> 2));
		pixel[3] = static_cast<Uint8>(((byte1 & 0x08) >> 1) | ((byte2 & 0x08) >> 2) | ((byte3 & 0x08) >> 3));
		pixel[4] = static_cast<Uint8>(((byte1 & 0x10) >> 2) | ((byte2 & 0x10) >> 3) | ((byte3 & 0x10) >> 4));
		pixel[5] = static_cast<Uint8>(((byte1 & 0x20) >> 3) | ((byte2 & 0x20) >> 4) | ((byte3 & 0x20) >> 5));
		pixel[6] = static_cast<Uint8>(((byte1 & 0x40) >> 4) | ((byte2 & 0x40) >> 5) | ((byte3 & 0x40) >> 6));
		pixel[7] = static_cast<Uint8>(((byte1 & 0x80) >> 5) | ((byte2 & 0x80) >> 6) | ((byte3 & 0x80) >> 7));

		for (int x = 0; x < 8; x++)
		{
			if (pixel[x])
			{
				*((Uint8 *) m_video_overlay[m_active_video_overlay]->pixels + ((ycoord + (yflip ? y : (7-y))) * COBRACONV_OVERLAY_W) + (xcoord + (xflip ? (7-x) : x))) = pixel[x] + (8*color);
			}
		}
	}
}

void cobraconv::draw_16x16(int character_number, Uint8 *character_set, int xcoord, int ycoord,
					int xflip, int yflip, int color)
{
	Uint8 pixel[16] = {0};
	
	for (int y = 0; y < 32; y++)
	{
		Uint8 byte1 = character_set[character_number*32+y];
		Uint8 byte2 = character_set[character_number*32+y+0x2000];
		Uint8 byte3 = character_set[character_number*32+y+0x4000];
		Uint8 byte4 = character_set[character_number*32+y+8];
		Uint8 byte5 = character_set[character_number*32+y+0x2000+8];
		Uint8 byte6 = character_set[character_number*32+y+0x4000+8];

		pixel[0]  = static_cast<Uint8>((((byte1 & 0x01) << 2) | ((byte2 & 0x01) << 1) | ((byte3 & 0x01) << 0)));
		pixel[1]  = static_cast<Uint8>((((byte1 & 0x02) << 1) | ((byte2 & 0x02) << 0) | ((byte3 & 0x02) >> 1)));
		pixel[2]  = static_cast<Uint8>((((byte1 & 0x04) << 0) | ((byte2 & 0x04) >> 1) | ((byte3 & 0x04) >> 2)));
		pixel[3]  = static_cast<Uint8>((((byte1 & 0x08) >> 1) | ((byte2 & 0x08) >> 2) | ((byte3 & 0x08) >> 3)));
		pixel[4]  = static_cast<Uint8>((((byte1 & 0x10) >> 2) | ((byte2 & 0x10) >> 3) | ((byte3 & 0x10) >> 4)));
		pixel[5]  = static_cast<Uint8>((((byte1 & 0x20) >> 3) | ((byte2 & 0x20) >> 4) | ((byte3 & 0x20) >> 5)));
		pixel[6]  = static_cast<Uint8>((((byte1 & 0x40) >> 4) | ((byte2 & 0x40) >> 5) | ((byte3 & 0x40) >> 6)));
		pixel[7]  = static_cast<Uint8>((((byte1 & 0x80) >> 5) | ((byte2 & 0x80) >> 6) | ((byte3 & 0x80) >> 7)));
		pixel[8]  = static_cast<Uint8>((((byte4 & 0x01) << 2) | ((byte5 & 0x01) << 1) | ((byte6 & 0x01) << 0)));
		pixel[9]  = static_cast<Uint8>((((byte4 & 0x02) << 1) | ((byte5 & 0x02) << 0) | ((byte6 & 0x02) >> 1)));
		pixel[10] = static_cast<Uint8>((((byte4 & 0x04) << 0) | ((byte5 & 0x04) >> 1) | ((byte6 & 0x04) >> 2)));
		pixel[11] = static_cast<Uint8>((((byte4 & 0x08) >> 1) | ((byte5 & 0x08) >> 2) | ((byte6 & 0x08) >> 3)));
		pixel[12] = static_cast<Uint8>((((byte4 & 0x10) >> 2) | ((byte5 & 0x10) >> 3) | ((byte6 & 0x10) >> 4)));
		pixel[13] = static_cast<Uint8>((((byte4 & 0x20) >> 3) | ((byte5 & 0x20) >> 4) | ((byte6 & 0x20) >> 5)));
		pixel[14] = static_cast<Uint8>((((byte4 & 0x40) >> 4) | ((byte5 & 0x40) >> 5) | ((byte6 & 0x40) >> 6)));
		pixel[15] = static_cast<Uint8>((((byte4 & 0x80) >> 5) | ((byte5 & 0x80) >> 6) | ((byte6 & 0x80) >> 7)));

		for (int x = 0; x < 16; x++)
		{
			if (pixel[x])
			{
				*((Uint8 *) m_video_overlay[m_active_video_overlay]->pixels + ((ycoord + y) * COBRACONV_OVERLAY_W) + (xcoord + (xflip ? (15-x) : x))) = pixel[x] + (8*color);
			}			
		}
	}
}
void cobraconv::draw_sprites(int offset, Uint8 *character_set)
{
	for (int sprites = 0; sprites < 0x32; sprites += 4)
	{
		if ((m_cpumem[offset + sprites] & 0x01) && (m_cpumem[offset + sprites + 3] < 240))
		{		
//			char s[80];
//			sprintf(s, "sprite %x, char=%x, x=%x, y=%x", sprites, m_cpumem[offset + sprites + 1],
//				m_cpumem[offset + sprites + 3], m_cpumem[offset + sprites + 2]);
//			printline(s);
			draw_16x16(m_cpumem[offset + sprites + 1],
				character_set,
				m_cpumem[offset + sprites + 3], 
				m_cpumem[offset + sprites + 2], 
				m_cpumem[offset + sprites] & 0x04,
				m_cpumem[offset + sprites] & 0x02,
				0); // this isn't the correct color... i'm not sure where color comes from right now
		}
	}
}
