/*
 * dl2.cpp
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// Dragon's Lair 2 shared library wrapper
// by Matt Ownby

#include "../io/serial.h"
#include "../io/input.h"
#include "../io/conout.h"
#include "../daphne.h"
#include "dl2.h"

typedef void (*startproc)(struct daphne_exported_api *);
startproc g_dl2_start;

struct daphne_exported_api g_api;	// fill this in later

//////////////////////////////////////////////////////////////////

unsigned char g_dl2_joystick = 0;	// byte compatible with DL2's joystick polling system
unsigned char g_dl2_coin = 0;	// byte compatible with DL2's coin system

// function that the DL2 code can call to query daphne input state
unsigned char dl2_get_joystick()
{
	return g_dl2_joystick;
}

unsigned char dl2_get_coin()
{
	return g_dl2_coin;
}

int dl2_serial_byte_available()
{
	int result = 0;
	if (serial_rx_char_waiting())
	{
		result = 1;
	}
	return result;
}

void dl2_play_sound(int which_sound)
{
	switch (which_sound)
	{
	case 1:	// warble
		sound_play(S_DL2_WARBLE);
		break;
	case 2:	// error
		sound_play(S_DL2_ERROR);
		break;
	case 3:	// good
		sound_play(S_DL2_GOOD);
		break;
	case 4:	// bad
		sound_play(S_DL2_BAD);
		break;
	case 5:	// TIC
		sound_play(S_DL2_TIC);
		break;
	case 6:	// TOC
		sound_play(S_DL2_TOC);
		break;
	case 9:	// WARN
		sound_play(S_DL2_WARN);
		break;
	default:
		printline("ERROR : DL2 sound unknown");
		break;
	}
}

//////////////////////////////////////////////////////////////////

// constructor
dl2::dl2()
{
	m_shortgamename = "dl2";
	m_disc_fps = 29.97;

	m_num_sounds = 7;
	m_sound_name[S_DL2_WARBLE] = "esh_beep.wav";
	m_sound_name[S_DL2_GOOD] = "dl_accept.wav";
	m_sound_name[S_DL2_BAD] = "dl_buzz.wav";
	m_sound_name[S_DL2_ERROR] = "sd_fail.wav";
	m_sound_name[S_DL2_TIC] = "sda_success_hi.wav";
	m_sound_name[S_DL2_TOC] = "sda_success_lo.wav";
	m_sound_name[S_DL2_WARN] = "gr_mineon.wav";
}

// load dl2 .dll
bool dl2::init()
{
	bool result = false;
	
    m_dll_instance = M_LOAD_LIB(dl2);	// load DL2 dll

    // If the handle is valid, try to get the function address. 
    if (m_dll_instance) 
    {
    	g_dl2_start = (startproc) M_GET_SYM(m_dll_instance, "dl2_start");

		if (g_dl2_start)
		{
			result = true;
		}
		else
		{
			printline("ERROR : some DLL functions could not be loaded");
		}
	}
	else
	{
		printline("ERROR: could not open the DL2 dynamic library");
	}
	
	return result;
}

// free dl2 .dll
void dl2::shutdown()
{
	if (m_dll_instance)
	{
		M_FREE_LIB(m_dll_instance);
		m_dll_instance = NULL;
	}
}

void dl2::start()
{
	g_api.get_switch_input = dl2_get_joystick;
	g_api.get_coin_input = dl2_get_coin;
	g_api.play_sound = dl2_play_sound;
	g_api.serial_send_byte = serial_tx;
	g_api.serial_byte_available = dl2_serial_byte_available;
	g_api.serial_read_byte = serial_rx;
	g_api.should_we_quit = get_quitflag;
	g_api.poll_input = SDL_check_input;
	g_dl2_start(&g_api);	// launch the game
}

void dl2::input_enable(Uint8 move)
{
	switch(move)
	{
	case SWITCH_UP:
		g_dl2_joystick |= 1;
		break;
	case SWITCH_DOWN:
		g_dl2_joystick |= 2;
		break;
	case SWITCH_LEFT:
		g_dl2_joystick |= 4;
		break;
	case SWITCH_RIGHT:
		g_dl2_joystick |= 8;
		break;
	case SWITCH_START1: // PLAYER 1
		g_dl2_joystick |= 32;
		break;
	case SWITCH_START2:	// player 2
		g_dl2_joystick |= 64;
		break;
	case SWITCH_BUTTON1: // SWORD
		g_dl2_joystick |= 16;
		break;
	case SWITCH_COIN1: 
		g_dl2_coin |= 0x30;
		break;
	case SWITCH_COIN2: 
		g_dl2_coin |= 0x50;
		break;
	default:
		break;
	}
}

void dl2::input_disable(Uint8 move)
{

	switch(move)
	{
	case SWITCH_UP:
		g_dl2_joystick &= ~1;
		break;
	case SWITCH_DOWN:
		g_dl2_joystick &= ~2;
		break;
	case SWITCH_LEFT:
		g_dl2_joystick &= ~4;
		break;
	case SWITCH_RIGHT:
		g_dl2_joystick &= ~8;
		break;
	case SWITCH_START1: // PLAYER 1
		g_dl2_joystick &= ~32;
		break;
	case SWITCH_START2:	// player 2
		g_dl2_joystick &= ~64;
		break;
	case SWITCH_BUTTON1: // SWORD
		g_dl2_joystick &= ~16;
		break;
	case SWITCH_COIN1: 
		g_dl2_coin &= ~0x30;
		break;
	case SWITCH_COIN2: 
		g_dl2_coin &= ~0x50;
		break;
	default:
		break;
	}
}
