/*
 * firefox.cpp
 *
 * Copyright (C) 2001 Mark Broadhead
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// firefox.cpp
// by Mark Broadhead
//

#include <string.h>
#include "firefox.h"
#include "../daphne.h"
#include "../cpu/cpu.h"
#include "../cpu/mc6809.h"
#include "../io/conout.h"
#include "../ldp-in/vp931.h"
#include "../ldp-out/ldp.h"
#include "../video/video.h"
#include "../video/palette.h"

////////////////

firefox::firefox()
{
	struct cpudef cpu;
	
	m_shortgamename = "firefox";
	memset(&cpu, 0, sizeof(struct cpudef));
	memset(banks, 0xFF, 4);	// fill banks with 0xFF's
//	banks[0] = 0xfb;
	banks[1] = 0x1f;
	banks[4] = 0x00;
	banks[5] = 0x00;
	m_disc_fps = 29.97;

	m_video_overlay_width = FIREFOX_OVERLAY_W;
	m_video_overlay_height = FIREFOX_OVERLAY_H;
	m_palette_color_count = FIREFOX_COLORS;

	cpu.type = CPU_M6809;
	cpu.hz = 4000000;	
	cpu.nmi_period = 1000.0 / (60.0);
	cpu.irq_period[0] = 1000.0 / (60.0 * 6.0);
	cpu.initial_pc = 0;
	cpu.must_copy_context = false;	// set to true for multiple 6809's
	cpu.mem = m_cpumem;
	add_cpu(&cpu);	// add a 6809 cpu
	
	ad_converter_channel = 0;
	current_bank = 0x0000;
	m_game_issues = "Inputs aren't hooked up and the LDP isn't implemented";

	const static struct rom_def firefox_roms[] =
	{
		// main program
		{ "136026.109", NULL, &m_cpumem[0x4000], 0x4000, 0x7639270c },
		{ "136026.110", NULL, &m_cpumem[0x8000], 0x4000, 0xf3102944 },
		{ "136026.111", NULL, &m_cpumem[0xc000], 0x4000, 0x8a230bb5 },

		// rom banks
		{ "136026.101", NULL, &rombank[0x0000], 0x4000, 0x91bba45a },
		{ "136026.102", NULL, &rombank[0x4000], 0x4000, 0x5f1e423d },
		{ "136026.105", NULL, &rombank[0x8000], 0x4000, 0x83f1d4ed },
		{ "136026.106", NULL, &rombank[0xc000], 0x4000, 0xc5d8d417 },

		// characters
		{ "136026.125", NULL, &character[0x0000], 0x2000, 0x8a32f9f1 },
		{ NULL }
	};

	m_rom_list = firefox_roms;

}

firefoxa::firefoxa()
{
	m_shortgamename = "firefoxa";

	const static struct rom_def firefoxa_roms[] =
	{
		// main program
		{ "136026.209", NULL, &m_cpumem[0x4000], 0x4000, 0 },
		{ "136026.210", NULL, &m_cpumem[0x8000], 0x4000, 0 },
		{ "136026.211", NULL, &m_cpumem[0xc000], 0x4000, 0 },

		// rom banks
		{ "136026.201", NULL, &rombank[0x0000], 0x4000, 0 },
		{ "136026.205", NULL, &rombank[0x8000], 0x4000, 0 },
		{ "136026.127", NULL, &rombank[0xc000], 0x2000, 0 },

		// characters
		{ "136026.125", "firefox", &character[0x0000], 0x2000, 0 },
		{ NULL }
	};

	m_rom_list = firefoxa_roms;

}

void firefox::do_irq(unsigned int which_irq)
{
	mc6809_irq = 1;
}

// we cheat here... we call this once per hblank to update the display
void firefox::do_nmi()
{
	video_blit();
}

// does anything special needed to send an FIRQ
void firefox::do_firq()
{
}

Uint8 firefox::cpu_mem_read(Uint16 addr)
{
	char s[81] = {0};

	Uint8 result = m_cpumem[addr];

	// Program RAM
	if (addr <= 0x0fff)
	{
	}

	// Graphics/Alphanumerics RAM
	else if (addr >= 0x1000 && addr <= 0x1fff)
	{
	}

	// Motion Object RAM
	else if (addr >= 0x2000 && addr <= 0x27ff)
	{
	}

	// Inputs 1
	else if (addr == 0x4100)
	{
		result = banks[0];
	}

	// Inputs 2
	else if (addr == 0x4101)
	{
		result = banks[1];
	}

	// Option Sw 1 
	else if (addr == 0x4103)
	{
//		result = banks[2];
	}

	// Option Sw 2 
	else if (addr == 0x4104)
	{
//		result = banks[3];
	}

	// Read A/D Converter
	else if (addr == 0x4107)
	{
		switch (ad_converter_channel)
		{
		case 0:
			result = banks[4];
			break;
		case 1:
			result = banks[5];
			break;
		default:
			printline("Invalid A/D Converter channel");
			break;
		}
	}

	// Table Rom
	else if (addr >= 0x3000 && addr <= 0x3fff)
	{
		result = rombank[current_bank | (addr & 0xfff)];
//		sprintf(s, "current bank %x reading m_cpumem[%x], returning rombank[%x]", current_bank, addr, current_bank + (addr - 0x3000));
//		printline(s);
	}

	// Disc status, bit 7 - Diskdav, 6 - Disk Full, 5 - Disk Opr
	else if (addr == 0x4102)
	{
		result = 0x00;
		result |= ((vp931_read_dav()?1:0) << 7);
		result |= ((vp931_read_dak()?1:0) << 6);
		result |= ((vp931_read_oprt()?1:0) << 5);
	}

	// Read disc interface data
	else if (addr == 0x4105)
	{
		result = read_vp931();
	}

	// Program ROM
	else if (addr >= 0x4400 && addr <= 0xffff)
	{
	}

	else
	{
//		sprintf(s, "Unmapped read from %x", addr);
//		printline(s);
	}

	return result;
}

void firefox::cpu_mem_write(Uint16 addr, Uint8 value)
{
	char s[81] = {0};

	// Program RAM
	if (addr <= 0x0fff)
	{
	}

	// Graphics/Alphanumerics RAM
	else if (addr >= 0x1000 && addr <= 0x1fff)
	{
		m_video_overlay_needs_update = true;
	}

	// Motion Object RAM
	else if (addr >= 0x2000 && addr <= 0x27ff)
	{
	}

	// Color Ram (sprites)
	else if (addr >= 0x2800 && addr <= 0x2aff)
	{
	}
	
	// Color Ram (alphanumerics)
	else if (addr >= 0x2c00 && addr <= 0x2fff)
	{
		palette_modified = true;
	}
	
	// Reset Watchdog
	else if (addr == 0x4210)
	{
	}
	
	// DSKREAD
	else if (addr == 0x4210)
	{
		vp931_set_read();
	}
	
	// Start A/D Converter Channel 0/1
	else if (addr >= 0x4220 && addr <= 0x4221)
	{
		ad_converter_channel = addr & 0x01;
	}

	// Reset Disc
	else if (addr == 0x4286)
	{
		if (value)
		{
			vp931_set_reset();
		}
	}
	
	// Write Disc (active low)
	else if (addr == 0x4287)
	{
		if (!value)
		{
			vp931_set_write();
		}
	}
	
	// Led 1-4
	else if (addr >= 0x428c && addr <= 0x428f)
	{
		if (value & 0x80)
		{
			sprintf(s,"Led %x off", (addr & 0x03) + 1);
		}
		else 
		{
			sprintf(s,"Led %x on", (addr & 0x03) + 1);
		}
		printline(s);
	}

	// Rom paging @ 3000
	else if (addr == 0x4290)
	{
		switch(value & 0x1c)
		{
			case 0x00: /* T0 */
				current_bank = (0x0000 + (0x1000 * (value & 0x3)));
	//			sprintf(s, "bank switch, T0: %x\n", value & 0x3);
	//			printline(s);
				break;
			case 0x04: /* T1 */
				current_bank = (0x4000 + (0x1000 * (value & 0x3)));
	//			sprintf(s, "bank switch, T1: %x\n", value & 0x3);
	//			printline(s);
				break;
			case 0x10: /* T4 */
				current_bank = (0x8000 + (0x1000 * (value & 0x3)));
	//			sprintf(s, "bank switch, T4: %x\n", value & 0x3);
	//			printline(s);
				break;
			case 0x14: /* T5 */
				current_bank = (0xc000 + (0x1000 * (value & 0x3)));
	//			sprintf(s, "bank switch, T5: %x\n", value & 0x3);
	//			printline(s);
				break;
			default:
				sprintf(s,"Invalid bank switch, %x", value);
				printline(s);
				break;
		}
	}

	// Write Data to Disc Interface
	else if (addr == 0x42a0)
	{
		write_vp931(value);
	}
	
	// Program ROM
	else if (addr >= 0x4400 && addr <= 0xffff)
	{
		printline("ERROR: Write to program rom!");
	}

	else
	{
//		sprintf(s, "Unmapped write to %x with %x", addr, value);
//		printline(s);
	}


	m_cpumem[addr] = value;
}

void firefox::palette_calculate()
{
	SDL_Color color;
	for (int x = 0; x < FIREFOX_COLORS; x++)
	{
		color.r = m_cpumem[0x2c00 + x];
		color.g = m_cpumem[0x2d00 + x];
		color.b = m_cpumem[0x2e00 + x] & 0xfd;
		palette_set_color(x, color);
	}
}

// updates firefox's video
void firefox::video_repaint()
{
	if (palette_modified)
	{
		palette_calculate();
		palette_finalize();
	}

	for (int charx = 0; charx < 64; charx++)
	{
		for (int chary = 0; chary < 64; chary++)
		{
			for (int x=0; x < 4; x++)
			{
				for (int y = 0; y < 8; y++)
				{
					Uint8 left_pixel = static_cast<Uint8>((character[m_cpumem[chary * 64 + charx + 0x1000]*32+x+4*y] & 0xf0) >> 4);
					Uint8 right_pixel = static_cast<Uint8>((character[m_cpumem[chary * 64 + charx + 0x1000]*32+x+4*y] & 0x0f));

					*((Uint8 *) m_video_overlay[m_active_video_overlay]->pixels + (((chary) * 8 + y) * FIREFOX_OVERLAY_W) + ((charx) * 8 + x * 2)) = left_pixel;
					*((Uint8 *) m_video_overlay[m_active_video_overlay]->pixels + (((chary) * 8 + y) * FIREFOX_OVERLAY_W) + ((charx) * 8 + x * 2 + 1)) = right_pixel;

//						*((Uint8 *) m_video_overlay[m_active_video_overlay]->pixels + (((chary) * 8 + y) * FIREFOX_OVERLAY_W) + ((charx - 10) * 16 + x * 4 + 0)) = left_pixel;
//						*((Uint8 *) m_video_overlay[m_active_video_overlay]->pixels + (((chary) * 8 + y) * FIREFOX_OVERLAY_W) + ((charx - 10) * 16 + x * 4 + 1)) = left_pixel;
//						*((Uint8 *) m_video_overlay[m_active_video_overlay]->pixels + (((chary) * 8 + y) * FIREFOX_OVERLAY_W) + ((charx - 10) * 16 + x * 4 + 2)) = right_pixel;
//						*((Uint8 *) m_video_overlay[m_active_video_overlay]->pixels + (((chary) * 8 + y) * FIREFOX_OVERLAY_W) + ((charx - 10) * 16 + x * 4 + 3)) = right_pixel;
				}
			}
		}
	}
}

// this gets called when the user presses a key or moves the joystick
void firefox::input_enable(Uint8 move)
{
	switch (move)
	{
	case SWITCH_UP:
		banks[4] = 0x00;
		break;
	case SWITCH_LEFT:
		banks[5] = 0x00;
		break;
	case SWITCH_RIGHT:
		banks[5] = 0xff;
		break;
	case SWITCH_DOWN:
		banks[4] = 0xff;
		break;
	case SWITCH_BUTTON1: // '1' on keyboard
		banks[0] &= ~0x08; 
		break;
	case SWITCH_BUTTON2: // '2' on keyboard
		banks[0] &= ~0x10; 
		break;
	case SWITCH_BUTTON3: // space on keyboard
		banks[0] &= ~0x20; 
		break;
	case SWITCH_COIN1: 
		banks[1] &= ~0x02; 
		break;
	case SWITCH_COIN2: 
		banks[1] &= ~0x01; 
		break;
	case SWITCH_SERVICE: 
		banks[1] &= ~0x04; 
		break;
	case SWITCH_TEST: 
		banks[1] &= 0x10; 
		break;
	default:
		printline("Error, bug in move enable");
		break;
	}
}  

// this gets called when the user releases a key or moves the joystick back to center position
void firefox::input_disable(Uint8 move)
{
	switch (move)
	{
	case SWITCH_UP:
		banks[4] = 0x7f;	// centered
		break;
	case SWITCH_LEFT:
		banks[5] = 0x7f;
		break;
	case SWITCH_RIGHT:
		banks[5] = 0x7f;
		break;
	case SWITCH_DOWN:
		banks[4] = 0x7f;
		break;
	case SWITCH_BUTTON1: // '1' on keyboard
		banks[0] |= 0x08; 
		break;
	case SWITCH_BUTTON2: // '2' on keyboard
		banks[0] |= 0x10; 
		break;
	case SWITCH_BUTTON3: // space on keyboard
		banks[0] |= 0x20; 
		break;
	case SWITCH_COIN1:
		banks[1] |= 0x02; 
		break;
	case SWITCH_COIN2:
		banks[1] |= 0x01; 
		break;
	case SWITCH_SERVICE:
		banks[1] |= 0x04; 
		break;
	case SWITCH_TEST:
		banks[1] |= 0x10; 
		break;
	default:
		printline("Error, bug in move enable");
		break;
	}
}

// used to set dip switch values
bool firefox::set_bank(unsigned char which_bank, unsigned char value)
{
	bool result = true;
	
	switch (which_bank)
	{
	case 0:	// dipswitch 1
		banks[2] = (unsigned char) (value ^ 0xFF);	// dip switches are active low
		break;
	case 1:	// dipswitch 2
		banks[3] = (unsigned char) (value ^ 0xFF);	// switches are active low
		break;
	default:
		printline("ERROR: Bank specified is out of range!");
		result = false;
		break;
	}
	
	return result;
}
