/*
 * game.cpp
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// game.cpp
// by Matt Ownby

#ifdef WIN32
#pragma warning (disable:4100) // disable warning about unreferenced parameter

// this doesn't disable the warnings I want disabled :(
#pragma warning (disable:4244) // disable stupid warnings in MSVC's include files
#pragma warning (disable:4511)
#pragma warning (disable:4512)
#pragma warning (disable:4663)
#endif

#include <stdio.h>

// for fstat
//#include <sys/types.h>
#include <sys/stat.h>

#include <zlib.h>	// for CRC checking
#include <string>	// STL strings, useful to prevent buffer overrun
#include "../daphne.h"
#include "../io/conout.h"
#include "../io/error.h"
#include "../io/unzip.h"
#include "game.h"
#include "../ldp-out/ldp.h"
#include "../cpu/cpu.h"
#include "../cpu/mamewrap.h"
#include "../timer/timer.h"
#include "../io/input.h"
#include "../io/sram.h"
#include "../video/video.h"
#include "../video/palette.h"



using namespace std;	// for STL string to compile without problems ...

///////////////////////////////////////////////////////////

#ifdef CPU_DEBUG


// a generic example of how to setup debug names
struct addr_name game_addr_names[] =
{
	{ "Start", 0x0000 },
	{ NULL, 0 }
};
#endif

// breaks cpu execution into trace mode from whichever point the program is at
// This function is executed by typing "break" at the daphne console
// This can't be part of the class because I can't figure out how to make it work =]
void g_cpu_break (char *s)
{

	// I put this here simply to get rid of warning messages
	if (s)
	{
	}

#ifdef CPU_DEBUG
	set_cpu_trace(1);
#else
	printline("You have to compile with CPU_DEBUG defined to use the debugger!");
#endif

}


///////////////////////////////////////////////////////////

game::game() :
	m_game_uses_video_overlay(true),	// since most games do use video overlay, we'll default this to true
	m_overlay_size_is_dynamic(false),	// the overlay size is usually static
	m_video_overlay_count(2),	// default to double buffering because it is conservative
	m_active_video_overlay(0),	// the first overlay (0) starts out as the active one
	m_finished_video_overlay(0),
	m_palette_color_count(0),	// force game to specify this
	m_video_row_offset(0),	// most games will want this to be 0
	m_video_col_offset(0),
	m_video_overlay_width(0),	// " " "
	m_video_overlay_height(0),	// " " "
	m_video_overlay_needs_update(true)	// it always needs to be updated the first time
{
	memset(m_video_overlay, 0, sizeof(m_video_overlay));	// clear this structure so we can easily detect whether we are using video overlay or not
	m_disc_fps = 0.0;
	m_disc_ms_per_frame = 0.0;
	m_game_type = GAME_UNDEFINED;
	m_game_issues = NULL;	// assume game has no issues unless we specify some
#ifdef CPU_DEBUG
	addr_names = game_addr_names;
#endif
	
	m_num_sounds = 0;
	m_cheat_requested = false;	// no cheats until specifically requested
	m_shortgamename = "game";	// change this to match your game
	
	m_nvram_begin = NULL;
	m_nvram_size = 0;	// no nvram by default

	m_rom_list = NULL;
	m_crc_disabled = false;
}

// call this instead of init() directly to ensure that some universal stuff gets taken care of
bool game::pre_init()
{
	// compute how many milliseconds per frame the disc runs at
	// We use this number a lot and division is expensive so we only want to compute it once
	if (m_disc_fps != 0.0)
	{
		m_disc_ms_per_frame = 1000.0 / m_disc_fps;
	}

	// if we have nvram that we need to load
	if (m_nvram_size > 0)
	{
		char filename[81];
		
		sprintf(filename, "%s.gz", m_shortgamename);	// trail with .gz since it will be compressed
		sram_load(filename, m_nvram_begin, m_nvram_size);
	}
	
	return init();
}

// generic game initialization
bool game::init()
{
	bool result = true;
	
	cpu_init();
	return result;
}

// generic game start function.  Starts the game playing (usually just begins executing the cpu(s) )
void game::start()
{
	cpu_execute();
}

// call this instead of shutdown directory
void game::pre_shutdown()
{
	// if we have nvram that we need to save to disk
	if (m_nvram_size > 0)
	{
		char filename[81];
		
		sprintf(filename, "%s.gz", m_shortgamename);	// trail w/ gz since it'll be compressed
		sram_save(filename, m_nvram_begin, m_nvram_size);
	}

	shutdown();
}

// generic game shutdown function
void game::shutdown()
{
	cpu_shutdown();
}

// generic game reset function
void game::reset()
{
	cpu_reset();
}

// does anything special needed to send an IRQ
void game::do_irq(unsigned int which_irq)
{
	// get rid of warnings
	if (which_irq)
	{
	}

	printline("WARNING : Unhandled IRQ in generic game class!  This is probably not what you want!");
}

// does anything special needed to send an NMI
void game::do_nmi()
{
	printline("WARNING : Unhandled NMI in generic game class!  This is probably not what you want!");
}

// reads a byte from the cpu's memory
Uint8 game::cpu_mem_read(Uint16 addr)
{
	return m_cpumem[addr];
}

// writes a byte to the cpu's memory
void game::cpu_mem_write(Uint16 addr, Uint8 value)
{
	m_cpumem[addr] = value;
}

// reads a byte from the cpu's port
Uint8 game::port_read(Uint16 port)
{
	char s[81] = { 0 };

	port &= 0xFF;	
	sprintf(s, "ERROR: CPU port %x read requested, but this function is unimplemented!", port);
	printline(s);

	return(0);
}

// writes a byte to the cpu's port
void game::port_write(Uint16 port, Uint8 value)
{
	char s[81] = { 0 };

	port &= 0xFF;
	sprintf(s, "ERROR: CPU port %x write requested (value %x) but this function is unimplemented!", port, value);
	printline(s);

}

// notifies us of the new Program Counter (which most games usually don't care about)
void game::update_pc(Uint32 new_pc)
{
	// I put this here to get rid of warnings
	if (new_pc)
	{
	}
}

void game::input_enable(Uint8 input)
{
	// get rid of warnings
	if (input)
	{
	}

	printline("Warning: generic input_enable function called, does nothing");
}

void game::input_disable(Uint8 input)
{
	// get rid of warnings
	if (input)
	{
	}

	printline("Warning: generic input_disable function called, does nothing");
}

// does any video initialization we might need
// returns 'true' if the initialization was successful, false if it failed
// It is good to use generic video init and shutdown functions because then we minimize the possibility of errors such as forgetting to call palette_shutdown
bool game::video_init()
{
	bool result = false;
	int index = 0;

	// if this particular game uses video overlay (most do)
	if (m_game_uses_video_overlay)
	{
		// safety check, make sure variables are initialized like we expect them to be ...
		if ((m_video_overlay_width != 0) && (m_video_overlay_height != 0) && (m_palette_color_count != 0))
		{
			result = true;	// it's easier to assume true here and find out false, than the reverse

			// create each buffer
			for (index = 0; index < m_video_overlay_count; index++)
			{
				m_video_overlay[index] = SDL_CreateRGBSurface(SDL_SWSURFACE, m_video_overlay_width, m_video_overlay_height, 8, 0, 0, 0, 0); // create an 8-bit surface

				// check to see if we got an error (this should never happen)
				if (!m_video_overlay[index])
				{
					printline("ODD ERROR : SDL_CreateRGBSurface failed in video_init!");
					result = false;
				}
			}

			// if we created the surfaces alright, then allocate space for the color palette
			if (result)
			{
				result = palette_initialize(m_palette_color_count);
				if (result)
				{
					palette_calculate();
					palette_finalize();
				}
			}
		} // end if video overlay is used

		// if the game has not explicitely specified those variables that we need ...
		else
		{
			printerror("See video_init() inside game.cpp for what you need to do to fix a problem");
			// If your game doesn't use video overlay, set m_game_uses_video_overlay to false ...
		}
	} // end if game uses video overlay

	// else game doesn't use video overlay, so we always return true here
	else
	{
		result = true;
	}

	return(result);
}

// shuts down any video we might have initialized
void game::video_shutdown()
{
	int index = 0;

	palette_shutdown();	// de-allocate memory in color palette routines

	for (index = 0; index < m_video_overlay_count; index++)
	{
		// only free surface if it has been allocated (if we get an error in video_init, some surfaces may not be allocated)
		if (m_video_overlay[index] != NULL)
		{
			SDL_FreeSurface(m_video_overlay[index]);
			m_video_overlay[index] = NULL;
		}
	}
}

// generic function to ensure that the video buffer gets drawn to the screen, will call video_repaint()
void game::video_blit()
{
	// if something has actually changed in the game's video (video_blit() will probably get called regularly on each screen refresh,
	// and we don't want to call the potentially expensive video_repaint() unless we have to)
	if (m_video_overlay_needs_update)
	{
		m_active_video_overlay++;	// move to the next video buffer (in case we are supporting more than one buffer)

		// check for wraparound ... (the count will be the last index+1, which is why we do >= instead of >)
		if (m_active_video_overlay >= m_video_overlay_count)
		{
			m_active_video_overlay = 0;
		}
		video_repaint();	// call game-specific function to get palette refreshed
		m_video_overlay_needs_update = false;	// game will need to set this value to true next time it becomes needful for us to redraw the screen

		// if we are in non-VLDP mode, then we can blit to the main surface right here,
		// otherwise we do nothing because the yuv_callback in ldp-vldp.cpp will take care of it
		if (g_ldp->is_blitting_allowed())
		{
			SDL_Rect dest;
			dest.x = 0;	// upper left corner
			dest.y = 0;
			dest.w = (unsigned short) m_video_overlay[m_active_video_overlay]->w;
			dest.h = (unsigned short) m_video_overlay[m_active_video_overlay]->h;

			SDL_BlitSurface(m_video_overlay[m_active_video_overlay], NULL, get_screen(), &dest);
			SDL_Flip(get_screen());
		}

		m_finished_video_overlay = m_active_video_overlay;
	}
}

// forces the video overlay to be redrawn to the screen
// This is necessary when the screen has been clobbered (such as when the debug console is down)
void game::video_force_blit()
{
	m_video_overlay_needs_update = true;
	video_blit();
}

// sets up the game's palette, this is a game-specific function
void game::palette_calculate()
{
}

// redraws the current active video overlay buffer (but doesn't blit anything to the screen)
// This is a game-specific function
void game::video_repaint()
{
}

void game::set_video_overlay_needs_update(bool value)
{
	m_video_overlay_needs_update = value;
}

// generic preset function, does nothing
void game::set_preset(int preset)
{
	printline("NOTE: There are no presets defined for the game you have chosen!");
}

// generic version function, does nothing
void game::set_version(int version)
{
	printline("NOTE: There are no alternate versions defined for the game you have chosen!");
}

// returns false if there was an error trying to set the bank value (ie if the user requested an illegal bank)
bool game::set_bank(unsigned char which_bank, unsigned char value)
{
	bool result = false;	// give them an error to help them troubleshoot any problem they are having with their command line
	
	printline("ERROR: The ability to set bank values is not supported in this game.");
	
	return result;
}

// prevents daphne from verifying that the ROM images' CRC values are correct, useful when testing new ROMs
void game::disable_crc()
{
	m_crc_disabled = true;
}

// routine to load roms
// returns true if there were no errors
bool game::load_roms()
{
	bool result = true;	// we must assume result is true in case the game has no roms to load at all

	// if a rom list has been defined, then begin loading roms
	if (m_rom_list)
	{
		int index = 0;
		const struct rom_def *rom = &m_rom_list[0];
		string opened_zip_name = "";	// the name of the zip file that we currently have open (if we have one open)
		unzFile zip_file = NULL;	// pointer to open zip file (NULL if file is closed)

		// go until we get an error or we run out of roms to load
		do
		{
			string path, zip_path = "roms/";
			unsigned int crc = crc32(0L, Z_NULL, 0);

			// if this game explicitely specifies a subdirectory
			if (rom->dir)
			{
				path = rom->dir;
				zip_path += rom->dir;
				zip_path += ".zip";	// append zip extension ...
			}

			// otherwise use shortgamename by default
			else
			{
				path = m_shortgamename;
				zip_path += m_shortgamename;
				zip_path += ".zip";	// append zip extension ...
			}

			// if we have not opened a ZIP file, or if we need to open a new zip file ...
			if ((!zip_file) || (zip_path.compare(opened_zip_name) != 0))
			{
				// if we need to open a zip file ...
				if (!zip_file)
				{
					zip_file = unzOpen(zip_path.c_str());
				}

				// we need to open a different zip file ...
				else
				{
					string s = "Closing " + opened_zip_name + " and attempting to open " + zip_path;	// just for debugging to make sure this is behaving properly
					printline(s.c_str());
					unzClose(zip_file);
					zip_file = unzOpen(zip_path.c_str());
				}

				// if we successfully opened the file ...
				if (zip_file)
				{
					opened_zip_name = zip_path;
				}
			}

			result = false;
			// if we have a zip file open, try to load the ROM from this file first ...
			if (zip_file)
			{
				result = load_compressed_rom(rom->filename, zip_file, rom->buf, rom->size);
			}

			// if we were unable to open the rom from a zip file, try to open it as an uncompressed file
			if (!result)
			{
				result = load_rom(rom->filename, path.c_str(), rom->buf, rom->size);
			}

			// if file was loaded and was proper length, check CRC
			if (result) 
			{
				if (!m_crc_disabled)  //skip if user doesn't care
				{
					crc = crc32(crc, rom->buf, rom->size);
					if (rom->crc32 == 0)  //skip if crc is set to 0 (game driver doesn't care)
					{
						crc=0;  
					}
					// if CRC's don't match
					if (crc != rom->crc32)
					{
						char s[160];
						sprintf(s, "ROM CRC checked failed for %s, expected %x, got %x", rom->filename, rom->crc32, crc);
						printerror(s);	// warn them about this disaster :)
						printline(s);
					}
				}
			}

			// if ROM could not be loaded
			else
			{
				string s = "ROM ";
				s += rom->filename;
				s += " couldn't be found in roms/";
				s += path;
				s += "/, or in ";
				s += zip_path;
				printerror(s.c_str());
				// if this game borrows roms from another game, point that out to the user to help
				// them troubleshoot
				if (rom->dir)
				{
					s = "NOTE : this ROM comes from the folder '";
					s += rom->dir;
					s += "', which belongs to another game.";
					printline(s.c_str());
					s = "You also NEED to get the ROMs for the game that uses the directory '";
					s += rom->dir;
					s += "'.";
					printline(s.c_str());
				}
			}

			index++;
			rom = &m_rom_list[index];	// move to next entry
		} while (result && rom->filename);

		// if we had a zip file open, close it now
		if (zip_file)
		{
			unzClose(zip_file);
			zip_file = NULL;
		}
		
		patch_roms();
	}

	return(result);
}

// returns true if the file in question exists, and has the proper CRC32
// 'gamedir' is which rom directory (or .ZIP file) this file is expected to be in
bool game::verify_required_file(char *filename, char *gamedir, Uint32 filecrc32)
{
	FILE *readme_file = NULL;
	Uint8 *readme_test = NULL;
	bool passed_test = false;
	struct stat filestats;	// to get length of file
	string path = "roms/";

	// TRY UNCOMPRESSED FIRST
	string uncompressed_path = path + gamedir + "/" + filename;
	readme_file = fopen(uncompressed_path.c_str(), "rb");
	// if file exists, check its length
	if (readme_file)
	{
		fstat(fileno(readme_file), &filestats);	// get stats for file to get file length


		readme_test = static_cast<Uint8 *>(malloc(filestats.st_size));
		if (readme_test)
		{
			unsigned int crc = crc32(0L, Z_NULL, 0);	// zlib crc32

			// if we are able to read in
			fread(readme_test, 1, filestats.st_size, readme_file);
			
			crc = crc32(crc, readme_test, filestats.st_size);
			
			// if the DLE readme file has been unaltered, allow user to continue
			if (crc == filecrc32)
			{
				passed_test = true;
			}

			free(readme_test);
			readme_test = NULL;
		}
		fclose(readme_file);
	}

	// IF UNCOMPRESSED TEST FAILED, TRY COMPRESSED TEST ...
	if (!passed_test)
	{
		string zip_path = path + gamedir + ".zip";

		unzFile zip_file = NULL;	// pointer to open zip file (NULL if file is closed)
		zip_file = unzOpen(zip_path.c_str());
		if (zip_file)
		{
			if (unzLocateFile(zip_file, filename, 2) == UNZ_OK)
			{
				unz_file_info info;

				// get CRC
				unzGetCurrentFileInfo OF((zip_file,
					     &info,
					     NULL,
					     0,
					     NULL,
					     0,
					     NULL,
					     0));

				if (info.crc == filecrc32)
				{
					passed_test = true;
				}
			}
			unzClose(zip_file);
		}
	}

	return passed_test;
}

// loads size # of bytes from filename into buf
// returns true if successful, or false if there was an error
bool game::load_rom(const char *filename, Uint8 *buf, Uint32 size)
{
	FILE *F = NULL;
	Uint32 bytes_read = 0;
	bool result = false;
	char fullpath[81] = { "roms/" }; // pathname to roms directory
	char s[81] = { 0 };

	strcat(fullpath, filename);	// append filename to roms directory	
	outstr("Loading ");
	outstr(fullpath);
	outstr(" ... ");
	F = fopen(fullpath, "rb");
	
	// if file was opened successfully
	if (F)
	{
		bytes_read = fread(buf, 1, size, F);
		
		// if we read in the # of bytes we expected to
		if (bytes_read == size)
		{
			result = true;
		}
		// notify the user what the problem is
		else
		{
			sprintf(s, "error in rom_load: expected %u bytes but only read %u", size, bytes_read);
			printline(s);
		}
		fclose(F);
	}
	
	sprintf(s, "%d bytes read into memory", bytes_read);
	printline(s);
	
	return(result);
	
}

// transition function ...
bool game::load_rom(const char *filename, const char *directory, Uint8 *buf, Uint32 size)
{
	string full_path = directory;
	full_path += "/";
	full_path += filename;
	return load_rom(full_path.c_str(), buf, size);
}

// similar to load_rom except this function loads a rom image from a .zip file
// the previously-opened zip file is indicated by 'opened_zip_file'
// true is returned only if the rom was loaded, and it was the expected length
bool game::load_compressed_rom(const char *filename, unzFile opened_zip_file, Uint8 *buf, Uint32 size)
{
	bool result = false;

	outstr("Loading compressed ROM image ");
	outstr(filename);
	outstr("...");

	// try to locate requested rom file inside .ZIP archive and proceed if successful
	// (the '2' indicates case-insensitivity)
	if (unzLocateFile(opened_zip_file, filename, 2) == UNZ_OK)
	{
		// try to open the current file that we've located
		if (unzOpenCurrentFile(opened_zip_file) == UNZ_OK)
		{
			// read this file
			Uint32 bytes_read = (Uint32) unzReadCurrentFile(opened_zip_file, buf, size);
			unzCloseCurrentFile(opened_zip_file);

			// if we read what we expected to read ...
			if (bytes_read == size)
			{
				char s[81];
				sprintf(s, "%d bytes read.", bytes_read);
				printline(s);
				result = true;
			}
			else
			{
				printline("unexpected read result!");
			}
		}
		else
		{
			printline("could not open current file!");
		}
	}
	else
	{
		printline("file not found in .ZIP archive!");
	}

	return result;
}


// modify roms (apply cheats, for example) after they are loaded
// this can also be used for any post-rom-loading procedure, such as verifying the existence of readme files for DLE/SAE
void game::patch_roms()
{
}

// how many pixels down to shift video overlay
int game::get_video_row_offset()
{
	return m_video_row_offset;
}

// how many pixels to the right to shift video overlay
int game::get_video_col_offset()
{
	return m_video_col_offset;
}

SDL_Surface *game::get_video_overlay(int index)
{
	SDL_Surface *result = NULL;

	// safety check
	if (index < m_video_overlay_count)
	{
		result = m_video_overlay[index];
	}

	return result;
}

// gets surface that's being drawn
SDL_Surface *game::get_active_video_overlay()
{
	return m_video_overlay[m_active_video_overlay];
}

// gets last surface to be completely drawn (so it can be displayed without worrying about tearing or flickering)
SDL_Surface *game::get_finished_video_overlay()
{
	return m_video_overlay[m_finished_video_overlay];
}

// mainly used by ldp-vldp.cpp so it doesn't print a huge warning message if the overlay's size is dynamic
bool game::is_overlay_size_dynamic()
{
	return m_overlay_size_is_dynamic;
}

// enables a cheat (any cheat) in the current game.  Unlimited lives is probably the most common.
void game::enable_cheat()
{
	m_cheat_requested = true;
}

double game::get_disc_fps()
{
	return m_disc_fps;
}

// returns how many ms per frame the disc runs at
double game::get_disc_ms_per_frame()
{
	return m_disc_ms_per_frame;
}

Uint8 game::get_game_type()
{
	return m_game_type;
}

Uint32 game::get_num_sounds()
{
	return m_num_sounds;
}

const char *game::get_sound_name(int whichone)
{
	return m_sound_name[whichone];
}

// returns a string of text that explains the problems with the game OR else null if the game has no problems :)
const char *game::get_issues()
{
	return m_game_issues;
}

#ifdef CPU_DEBUG

// returns either a name for the address or else NULL if no name exists
// Used for debugging to improve readability (to assign function and variable names)
char *game::get_address_name(Uint16 addr)
{

	char *name = NULL;
	int i = 0;

	// loop until we've hit the end of our memory names or until we've found a match
	for (;;)
	{
		// if we're at the end of the list
		if (addr_names[i].name == NULL)
		{
			break;
		}
		else if (addr_names[i].address == addr)
		{
			name = addr_names[i].name;
			break;
		}
		i++;
	}

	return(name);
}

#endif
