/*
 * lair.cpp
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// Dragon's Lair/Space Ace driver for DAPHNE
// started by Matt Ownby
// contributions made by Mark Broadhead, Robert DiNapoli, Jeff Kulczycki
// If you don't see your name here, feel free to add it =]

#ifdef WIN32
#pragma warning (disable:4100) // disable warning about unreferenced parameter
#endif

// Win32 doesn't use strcasecmp, it uses stricmp (lame)
#ifdef WIN32
#define strcasecmp stricmp
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <zlib.h>	// for crc32 calculation
#include "../daphne.h"
#include "lair.h"
#include "../ldp-in/ldv1000.h"
#include "../ldp-in/pr7820.h"
#include "../ldp-out/ldp.h"
#include "../video/scoreboard.h"
#include "../video/led.h"
#include "../video/palette.h"
#include "../io/conout.h"
#include "../io/error.h"
#include "../sound/sound.h"
#include "../cpu/cpu.h"
#include "../cpu/generic_z80.h"

//////////////////////////////////////////////////////////////////////////

// lair class constructor (default the rev F2 roms)
lair::lair()
{
	m_shortgamename = "lair";
	memset(m_cpumem, 0, CPU_MEM_SIZE);
	m_switchA = 0x22;
	m_switchB = 0xD8;
	m_joyskill_val = 0xFF;	// all input cleared
	m_misc_val = 0x7F;	// bit 7 and 6 alternated, and bits 0-3 all set (bits 4-5 are unused)

    // Button3 is used to toggle on/of MPEG overlay scoreboard if -useoverlaysb
    // switch in effect. In testing a false "key up" event w/o corresponding
    // "key down" was seen on (Linux) prog startup resulting in the overlay
    // getting hidden when it wasn't supposed to. Sigh.
	m_button3_down = false;

	struct cpudef cpu;
	memset(&cpu, 0, sizeof(struct cpudef));
	cpu.type = CPU_Z80;
	cpu.hz = LAIR_CPU_HZ;
	cpu.irq_period[0] = LAIR_IRQ_PERIOD;
	cpu.nmi_period = (1000.0 / 60.0);	// dragon's lair has no NMI.  We use this to time the LD-V1000 strobes since it's convenient
	cpu.initial_pc = 0;
	cpu.must_copy_context = false;
	cpu.mem = m_cpumem;
	add_cpu(&cpu);	// add this cpu to the list (it will be our only one)

	m_disc_fps = 23.976;
	m_game_type = GAME_LAIR;

	m_game_uses_video_overlay = false;	// this game doesn't (by default) use video overlay
    m_video_overlay_needs_update = false;

	ldv1000_enable_instant_seeking();	// make the LD-V1000 perform instantaneous seeks because we can
	m_status_strobe_timer = 0;
	m_time_strobes = true;	// time full LD-V1000 strobes (this can be disabled)
	m_uses_pr7820 = false;  // only used by lairalt()

	m_num_sounds = 3;
	m_sound_name[S_DL_CREDIT] = "dl_credit.wav";
	m_sound_name[S_DL_ACCEPT] = "dl_accept.wav";
	m_sound_name[S_DL_BUZZ] = "dl_buzz.wav";

	// ROM images for Dragon's Lair (this must be static!)
	const static struct rom_def lair_roms[] =
	{
		{ "dl_f2_u1.bin", NULL, &m_cpumem[0x0000], 0x2000, 0xF5EA3B9D },
		{ "dl_f2_u2.bin", NULL, &m_cpumem[0x2000], 0x2000, 0xDCC1DFF2 },
		{ "dl_f2_u3.bin", NULL, &m_cpumem[0x4000], 0x2000, 0xAB514E5B },
		{ "dl_f2_u4.bin", NULL, &m_cpumem[0x6000], 0x2000, 0xF5EC23D2 },
		{ NULL }
	};

	m_rom_list = lair_roms;

}

void lair::set_version(int version)
{
	//Since other game classes are derived from this one,
	//we need to make sure the right game is loaded for these
	//alternate rom versions
	if (strcasecmp(m_shortgamename, "lair") == 0)
	{
		if (version == 1) //rev F2
		{
			//rev F2 is already loaded, do nothing
		}
		else if (version == 2)  //rev F
		{
			const static struct rom_def roms[] =
			{
				{ "dl_f_u1.bin", "lair", &m_cpumem[0x0000], 0x2000, 0x06fc6941 }, 
				{ "dl_f_u2.bin", "lair", &m_cpumem[0x2000], 0x2000, 0xdcc1dff2 }, //same as rev F2
				{ "dl_f_u3.bin", "lair", &m_cpumem[0x4000], 0x2000, 0xab514e5b }, //same as rev F2
				{ "dl_f_u4.bin", "lair", &m_cpumem[0x6000], 0x2000, 0xa817324e }, 
				{ NULL }
			};
			m_rom_list = roms;
		}
		else if (version == 3)  //rev E
		{
			static struct rom_def roms[] =
			{
				{ "dl_e_u1.bin", "lair", &m_cpumem[0x0000], 0x2000, 0x02980426 },
				{ "dl_e_u2.bin", "lair", &m_cpumem[0x2000], 0x2000, 0x979d4c97 },
				{ "dl_e_u3.bin", "lair", &m_cpumem[0x4000], 0x2000, 0x897bf075 },
				{ "dl_e_u4.bin", "lair", &m_cpumem[0x6000], 0x2000, 0x4ebffba5 },
				{ NULL }
			};
			m_rom_list = roms;
		}
	}
	else
	{
		//call the generic non-functional method instead
		game::set_version(version);
	}
}

// dragon's lair enhanced class constructor
dle11::dle11()
{
	m_shortgamename = "dle11";

	// NOTE : this must be static
	static struct rom_def roms[] =
	{
		{ "dle11u1l.bin", NULL, &m_cpumem[0x0000], 0x2000, 0x9E65B33D },
		{ "dle11u2l.bin", NULL, &m_cpumem[0x2000], 0x2000, 0xF16FA36F },
		{ "dle11u3l.bin", NULL, &m_cpumem[0x4000], 0x2000, 0xB8D07A16 },
		{ "dle11u4l.bin", NULL, &m_cpumem[0x6000], 0x2000, 0x20FC79B7 },
		{ NULL }
	};

	m_rom_list = roms;

}

void dle11::patch_roms()
{
	bool passed_test = false;

	// NOW check to make sure the DLE readme file is present and unaltered
	// Dave Hallock requested this check to make sure sites don't remove his readme file
	// and distribute DLE.

	passed_test = verify_required_file("readme11.txt", "dle11", 0x4BF84551);

	// if they failed the test then exit daphne
	if (!passed_test)
	{
		printerror("DLE readme11.txt file is missing or altered.");
		printerror("Please get the original readme11.txt file from www.d-l-p.com, thanks.");
		set_quitflag();
	}
}

// dragon's lair enhanced v2.0 class constructor
dle20::dle20()
{
	m_shortgamename = "dle20";

	// NOTE : this must be static
	static struct rom_def roms[] =
	{
		{ "DLE20_U1.bin", NULL, &m_cpumem[0x0000], 0x2000, 0x0ACA15B4 },
		{ "DLE20_U2.bin", NULL, &m_cpumem[0x2000], 0x2000, 0x1CEA7622 },
		{ "DLE20_U3.bin", NULL, &m_cpumem[0x4000], 0x2000, 0x21A7E7CA },
		{ "DLE20_U4.bin", NULL, &m_cpumem[0x6000], 0x2000, 0x4AE26073 },
		{ NULL }
	};

	m_rom_list = roms;

}

// DLE v2.0 test to make sure readme20.txt file is present and unaltered
void dle20::patch_roms()
{
	bool passed_test = false;

	passed_test = verify_required_file("readme20.txt", "dle20", 0x51C50010);

	// if they failed the test then exit daphne
	if (!passed_test) 
	{
		printerror("DLE readme20.txt file is missing or altered.");
		printerror("Please get the original file from www.d-l-p.com, thanks.");
		set_quitflag();
	}
}


// Space Ace class constructor
ace::ace()
{
	m_shortgamename = "ace";
	m_game_type = GAME_ACE;	
	m_switchA = 0x3D;	// 2 coins/credit, attract mode audio always on
	m_switchB = 0xFE;	// 5 lives	

	// NOTE : this must be static
	const static struct rom_def ace_roms[] =
	{
		{ "sa_a3_u1.bin", NULL, &m_cpumem[0x0000], 0x2000, 0x427522D0 },
		{ "sa_a3_u2.bin", NULL, &m_cpumem[0x2000], 0x2000, 0x18D0262D },
		{ "sa_a3_u3.bin", NULL, &m_cpumem[0x4000], 0x2000, 0x4646832D },
		{ "sa_a3_u4.bin", NULL, &m_cpumem[0x6000], 0x2000, 0x57DB2A79 },
		{ "sa_a3_u5.bin", NULL, &m_cpumem[0x8000], 0x2000, 0x85CBCDC4 },
		{ NULL }
	};

	m_rom_list = ace_roms;

}

// ace supports multiple rom revs
void ace::set_version(int version)
{
	if (version == 1) //rev A3
	{
		//rev A3 is already loaded, do nothing
	}
	else if (version == 2)  //rev A2
	{
		const static struct rom_def ace_roms[] =
		{
			{ "sa_a2_u1.bin", NULL, &m_cpumem[0x0000], 0x2000, 0x71b39e27 },
			{ "sa_a2_u2.bin", NULL, &m_cpumem[0x2000], 0x2000, 0x18D0262D },
			{ "sa_a2_u3.bin", NULL, &m_cpumem[0x4000], 0x2000, 0x4646832D },
			{ "sa_a2_u4.bin", NULL, &m_cpumem[0x6000], 0x2000, 0x57DB2A79 },
			{ "sa_a2_u5.bin", NULL, &m_cpumem[0x8000], 0x2000, 0x85CBCDC4 },
			{ NULL }
		};
		m_rom_list = ace_roms;
	}
	else if (version == 3)  //rev A
	{
		const static struct rom_def ace_roms[] =
		{
			{ "sa_a_u1.bin", NULL, &m_cpumem[0x0000], 0x2000, 0x8eb1889e },
			{ "sa_a_u2.bin", NULL, &m_cpumem[0x2000], 0x2000, 0x18D0262D },
			{ "sa_a_u3.bin", NULL, &m_cpumem[0x4000], 0x2000, 0x4646832D },
			{ "sa_a_u4.bin", NULL, &m_cpumem[0x6000], 0x2000, 0x57DB2A79 },
			{ "sa_a_u5.bin", NULL, &m_cpumem[0x8000], 0x2000, 0x85CBCDC4 },
			{ NULL }
		};
		m_rom_list = ace_roms;
	}
	else
	{
		printline("ACE:  Unsupported -version paramter, ignoring...");
	}
}	



// Space Ace class constructor
sae::sae()
{
	m_shortgamename = "sae";
	m_game_type = GAME_SAE;	
	m_switchA = 0x66;	
	m_switchB = 0x98;	

	// NOTE : this must be static
	const static struct rom_def sae_roms[] =
	{
		{ "sae10_u1.bin", NULL, &m_cpumem[0x0000], 0x2000, 0xCBC5E425 },
		{ "sae10_u2.bin", NULL, &m_cpumem[0x2000], 0x2000, 0x71A26F47 },
		{ "sae10_u3.bin", NULL, &m_cpumem[0x4000], 0x2000, 0xBAC5CDD8 },
		{ "sae10_u4.bin", NULL, &m_cpumem[0x6000], 0x2000, 0xE18380F9 },
		{ "sae10_u5.bin", NULL, &m_cpumem[0x8000], 0x2000, 0x8A536CB0 },
		{ NULL }
	};

	m_rom_list = sae_roms;

}


void sae::patch_roms()
{
	bool passed_test = false;

	// NOW check to make sure the SAE readme file is present and unaltered
	// Dave Hallock requested this check to make sure sites don't remove his readme file
	// and distribute SAE.

	passed_test = verify_required_file("readme.txt", "sae", 0xCA4E20E6);

	// if they failed the test then exit daphne
	if (!passed_test) 
	{
		printerror("The SAE readme.txt file is missing or altered.");
		printerror("Please get the original file from www.d-l-p.com, thanks.");
		set_quitflag();
	}
}


lairalt::lairalt()
{
	m_shortgamename = "lairalt";
	m_uses_pr7820 = true;
	
	// NOTE : this must be static
	static struct rom_def roms[] =
	{
		{ "dl_a_u1.bin", "lair", &m_cpumem[0x0000], 0x2000, 0xd76e83ec },
		{ "dl_a_u2.bin", "lair", &m_cpumem[0x2000], 0x2000, 0xa6a723d8 },
		{ "dl_a_u3.bin", "lair", &m_cpumem[0x4000], 0x2000, 0x52c59014 },
		{ "dl_a_u4.bin", "lair", &m_cpumem[0x6000], 0x2000, 0x924d12f2 },
		{ "dl_a_u5.bin", "lair", &m_cpumem[0x8000], 0x2000, 0x6ec2f9c1 },
		{ NULL }
	};

	m_rom_list = roms;
	//cpu.nmi_period = 0 // dragon's lair has no NMI, and the PR-7820 has 
						 //no strobes to keep track of.  
						 //FIXME:  How can we disable the NMI for lairalt()?

	set_bank(0, 0xFF);  //set more reasonable defaults for this ROM rev
	set_bank(1, 0xF7);
	
}

// for lairalt, we support multiple rom revs
void lairalt::set_version(int version)
{
	if (version == 1) //rev A
	{
		//rev A is already loaded, do nothing
	}
	else if (version == 2)  //rev B
	{
		const static struct rom_def roms[] =
		{
			{ "dl_b_u1.bin", "lair", &m_cpumem[0x0000], 0x2000, 0xd76e83ec }, //same as rev a
			{ "dl_b_u2.bin", "lair", &m_cpumem[0x2000], 0x2000, 0x6751103d },
			{ "dl_b_u3.bin", "lair", &m_cpumem[0x4000], 0x2000, 0x52c59014 }, //same as rev a
			{ "dl_b_u4.bin", "lair", &m_cpumem[0x6000], 0x2000, 0x924d12f2 }, //same as rev a
			{ "dl_b_u5.bin", "lair", &m_cpumem[0x8000], 0x2000, 0x6ec2f9c1 }, //same as rev a
			{ NULL }
		};
		m_rom_list = roms;
	}
	else if (version == 3)  //rev C
	{
		static struct rom_def roms[] =
		{
			{ "dl_c_u1.bin", "lair", &m_cpumem[0x0000], 0x2000, 0xcebfe26a },
			{ "dl_c_u2.bin", "lair", &m_cpumem[0x2000], 0x2000, 0x6751103d },
			{ "dl_c_u3.bin", "lair", &m_cpumem[0x4000], 0x2000, 0x52c59014 }, //same as rev A
			{ "dl_c_u4.bin", "lair", &m_cpumem[0x6000], 0x2000, 0x924d12f2 }, //same as rev A
			{ "dl_c_u5.bin", "lair", &m_cpumem[0x8000], 0x2000, 0x6ec2f9c1 }, //same as rev A
			{ NULL }
		};
		m_rom_list = roms;
	}
	else if (version == 4)  //rev D
	{
		m_uses_pr7820 = false;  //also LD-V1000?

		static struct rom_def roms[] =
		{
			{ "dl_d_u1.bin", "lair", &m_cpumem[0x0000], 0x2000, 0x0b5ab120 },
			{ "dl_d_u2.bin", "lair", &m_cpumem[0x2000], 0x2000, 0x93ebfffb },
			{ "dl_d_u3.bin", "lair", &m_cpumem[0x4000], 0x2000, 0x22e6591f },
			{ "dl_d_u4.bin", "lair", &m_cpumem[0x6000], 0x2000, 0x5f7212cb },
			{ "dl_d_u5.bin", "lair", &m_cpumem[0x8000], 0x2000, 0x2b469c89 },
			{ NULL }
		};
		m_rom_list = roms;
	}
/*	else if (version == 5)  //rev E
	{
		//rev E only supports the LD-V1000
		m_uses_pr7820 = false;
		
		static struct rom_def roms[] =
		{
			{ "dlu1e.bin", "lair", &m_cpumem[0x0000], 0x2000, 0x02980426 },
			{ "dlu2e.bin", "lair", &m_cpumem[0x2000], 0x2000, 0x979d4c97 },
			{ "dlu3e.bin", "lair", &m_cpumem[0x4000], 0x2000, 0x897bf075 },
			{ "dlu4e.bin", "lair", &m_cpumem[0x6000], 0x2000, 0x4ebffba5 },
			//{ "dlu5e.bin", "lair", &m_cpumem[0x8000], 0x2000, 0x6ec2f9c1 },  //not used
			{ NULL }
		};
		m_rom_list = roms;
	}
*/	
}


// lairalt old revs don't use the LDV1000, so there are no strobe -presets
void lairalt::set_preset(int preset)
{
	printline("NOTE: lairalt has no presets defined.  The -preset option will be ignored.");
}


///////////////////////////////////////////////////////////

// raise IRQ line
void lair::do_irq(unsigned int which_irq)
{
	// assume that the IRQ is 0 since we only have 1 IRQ
	Z80_ASSERT_IRQ;
}

// Dragon's Lair has no NMI
// We just use the timer as a convenience to keep track of the LD-V1000 status strobe
void lair::do_nmi()
{
	if (!m_uses_pr7820)
	{
		m_status_strobe_timer = get_total_cycles_executed(0);
	}

    // If using the overlay scoreboard, this is how we're notified
    // of an MPEG size change.
    if (m_game_uses_video_overlay && m_video_overlay_needs_update)
        video_blit();
}


void lair::cpu_mem_write(Uint16 Addr, Uint8 Value)
// Called whenever the Z80 emulator wants to write to memory
{
	char s[81] = { 0 };

	// If the program is trying to write to hardware outputs
	if (Addr >= 0xE000)
	{
		// if the program is trying to write outside of its boundaries
	  if (Addr > CPU_MEM_SIZE)
	  {
		  printline("*** FATAL: MEM SIZE exceeded!");
	  }

	  else
	  {
			  switch (Addr)
			  {
			  	// real sound chip data is sent here
			  case 0xE000:
			  	break;

				  // I believe this controlled whether the bus was in input or output mode but it is not important
				  // for emulation
			  case 0xE008:
				  // it seems to be safe to totally ignore this =]
				  break;

			  // sound chip I/O (also controls dip switches)
			  case 0xE010:
				  switch (Value)
				  {
				  case 0x0E:
					// read dip-switch 1 and put result at 0xC000
					m_cpumem[0xC000] = m_switchA;
					break;
				  case 0x0F:
					// read dip-switch 2 and put result at 0xC000
					m_cpumem[0xC000] = m_switchB;
					break;
				  default:
					  break;
				  }
				  break;

			  // laserdisc control
			  case 0xE020:
				  if (m_uses_pr7820)
				  {
					  write_pr7820(Value);
				  }
				  else
				  {
					  write_ldv1000(Value);
				  }
				  break;

			  // E030-E03F are all LED's on the Dragon's Lair scoreboard
			  case 0xE030:
				  update_player_score(0, (Value & 0x0F), 1);
				  break;
			  case 0xE031:
				  update_player_score(1, (Value & 0x0F), 1);
				  break;
			  case 0xE032:
				  update_player_score(2, (Value & 0x0F), 1);
				  break;
			  case 0xE033:
				  update_player_score(3, (Value & 0x0F), 1);
				  break;
			  case 0xE034:
				  update_player_score(4, (Value & 0x0F), 1);
				  break;
			  case 0xE035:
				  update_player_score(5, (Value & 0x0F), 1);
				  break;

				  // 10's position of how many credits are deposited
			  case 0xE036:
				  update_credits(0, Value & 0x0F);
				  break;
				  // 1's position of how many credits are deposited
			  case 0xE037:
				  update_credits(1, Value & 0x0F);
				  break;

			  case 0xE038:
				  update_player_score(0, (Value & 0x0F), 0);
				  break;
			  case 0xE039:
				  update_player_score(1, (Value & 0x0F), 0);
				  break;
			  case 0xE03A:
				  update_player_score(2, (Value & 0x0F), 0);
				  break;
			  case 0xE03B:
				  // Space Ace Skill LED
			  	if (Value == 0xCC)
			  	{
					change_led(false, false, true);
			  	}
			  	else
			  	{
				  	update_player_score(3, (Value & 0x0F), 0);
				  }
				  break;
			  case 0xE03C:
				  update_player_score(4, (Value & 0x0F), 0);
				  break;
			  case 0xE03D:
				  // Captain Skill LED
			  	if (Value == 0xCC)
			  	{
					change_led(false, true, false);
			  	}
			  	else
			  	{
				  	update_player_score(5, (Value & 0x0F), 0);
				}
				break;

				  // lives of player 1
			  case 0xE03E:
			  	// show cadet skill LED
			  	if (Value == 0xCC)
			  	{
					change_led(true, false, false);
			  	}
			  	else
			  	{
				  	update_player_lives(Value & 0x0F, 0);
				}
				break;
				// lives of player 2
			  case 0xE03F:
			  	// not sure what this does?
			  	if (Value == 0xCC)
			  	{
//			  		printline("Clears LEDs but is not needed for emulation");
			  	}
			  	else
			  	{
				  	update_player_lives(Value & 0x0F, 1);
				  }
				  break;
			  default:
			  	sprintf(s, "Unknown hardware output at %x, value of %x, PC %x", Addr, Value, Z80_GET_PC);
			  	printline(s);
			  	break;
			  } // end switch
	  } // end if we're not writing above RAM limit
	} // end if we're writing to 0xE000 region

	// else if we're writing to RAM
	else if ((Addr >= 0xA000) && (Addr <= 0xAFFF))
	{
		switch(Addr)
		{
				  // We do not emulate the sound chip correctly.  Instead we do a cheat which gives us the benefit
				  // of less CPU cycles and also better quality sound.
				  // A01C gets written to every time a beep is supposed to be played, and A01D gets the memory location where
				  // the beep data is, so whenever A01C gets a 1 written to it, we check the value of A01D to see which type
				  // of beep we are supposed to play.  If anyone ever re-assembled the Dragon's Lair ROM, this scheme would
				  // likely break.
		case 0xA01C:
				  if (Value & 1)
				  {
					  int index = m_cpumem[0xA01D] | (m_cpumem[0xA01E] << 8);	// pointer to audio data

					  // only the 'accept' sound data has a D5 as the second byte
					  if (m_cpumem[index+1] == 0xD5)
					  {
						  sound_play(S_DL_ACCEPT);
					  }
					  // only the 'credit' sound data has a 0x66 as the second byte
					  else if (m_cpumem[index+1] == 0x66)
					  {
						  sound_play(S_DL_CREDIT);
					  }
					  // only the 'buzz' sound data has a 0x99 as its third byte
					  else if (m_cpumem[index+1] == 0x99)
					  {
						  sound_play(S_DL_BUZZ);
					  }
					  // else unknown sound, play an error
					  else
					  {
						printline("WARNING : Unknown dragon's lair sound!");
					  }
				  }
				  break;
		} // end switch

		m_cpumem[Addr] = Value; // always store to RAM if we write to it

	}	// end writing to RAM (>= 0xA000)

	// if we are trying to write below 0xA000, it means we are trying to write to ROM
	else
	{
		sprintf(s, "Error, program attempting to write to ROM (%x), PC is %x", Addr, Z80_GET_PC);
		printline(s);
	}
}


// Called whenever the Z80 emulator wants to read from memory
Uint8 lair::cpu_mem_read(Uint16 Addr)
{

	Uint8 result = m_cpumem[Addr];

	// the memory range of 0xC000 to somewhere less than 0xE000 is used by Lair/Ace to read external input
	// such as the joystick, the dip switches, or the laserdisc player
	if (Addr >= 0xC000)
	{
		// if we're within memory limits (this is kind of a pointless check since we can't ever exceed 16-bits anyway)
		if (Addr < CPU_MEM_SIZE)
		{
			  switch(Addr)
			  {
			  case 0xC008:	// joystick/spaceace skill query ...
				  result = m_joyskill_val;
				  break;
			  case 0xC010:
				  result = read_C010();
				  break;
			  case 0xC020:
				  result = read_ldv1000();
				  break;
			  default:
				  break;
			  }	// end switch
		}	// end else
	} // end if addr > 0xC000 ...

	return(result);
}

bool lair::init()
{
	cpu_init();

    if (m_video_overlay[m_active_video_overlay])
        draw_overlay_scoreboard(m_video_overlay[m_active_video_overlay], true);

	return scoreboard_init();
}

// Initialize overlay scoreboard if -useroverlaysb command line switch used.
void lair::init_overlay_scoreboard()
{
    m_game_uses_video_overlay = true;
    m_overlay_size_is_dynamic = true;
    m_video_overlay_count = 1;
	m_video_overlay_width = 320;
	m_video_overlay_height = 240;
	m_palette_color_count = 256;
    set_overlay_scoreboard(true);
}

void lair::palette_calculate()
{
	SDL_Color temp_color;

	// fill color palette with sensible grey defaults
	for (int i = 0; i < 256; i++)
	{
		temp_color.r = (unsigned char) i;
		temp_color.g = (unsigned char) i;
		temp_color.b = (unsigned char) i;
		
		palette_set_color(i, temp_color);
	}
}

// frees any images we loaded in, etc
void lair::shutdown()
{
	scoreboard_shutdown();
	cpu_shutdown();
}

// redraws the scoreboard on the screen
void lair::video_repaint()
{
    if (! get_overlay_scoreboard() && NULL == m_video_overlay[m_active_video_overlay])
	    scoreboard_repaint();
    else if (m_video_overlay[m_active_video_overlay])
    {
	    Uint32 cur_w = g_ldp->get_discvideo_width() >> 1;	// width overlay should be
	    Uint32 cur_h = g_ldp->get_discvideo_height() >> 1;	// height overlay should be
	    char s[128] = {0};

	    // if the width or height of the mpeg video has changed since we last were here (ie, opening a new mpeg)
	    // then reallocate the video overlay buffer
	    if ((cur_w != m_video_overlay_width) || (cur_h != m_video_overlay_height))
	    {
		    if (g_ldp->lock_overlay(1000))
		    {
			    m_video_overlay_width = cur_w;
			    m_video_overlay_height = cur_h;

                sprintf(s, "%s : Re-allocated overlay surface (%d x %d)...",
                        m_shortgamename, m_video_overlay_width, m_video_overlay_height);

		        printline(s);

			    video_shutdown();

			    if (! video_init())
			    {
				    printline("Fatal Error trying to re-allocate overlay surface!");
				    set_quitflag();
			    }

			    g_ldp->unlock_overlay(1000);	// unblock game video overlay
		    }
		    else
		    {
			    sprintf(s, "%s : Timed out trying to get a lock on the yuv overlay", m_shortgamename);
		    }
	    } // end if dimensions are incorrect

        if (get_overlay_scoreboard())
            draw_overlay_scoreboard(m_video_overlay[m_active_video_overlay], true);
    }
}

// basically 'preset' is a macro to set a bunch of other options; useful as a good shortcut
void lair::set_preset(int preset)
{
	if (preset == 1)
	{
		printline("LD-V1000 strobes enabled!");
		m_time_strobes = true;
	}
	else if (preset == 2)
	{
		printline("LD-V1000 strobes disabled!");
		m_time_strobes = false;
	}
}

// used to set dip switch values
bool lair::set_bank(unsigned char which_bank, unsigned char value)
{
	bool result = true;
	
	switch (which_bank)
	{
	case 0:	// bank A
		m_switchA = (unsigned char) (value ^ 0xFF);	// in dragon's lair, dip switches are active low
		break;
	case 1:	// bank B
		m_switchB = (unsigned char) (value ^ 0xFF);	// switches are active low
		break;
	default:
		printline("ERROR: Bank specified is out of range!");
		result = false;
		break;
	}
	
	return result;
}

Uint8 lair::read_C010()
// emulates the C010 memory location
// bit 0 = player 1 button, bit 1 = player 2 button (clear = button is depressed)
// bit 2 = coin 1 inserted, bit 3 = coin 2 inserted
// bit 6 = status strobe, bit 7 = command strobe
{
	if (!m_uses_pr7820) //pr-7820 has no strobes, so skip all this
	{
		// if user chooses to use strobe timer
		if (m_time_strobes)
		{
			Uint64 cur_total_cycles = get_total_cycles_executed(0);
			
			// make sure that flush_cpu_timers has not been called
			if (cur_total_cycles > m_status_strobe_timer)
			{
				Uint64 elapsed_cycles = cur_total_cycles - m_status_strobe_timer;	// how many CPU cycles have elapsed since status strobe
	
				// I made these constants here to force the compiler to compute them at compile time, just in case the compiler was too stupid to do so :)
				// 0.5 is added to avoid truncation errors
	
				const static Uint32 until_status_end = (Uint32) (((LAIR_CPU_HZ / 1000000.0) * 26.0) + 0.5);
				// how many cycles per 26 uS.  Status strobe lasts 26 uS
	
				const static Uint32 until_command_start = (Uint32) (((LAIR_CPU_HZ / 1000000.0) * (26.0 + 54.0)) + 0.5);
				// how many cycles until command strobe begins, relative to status strobe begin (26 uS for status, 54 between them)

				const static Uint32 until_command_end = (Uint32) (((LAIR_CPU_HZ / 1000000.0) * (26.0 + 54.0 + 25.0)) + 0.5);
				// how many cycles until command strobe ends, relative to status strobe begin (26 uS for status, 54 in between, 25 for command)

				// if status strobe hasn't ended yet
				if (elapsed_cycles < until_status_end) // status strobe is 26 uS
				{
					m_misc_val &= 0x3f; //clear the strobes
					m_misc_val |= 0x40; //set status strobe
				}
				// if status strobe has ended, but command strobe has not begun, then clear both strobes
				else if (elapsed_cycles < until_command_start) // status to command strobe is 54 uS
				{
					m_misc_val &= 0x3f; //clear the strobes
				}
				// if command strobe has begun and hasn't ended yet
				else if (elapsed_cycles < until_command_end) // command strobe is 25 uS
				{
					m_misc_val &= 0x3f; //clear the strobes
					m_misc_val |= 0x80; //set command strobe
				}
				// if neither command nor status strobe has enabled, clear strobe values
				else 
				{
					m_misc_val &= 0x3f; //clear the strobes
				}
			}
			
			// else if flush_cpu_timers was called, we can't compute strobes, so we just flip them
			else
			{
				m_misc_val ^= 0xC0;
			}
		} // end if we're computing strobes

		// if we want to use the old, reliable method
		else
		{
			m_misc_val ^= 0xC0;	// flip top two bits to alternate strobe
		}
	}
	else  //pr-7820 handler
	{
		// the PR-7820 doesn't generate strobes, so this function is much simpler.
		// all it does is set bit 7 (READY) low when it's not busy searching
		// (if the game CPU is halted during searches, then this will never be 
		//  set high anyway.  Maybe in the future...)
		
		if (read_pr7820_ready())
		{
			m_misc_val |= 0x80;
		}
		else
		{
			m_misc_val &= ~0x80;
		}
	}

return(m_misc_val);
	
}

void lair::input_enable(Uint8 move)
{

		switch(move)
		{
		case SWITCH_UP:
			m_joyskill_val &= (unsigned char) ~1;	// clear bit 0
			break;
		case SWITCH_DOWN:
			m_joyskill_val &= (unsigned char) ~2;	// clear bit 1
			break;
		case SWITCH_LEFT:
			m_joyskill_val &= (unsigned char) ~4;	// clear bit 2
			break;
		case SWITCH_RIGHT:
			m_joyskill_val &= (unsigned char) ~8;	// clear bit 3
			break;
		case SWITCH_START1: // PLAYER 1
			m_misc_val &= ~1;	// clear bit 0 ...
			break;
		case SWITCH_START2:
			m_misc_val &= ~2;	// clear bit 1
			break;
		case SWITCH_BUTTON1: // SWORD
			m_joyskill_val &= (unsigned char) ~0x10;	// clear bit 4
			break;
		case SWITCH_BUTTON3:
			m_button3_down = true;
			break;
		case SWITCH_COIN1: 
			m_misc_val &= (unsigned char) ~0x04;	
			break;
		case SWITCH_COIN2: 
			m_misc_val &= (unsigned char) ~0x08;	
			break;
		case SWITCH_SKILL1:	// cadet
			m_joyskill_val &= (unsigned char) ~0x20;	// clear bit 5
			break;
		case SWITCH_SKILL2:	// captain
			m_joyskill_val &= (unsigned char) ~0x40;	// clear bit 6
			break;
		case SWITCH_SKILL3:	// space ace
			m_joyskill_val &= (unsigned char) ~0x80;	// clear bit 7
			break;
		default:
			//unused key, take no action

			//printline("Error, bug in Dragon's Lair's input enable");
			break;
		}
}


void lair::input_disable(Uint8 move)
{

		switch(move)
		{
		case SWITCH_UP:
			m_joyskill_val |= 1;	// set bit 0
			break;
		case SWITCH_DOWN:
			m_joyskill_val |= 2;	// set bit 1
			break;
		case SWITCH_LEFT:
			m_joyskill_val |= 4;	// set bit 2
			break;
		case SWITCH_RIGHT:
			m_joyskill_val |= 8;	// set bit 3
			break;
		case SWITCH_START1:
			m_misc_val |= 1; // PLAYER 1
			break;
		case SWITCH_START2:
			m_misc_val |= 2;	// set bit 1
			break;
		case SWITCH_BUTTON1: // SWORD
			m_joyskill_val |= 0x10;	// set bit 4
			break;
        case SWITCH_BUTTON3:    // Hide or show overlay scoreboard (if used).
            if (m_video_overlay[m_active_video_overlay] && m_button3_down)
            {
                if (get_overlay_scoreboard())
                {
                    draw_overlay_scoreboard(m_video_overlay[m_active_video_overlay], false);
                    set_overlay_scoreboard(false);
                }
                else
                {
                    draw_overlay_scoreboard(m_video_overlay[m_active_video_overlay], true);
                    set_overlay_scoreboard(true);
                }
            }

	        m_button3_down = false;
	        break;
		case SWITCH_COIN1: 
			m_misc_val |= 0x04;	
			break;
		case SWITCH_COIN2: 
			m_misc_val |= 0x08;	
			break;
		case SWITCH_SKILL1:	// cadet
			m_joyskill_val |= (unsigned char) 0x20;	// set bit 5
			break;
		case SWITCH_SKILL2:	// captain
			m_joyskill_val |= (unsigned char) 0x40;	// set bit 6
			break;
		case SWITCH_SKILL3:	// space ace
			m_joyskill_val |= (unsigned char) 0x80;	// set bit 7
			break;
		default:
			//unused key, take no action
			
			//printline("Error, bug in Dragon's Lair's move disable");
			break;
		}
}
