/*
 * laireuro.cpp
 *
 * Copyright (C) 2001 Mark Broadhead
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// laireuro.cpp
// by Mark Broadhead
//
// this driver is very much incomplete, in fact, I shouldn't have submitted yet, hehe
//
// the main things it needs is emulation of the CTC Z8430A and DART Z8470 chips.
// in case anybody wants to work on this before I get back to it a datasheet that covers 
// those two chips can be found at http://www.zilog.com/docs/z80/um0081.pdf

#include <string.h>
#include "laireuro.h"
#include "../daphne.h"
#include "../cpu/cpu.h"
#include "../cpu/generic_z80.h"
#include "../ldp-out/ldp.h"
#include "../io/conout.h"
#include "../video/video.h"

laireuro::laireuro()
{
	struct cpudef cpu;
	
	m_shortgamename = "laireuro";
	memset(&cpu, 0, sizeof(struct cpudef));
	memset(banks, 0xFF, 4);	// fill banks with 0xFF's

	m_disc_fps = 25; // TODO: look up pal frame rate
	m_game_type = GAME_LAIREURO;

	cpu.type = CPU_Z80;
	cpu.hz = LAIREURO_CPU_HZ;
	cpu.irq_period[0] = LAIREURO_IRQ_PERIOD;
	cpu.must_copy_context = false;
	cpu.mem = m_cpumem;
	add_cpu(&cpu);	// add this cpu to the list (it will be our only one)

	const static struct rom_def laireuro_roms[] =
	{
		// z80 program
		{ "u45", NULL, &m_cpumem[0x0000], 0x2000, 0x4D3A9EAC },
		{ "u46", NULL, &m_cpumem[0x2000], 0x2000, 0x8479612B },
		{ "u47", NULL, &m_cpumem[0x4000], 0x2000, 0x6a66f6b4 },
		{ "u48", NULL, &m_cpumem[0x6000], 0x2000, 0x36575106 },

		// character tiles
		{ "u33", NULL, &character[0x0000], 0x2000, 0xe7506d96 },
		{ NULL }
	};

	m_rom_list = laireuro_roms;

}


void laireuro::do_irq(unsigned int which_irq)
{
	// Redraws the screen (if needed) on interrupt
	video_blit();
//	Z80_ASSERT_IRQ;
}

Uint8 laireuro::cpu_mem_read(Uint16 addr)
{
	Uint8 result = m_cpumem[addr];

	// main rom
	if (addr >= 0x0000 && addr <= 0x9fff)
	{
	}

	// ram
	else if (addr >= 0xa000 && addr <= 0xa7ff)
	{
	}

	// video ram
	else if (addr >= 0xc000 && addr <= 0xc7ff)
	{
		m_video_overlay_needs_update = true;
	}
	
	else
	{
		char s[81] = { 0 };
		sprintf(s, "Unmapped read from %x", addr);
		printline(s);
	}

	return result;
}

void laireuro::cpu_mem_write(Uint16 addr, Uint8 value)
// Called whenever the Z80 emulator wants to write to memory
{	
	m_cpumem[addr] = value;

	// main rom
	if (addr >= 0x0000 && addr <= 0x9fff)
	{
		printline("ERROR: WRITE TO MAIN ROM");
	}

	// ram
	else if (addr >= 0xa000 && addr <= 0xa7ff)
	{
	}

	// video ram
	else if (addr >= 0xc000 && addr <= 0xc7ff)
	{
		m_video_overlay_needs_update = true;
	}
	
	// watchdog reset
	else if (addr == 0xe030)
	{
	}

	else
	{
		char s[81] = { 0 };
		sprintf(s, "Unmapped write to %x with %x", addr, value);
		printline(s);
	}
}

void laireuro::port_write(Uint16 port, Uint8 value)
// Called whenever the emulator wants to output to a port
{

    char s[81] = { 0 };

    port &= 0xFF;	// strip off high byte

    switch(port)
	{
	// CTC I/O
	case 0x00:	
	case 0x01:	
	case 0x02:	
	case 0x03:	
		CTC_register[port] = value;
		break;
	// UART (SIO) 
	case 0x80:
	case 0x81:
	case 0x82:
	case 0x83:
        sprintf(s, "LAIREURO: Unsupported Port Output-> %x:%x (PC is %x)", port, value, Z80_GET_PC);
        printline(s);
		break;
	default:
        sprintf(s, "LAIREURO: Unsupported Port Output-> %x:%x (PC is %x)", port, value, Z80_GET_PC);
        printline(s);
        break;

	}		
}

Uint8 laireuro::port_read(Uint16 port)
// Called whenever the emulator wants to read from a port
{

    char s[81] = { 0 };
    unsigned char result = 0;

	port &= 0xFF;	// strip off high byte

    switch(port)
    {
	// CTC I/O
	case 0x00:	
	case 0x01:	
	case 0x02:	
	case 0x03:	
		result = CTC_register[port];
		break;
	// UART (SIO) 
	case 0x80:
	case 0x81:
	case 0x82:
	case 0x83:
        sprintf(s, "LAIREURO: Unsupported Port Input-> %x (PC is %x)", port, Z80_GET_PC);
        printline(s);
		break;
    default:
        sprintf(s, "LAIREURO: Unsupported Port Input-> %x (PC is %x)", port, Z80_GET_PC);
        printline(s);
        break;
    }

    return(result);

}

void laireuro::palette_calculate()
{
}

// updates laireuro's video
void laireuro::video_repaint()
{
	for (int charx = 0; charx < 32; charx++)
	{
		for (int chary = 0; chary < 16; chary++)
		{
			for (int y = 0; y < 16; y++)
			{
				for (int x = 0; x < 8; x++)
				{
					Uint8 pixel = static_cast<Uint8>(character[(m_cpumem[chary * 64 + charx * 2 + 0xc001]*16+y) | ((m_cpumem[0xe028] & 0x02) << 11)] & (0x01 << x));
					*((Uint8 *) m_video_overlay[m_active_video_overlay]->pixels + ((chary * 16 + y) * LAIREURO_OVERLAY_W) + (charx * 8 + x)) = pixel?m_cpumem[chary * 64 + charx * 2 + 0xc002]:((m_cpumem[0xe028] & 0x02)?0:4);
				}
			}
		}
	}
}

// this gets called when the user presses a key or moves the joystick
void laireuro::input_enable(Uint8 move)
{
	switch (move)
	{
	case SWITCH_UP:
		banks[0] &= ~0x80; 
		break;
	case SWITCH_LEFT:
		banks[0] &= ~0x02;
		break;
	case SWITCH_RIGHT:
		banks[0] &= ~0x20;
		break;
	case SWITCH_DOWN:
		banks[0] &= ~0x08; 
		break;
	case SWITCH_BUTTON1: // '1' on keyboard
		banks[1] &= ~0x80; 
		break;
	case SWITCH_BUTTON2: // '2' on keyboard
		banks[1] &= ~0x40;
		break;
	case SWITCH_BUTTON3: // space on keyboard
		banks[1] &= ~0x20;
		break;
	case SWITCH_COIN1: 
		banks[1] &= ~0x08;
		break;
	case SWITCH_COIN2: 
		banks[1] &= ~0x04;
		break;
	case SWITCH_SERVICE: 
		banks[1] &= ~0x01;
		break;
	case SWITCH_TEST: 
		banks[1] &= ~0x02;
		break;
	default:
		printline("Error, bug in move enable");
		break;
	}
}  

// this gets called when the user releases a key or moves the joystick back to center position
void laireuro::input_disable(Uint8 move)
{
	switch (move)
	{
	case SWITCH_UP:
		banks[0] |= 0x80; 
		break;
	case SWITCH_LEFT:
		banks[0] |= 0x02;
		break;
	case SWITCH_RIGHT:
		banks[0] |= 0x20;
		break;
	case SWITCH_DOWN:
		banks[0] |= 0x08; 
		break;
	case SWITCH_BUTTON1: // '1' on keyboard
		banks[1] |= 0x80; 
		break;
	case SWITCH_BUTTON2: // '2' on keyboard
		banks[1] |= 0x40;
		break;
	case SWITCH_BUTTON3: // space on keyboard
		banks[1] |= 0x20;
		break;
	case SWITCH_COIN1: 
		banks[1] |= 0x08;
		break;
	case SWITCH_COIN2: 
		banks[1] |= 0x04;
		break;
	case SWITCH_SERVICE: 
		banks[1] |= 0x01;
		break;
	case SWITCH_TEST: 
		banks[1] |= 0x02;
		break;
	default:
		printline("Error, bug in move enable");
		break;
	}
}
