/*
 * seektest.cpp
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// seektest.cpp
// by Matt Ownby

// used to test mpeg seeking accuracy

#include <string.h>
#include "../io/conout.h"
#include "../ldp-out/ldp.h"
#include "seektest.h"
#include "../daphne.h"
#include "../io/input.h"
#include "../video/palette.h"
#include "../video/video.h"

seektest::seektest()
: m_frame_offset(0),
	m_overlay(true),
	m_locked(true)
{
	m_shortgamename = "seektest";
	m_disc_fps = 29.97;	// all these values ought to be changed.. use a preset
	m_early1 = 1;
	m_early2 = 1;
	m_late1 = 1;
	m_late2 = 1;
	strcpy(m_name, "[Undefined game]");
	m_video_overlay_width = 320;	// sensible default
	m_video_overlay_height = 240;	// " " "
	m_palette_color_count = 256;
	m_overlay_size_is_dynamic = true;	// this 'game' does reallocate the size of its overlay
}

void seektest::start()
{
	go(m_early1);	// seek to the first frame to get us started

	while (!get_quitflag())
	{
		m_video_overlay_needs_update = true;
		video_blit();
		SDL_check_input();
		SDL_Delay(10);	// don't hog CPU
	}
}

// do the search, adjusting by the frame offset
void seektest::go(Uint16 target_frame)
{
	char s[81] = { 0 };

	target_frame = (Uint16) (target_frame + m_frame_offset);
	sprintf(s, "%05u", target_frame);

	g_ldp->pre_search(s);
}

void seektest::input_enable(Uint8 input)
{
	switch (input)
	{
	case SWITCH_UP:
		// if we're currently looking at early1, then seek to early2
		if ((g_ldp->get_current_frame() + m_frame_offset) == m_early1)
		{
			if (m_locked)
			{
				go((Uint16) (m_early2 - (m_frame_offset << 1)));
			}
			else
			{
				m_frame_offset = m_early2 - g_ldp->get_current_frame();
			}
		}
		else
		{
			if (m_locked)
			{
				go((Uint16) (m_early1 - (m_frame_offset << 1)));
			}
			else
			{
				m_frame_offset = m_early1 - g_ldp->get_current_frame();
			}
		}
		break;
	case SWITCH_DOWN:
		// if we're currently looking at late1, then seek to late2
		if ((g_ldp->get_current_frame() + m_frame_offset) == m_late1)
		{
			if (m_locked)
			{
				go((Uint16) (m_late2 - (m_frame_offset << 1)));
			}
			else
			{
				m_frame_offset = m_late2 - g_ldp->get_current_frame();
			}
		}
		else
		{
			if (m_locked)
			{
				go((Uint16) (m_late1 - (m_frame_offset << 1)));
			}
			else
			{
				m_frame_offset = m_late1 - g_ldp->get_current_frame();
			}
		}
		break;
	case SWITCH_LEFT:
		if (m_locked)
		{
			go((Uint16) (g_ldp->get_current_frame() - 1 - m_frame_offset));
		}
		else
		{
			m_frame_offset--;
		}
		break;
	case SWITCH_RIGHT:
		if (m_locked)
		{
			go((Uint16) (g_ldp->get_current_frame() + 1 - m_frame_offset));
		}
		else
		{
			m_frame_offset++;
		}
		break;
	case SWITCH_BUTTON1:
		// toggle locked status
		if (m_locked)
		{
			m_locked = false;
		}
		else
		{
			m_locked = true;
		}
		break;
	case SWITCH_BUTTON2:	// toggle video overlay
		m_overlay = !m_overlay;
		break;
	case SWITCH_START1:
		g_ldp->pre_play();
		break;
	case SWITCH_START2:
		g_ldp->pre_pause();
		break;
	}
}

void seektest::input_disable(Uint8 input)
{
	// get rid of warnings
	if (input)
	{
	}
}

// set which disc you're testing
void seektest::set_preset(int val)
{
	switch (val)
	{
	case 0:	// dragon's lair
		m_disc_fps = 23.976;
		m_early1 = 323;
		m_early2 = 322;
		m_late1 = 31615;
		m_late2 = 31616;
		strcpy(m_name, "Dragon's Lair NTSC");
		break;
	case 1:	// space ace
		m_disc_fps = 23.976;
		m_early1 = 1161;	// end of attract mode
		m_early2 = 1162;	// first frame of There's Borf's Ship
		m_late1 = 33185;	// black screen (end of victory)
		m_late2 = 33186;	// first frame of Dexter scene in final showdown
		strcpy(m_name, "Space Ace '83 NTSC");
		break;
	case 2:	// super don
		m_disc_fps = 29.97;
		m_early1 = 5400;	// black frame
		m_early2 = 5401;	// purple-blue gradiant frame 'super don quixote'
		m_late1 = 35850;	// black frame
		m_late2 = 35851;	// first frame of witch and the announcer saying 'game over'
		strcpy(m_name, "Super Don Quix-ote");
		break;
	case 3:	// cliff hanger
		m_disc_fps = 29.97;
		m_early1 = 1544;	// last frame of attract mode with white text
		m_early2 = 1545;	// first frame of 1st stage still with white text
		m_late1 = 49665;	// blank screen
		m_late2 = 49666;	// first grey slide at end of disc
		strcpy(m_name, "Cliff Hanger");
		break;
	case 4:	// astron belt
		m_disc_fps = 29.97;
		m_early1 = 1940;	// space with small ship on left of screen
		m_early2 = 1941;	// first frame of the first explosion on the disc
		m_late1 = 51330;	// black screen
		m_late2 = 51331;	// first frame of right speaker test
		strcpy(m_name, "Astron Belt");
		break;
	case 5:	// galaxy ranger/star blazer
		m_disc_fps = 29.97;
		m_early1 = 1017;	// black screen
		m_early2 = 1018;	// start of launch scene right after titles
		m_late1 = 51667;	// black screen
		m_late2 = 51668;	// first frame of right speaker test
		strcpy(m_name, "Galaxy Ranger / Star Blazer");
		break;
	case 6:	// Thayer's Quest
		m_disc_fps = 29.97;
		m_early1 = 716;	// logo with light to the upper left of the T
		m_early2 = 717;	// logo with light having moved to the h a little bit
		m_late1 = 49990;	// relics of quoid
		m_late2 = 49991;	// sorsabal telling the black magician to walk into the camera :)
		strcpy(m_name, "Thayer's Quest Arcade NTSC");
		break;
	case 7:	// Cobra Command
		m_disc_fps = 29.97;
		m_early1 = 300;	// black frame
		m_early2 = 301;	// first frame that's not black.. helicopter coming over mountains
		m_late1 = 46152;	// a totally white frame
		m_late2 = 46153;	// a totally black frame
		strcpy(m_name, "Cobra Command / Thunderstorm");
		break;
	case 8:	// Esh's Aurunmilla (info from Antonio "Ace64" Carulli)
		m_disc_fps = 29.97;
		m_early1 = 450;	// black frame
		m_early2 = 451;	// first frame of attract
//		m_late1 = 39154;	// blue frame, last frame of last scene (before final game over)
//		m_late2 = 39155;	// black frame
		m_late1 = 39348;	// last frame of last game over
		m_late2 = 39349;	// black frame
		strcpy(m_name, "Esh's Aurunmilla");
		break;
	case 9:	// Badlands
		m_disc_fps = 29.97;
		m_early1 = 150;	// black frame
		m_early2 = 151;	// first frame of attract
		m_late1 = 44249;	// last frame of color test
		m_late2 = 44250;	// first frame of cross hatch
		strcpy(m_name, "Badlands");
		break;
	case 10:	// Bega's Battle
		m_disc_fps = 29.97;
		m_early1 = 300;	// first frame of attract
		m_early2 = 299;	// black frame
		m_late1 = 46733;	// last frame of starburst
		m_late2 = 46734;	// black frame
		strcpy(m_name, "Bega's Battle");
		break;
	case 11:	// Freedom Fighter
		m_disc_fps = 29.97;
		m_early1 = 30;	// yellow slide
		m_early2 = 31;	// magenta slide
		m_late1 = 43049;	// last non-black frame on disc (blue, train going into sky)
		m_late2 = 43050;	// black frame (second to last on disc)
		strcpy(m_name, "Freedom Fighter");
		break;
	case 12:	// GP World
		m_disc_fps = 29.97;
		m_early1 = 5039; // black
		m_early2 = 5040; // first frame of logo
		m_late1 = 39569; // "next race"
		m_late2 = 39570; // black
		strcpy(m_name, "GP World");
		break;
	case 13:	// Dragon's Lair PAL (Original)
		m_disc_fps = 25.0;	// PAL speed
		m_early1 = 323 - 152;
		m_early2 = 322 - 152;
		// don't have any idea what the end frames should be
		strcpy(m_name, "Dragon's Lair PAL (Original)");
		break;
	case 14:	// Dragon's Lair PAL (Software Corner)
		m_disc_fps = 25.0;
		m_early1 = 323 - 230;
		m_early2 = 322 - 230;
		// don't know what to do for late frames
		strcpy(m_name, "Dragon's Lair PAL (Software Corner)");
		break;
	case 15:	// Interstellar
		m_disc_fps = 29.97;
		m_early1 = 1800; // black
		m_early2 = 1801; // first frame of attract
		m_late1 = 38340; // black
		m_late2 = 38341; // color test
		strcpy(m_name, "Interstellar");
		break;
	case 16:	// Dragon's Lair 2 (one single large mpeg2, obviously not appropriate for split mpegs)
		m_disc_fps = 29.97;
		m_early1 = 6;	// Dragon's Lair II TIMEWARP logo
		m_early2 = 7;	// white grid on black background
		m_late1 = 46349;	// last frame of test pattern
		m_late2 = 46350;	// totally black frame
		strcpy(m_name, "Dragon's Lair 2");
		break;
	default:
		printline("SEEKTEST ERROR : unknown preset");
		break;
	}
}

void seektest::palette_calculate()
{
	SDL_Color temp_color;		
	// fill color palette with sensible grey defaults
	for (int i = 0; i < 256; i++)
	{
		temp_color.r = (unsigned char) i;
		temp_color.g = (unsigned char) i;
		temp_color.b = (unsigned char) i;
		palette_set_color(i, temp_color);
	}
}

// redraws video
void seektest::video_repaint()
{
	Uint32 cur_w = g_ldp->get_discvideo_width() >> 1;	// width overlay should be
	Uint32 cur_h = g_ldp->get_discvideo_height() >> 1;	// height overlay should be
	char s[81] = { 0 };

	// if the width or height of the mpeg video has changed since we last were here (ie, opening a new mpeg)
	// then reallocate the video overlay buffer
	if ((cur_w != m_video_overlay_width) || (cur_h != m_video_overlay_height))
	{
		printline("SEEKTEST : Surface does not match mpeg, re-allocating surface!"); // remove me
		if (g_ldp->lock_overlay(1000))
		{
			m_video_overlay_width = cur_w;
			m_video_overlay_height = cur_h;
			video_shutdown();
			if (!video_init())
			{
				printline("Fatal Error, trying to re-create the surface failed!");
				set_quitflag();
			}
			g_ldp->unlock_overlay(1000);	// unblock game video overlay
		}
		else
		{
			printline("SEEKTEST ERROR : Timed out trying to get a lock on the yuv overlay");
			return;
		}
	} // end if dimensions are incorrect

	SDL_FillRect(m_video_overlay[m_active_video_overlay], NULL, 0);	// erase anything on video overlay

	// if video overlay is enabled
	if (m_overlay)
	{
		draw_string(m_name, 1, 1, m_video_overlay[m_active_video_overlay]);	// draw the game name

		sprintf(s, "%u x %u", cur_w << 1, cur_h << 1);
		draw_string(s, 1, 2, m_video_overlay[m_active_video_overlay]);	// draw the resolution of the mpeg
		sprintf(s, "Current frame : %d", g_ldp->get_current_frame() + m_frame_offset);
		if (m_locked)
		{
			strcat(s, " (LOCKED)");
		}
		else
		{
			strcat(s, " (UNLOCKED)");
		}
		draw_string(s, 1, 3, m_video_overlay[m_active_video_overlay]);
	
		if (m_frame_offset != 0)
		{
			sprintf(s, "* Adjust framefile by %d frames *", m_frame_offset);
			draw_string(s, 1, 4, m_video_overlay[m_active_video_overlay]);
		}

		draw_string("1     - Start playing the disc at 1X", 1, 11, m_video_overlay[m_active_video_overlay]);
		draw_string("FIRE1 - Lock/Unlock frame", 1, 12, m_video_overlay[m_active_video_overlay]);
		draw_string("FIRE2 - Toggle Text Overlay", 1, 13, m_video_overlay[m_active_video_overlay]);
		sprintf(s, "UP    - toggle between %u and %u", m_early1, m_early2);
		draw_string(s, 1, 14, m_video_overlay[m_active_video_overlay]);

		sprintf(s, "DOWN  - toggle between %u and %u", m_late1, m_late2);
		draw_string(s, 1, 15, m_video_overlay[m_active_video_overlay]);
	
		draw_string("LEFT  - steps down one frame", 1, 16, m_video_overlay[m_active_video_overlay]);
		draw_string("RIGHT - steps up one frame", 1, 17, m_video_overlay[m_active_video_overlay]);
	} // end if overlay is enabled
	
	// else if overlay is disabled, don't draw anything
}
