/*
 * cmdline.cpp
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// Parses our command line and sets our variables accordingly
// by Matt Ownby

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "cmdline.h"
#include "conout.h"
#include "network.h"
#include "input.h"	// to disable joystick use
#include "../video/video.h"
#include "../video/led.h"
#include "../daphne.h"
#include "../game/lair.h"
#include "../game/cliff.h"
#include "../cpu/cpu.h"
#include "../cpu/cpu-debug.h"
#include "../game/game.h"
#include "../game/superd.h"
#include "../game/thayers.h"
#include "../game/speedtest.h"
#include "../game/seektest.h"
#include "../game/cputest.h"
#include "../game/multicputest.h"
#include "../game/firefox.h"
#include "../game/ffr.h"
#include "../game/astron.h"
#include "../game/esh.h"
#include "../game/laireuro.h"
#include "../game/badlands.h"
#include "../game/starrider.h"
#include "../game/bega.h"
#include "../game/cobraconv.h"
#include "../game/gpworld.h"
#include "../game/dl2.h"
#include "../game/interstellar.h"
#include "../game/benchmark.h"
#include "../ldp-out/ldp.h"
#include "../ldp-out/sony.h"
#include "../ldp-out/pioneer.h"
#include "../ldp-out/ld-v6000.h"
#include "../ldp-out/hitachi.h"
#include "../ldp-out/philips.h"
#include "../ldp-out/ldp-combo.h"
#include "../ldp-out/ldp-vldp.h"
#include "../ldp-out/ldp-vldp-legacy.h"

int g_argc = 0;
char **g_argv = NULL;
int g_arg_index = 0;

// Win32 doesn't use strcasecmp, it uses stricmp (lame)
#ifdef WIN32
#define strcasecmp stricmp
#endif


// parses the game type from the command line and allocates g_game
// returns true if successful or false of failed
bool parse_game_type()
{
	bool result = true;
	char s[81] = { 0 };

	// first thing we need to get from the command line is the game type
	get_next_word(s, sizeof(s));

	net_set_gamename(s);	// report to server the game we are running

	if (strcasecmp(s, "ace") == 0)
	{
		g_game = new ace();
	}
	else if (strcasecmp(s, "aceeuro")==0)
	{
		g_game = new aceeuro();
	}
	else if (strcasecmp(s, "astron")==0)
	{
		g_game = new astronh();
	}
	else if (strcasecmp(s, "astronp")==0)
	{
		g_game = new astron();
	}
	else if (strcasecmp(s, "badlands")==0)
	{
		g_game = new badlands();
	}
	else if (strcasecmp(s, "bega")==0)
	{
		g_game = new bega();
	}
	else if (strcasecmp(s, "benchmark")==0)
	{
		g_game = new benchmark();
	}
	else if (strcasecmp(s, "blazer")==0)
	{
		g_game = new blazer();
	}
	else if (strcasecmp(s, "cliff")==0)
	{
		g_game = new cliff();
	}
	else if (strcasecmp(s, "cliffalt")==0)
	{
		g_game = new cliffalt();
	}
	else if (strcasecmp(s, "cobra")==0)
	{
		g_game = new cobra();
	}
	else if (strcasecmp(s, "cobraab")==0)
	{
		g_game = new cobraab();
	}
	else if (strcasecmp(s, "cobraconv")==0)
	{
		g_game = new cobraconv();
	}
	else if (strcasecmp(s, "cputest")==0)
	{
		g_game = new cputest();
	}
	else if (strcasecmp(s, "dle11") == 0)
	{
		g_game = new dle11();
	}
	else if (strcasecmp(s, "dle20") == 0)
	{
		g_game = new dle20();
	}
	else if (strcasecmp(s, "esh")==0)
	{
		g_game = new esh();
	}
	else if (strcasecmp(s, "firefox")==0)
	{
		g_game = new firefox();
	}
	else if (strcasecmp(s, "firefoxa")==0)
	{
		g_game = new firefoxa();
	}
	else if (strcasecmp(s, "ffr")==0)
	{
		g_game = new ffr();
	}
	else if (strcasecmp(s, "galaxy")==0)
	{
		g_game = new galaxy();
	}
	else if (strcasecmp(s, "gpworld")==0)
	{
		g_game = new gpworld();
	}
	else if (strcasecmp(s, "gtg") == 0)
	{
		g_game = new gtg();
	}
	else if (strcasecmp(s, "interstellar")==0)
	{
		g_game = new interstellar();
	}
	else if (strcasecmp(s, "lair")==0)
	{
		g_game = new lair();
	}
	else if (strcasecmp(s, "lairalt") == 0)
	{
		g_game = new lairalt();
	}
	else if (strcasecmp(s, "laireuro")==0)
	{
		g_game = new laireuro();
	}
	else if (strcasecmp(s, "mcputest")==0)
	{
		g_game = new mcputest();
	}
	else if (strcasecmp(s, "sae") == 0)
	{
		g_game = new sae();
	}
	else if (strcasecmp(s, "seektest")==0)
	{
		g_game = new seektest();
	}
	else if (strcasecmp(s, "speedtest")==0)
	{
		g_game = new speedtest();
	}
	else if (strcasecmp(s, "sdq")==0)
	{
		g_game = new superd();
	}
	else if (strcasecmp(s, "sdqshort")==0)
	{
		g_game = new sdqshort();
	}
	else if (strcasecmp(s, "sdqshortalt")==0)
	{
		g_game = new sdqshortalt();
	}
	else if (strcasecmp(s, "starrider")==0)
	{
		g_game = new starrider();
	}
	else if (strcasecmp(s, "superdon")==0)  //left in for old times' sake
	{
		g_game = new superd();
	}
	else if (strcasecmp(s, "tq")==0)
	{
		g_game = new thayers();
	}
	else
	{
		outstr("ERROR: Unknown game type specified : ");
		printline(s);
		result = false;
	}

	// safety check
	if (!g_game)
	{
		result = false;
	}

	return(result);
}

// parses the LDP type from the command line and allocates g_ldp
// returns true if successful or false if failed
bool parse_ldp_type()
{
	bool result = true;
	char s[81] = { 0 };

	get_next_word(s, sizeof(s));

	net_set_ldpname(s);	// report to server which ldp we are using

	if (strcasecmp(s, "combo")==0)
	{
		g_ldp = new combo();
	}
	else if (strcasecmp(s, "hitachi")==0)
	{
		g_ldp = new hitachi();
	}
	else if (strcasecmp(s, "noldp")==0)
	{
		g_ldp = new ldp();	// generic interface
	}
	else if (strcasecmp(s, "philips")==0)
	{
		g_ldp = new philips();
	}
	else if (strcasecmp(s, "pioneer") == 0)
	{
		g_ldp = new pioneer();
	}
	else if (strcasecmp(s, "sony")==0)
	{
		g_ldp = new sony();
	}
	else if (strcasecmp(s, "v6000")==0)
	{
		g_ldp = new v6000();
	}
	else if (strcasecmp(s, "vldp")==0)
	{
		g_ldp = new ldp_vldp();
	}
	else if (strcasecmp(s, "vldp_legacy")==0)
	{
		g_ldp = new ldp_vldp_legacy();
	}
	else
	{
		printline("ERROR: Unknown laserdisc player type specified");
		result = false;
	}

	// safety check
	if (!g_ldp)
	{
		result = false;
	}

	return(result);
}

// parses command line
// Returns 1 on success or 0 on failure
bool parse_cmd_line(int argc, char **argv)
{
	bool result = true;
	char s[320] = { 0 };	// for huge long string
	int i = 0;

	//////////////////////////////////////////////////////////////////////////////////////

	g_argc = argc;
	g_argv = argv;
	g_arg_index = 1;	// skip name of executable from command line

	// if game and ldp types are correct
	if (parse_game_type() && parse_ldp_type())
	{		
  	  // while we have stuff left in the command line to parse
	  for (;;)
	  {
		get_next_word(s, sizeof(s));

		// if there is nothing left for us to parse, break out of the while loop
		if (s[0] == 0)
		{
			break;
		}

		// if user wants laserdisc player to blank video while searching (VLDP only)
		else if (strcasecmp(s, "-blank_searches")==0)
		{
			g_ldp->set_search_blanking(true);
		}
		
		// if user wants to laserdisc player to blank video while skipping (VLDP only)
		else if (strcasecmp(s, "-blank_skips")==0)
		{
			g_ldp->set_skip_blanking(true);
		}

		// if they are pointing to a framefile to be used by VLDP
		else if (strcasecmp(s, "-framefile")==0)
		{
			ldp_vldp *cur_ldp = dynamic_cast<ldp_vldp *>(g_ldp);	// see if the currently selected LDP is VLDP
			get_next_word(s, sizeof(s));
			// if it is a vldp, then this option has meaning and we can use it
			if (cur_ldp)
			{
				cur_ldp->set_framefile(s);
			}
			else
			{
				printline("You can only set a framefile using VLDP as your laserdisc player!");
				result = 0;
			}
		}

		// if they are defining an alternate soundtrack to be used by VLDP
		else if (strcasecmp(s, "-altaudio")==0)
		{
			ldp_vldp *cur_ldp = dynamic_cast<ldp_vldp *>(g_ldp);	// see if the currently selected LDP is VLDP
			get_next_word(s, sizeof(s));
			// if it is a vldp, then this option has meaning and we can use it
			if (cur_ldp)
			{
				cur_ldp->set_altaudio(s);
			}
			else
			{
				printline("You can only set an alternate soundtrack when using VLDP as your laserdisc player!");
				result = 0;
			}
		}
		// to disable any existing joysticks that may be plugged in that may interfere with input
		else if (strcasecmp(s, "-nojoystick")==0)
		{
			set_use_joystick(false);
		}
		// if they are paranoid and don't want data sent to the server
		// (don't be paranoid, read the source code, nothing bad is going on)
		else if (strcasecmp(s, "-noserversend")==0)
		{
			net_no_server_send();
		}
		else if (strcasecmp(s, "-nosound")==0)
		{
			set_sound_enabled_status(false);
			printline("Disabling sound...");
		}
		else if (strcasecmp(s, "-sound_buffer")==0)
		{
			get_next_word(s, sizeof(s));
			Uint16 sbsize = (Uint16) atoi(s);
			set_soundbuf_size(sbsize);
			sprintf(s, "Setting sound buffer size to %d", sbsize);
			printline(s);
		}
		else if (strcasecmp(s, "-nocrc")==0)
		{
			g_game->disable_crc();
			printline("Disabling ROM CRC check...");
		}

		else if (strcasecmp(s, "-scoreboard")==0)
		{
			set_scoreboard(1);
			printline("Enabling external scoreboard...");
		}
		else if (strcasecmp(s, "-scoreport")==0)
		{
			get_next_word(s, sizeof(s));
			i = atoi(s);
			set_scoreboard_port((unsigned char) i);
			sprintf(s, "Setting scoreboard port to %d", i);
			printline(s);
		}
		else if (strcasecmp(s, "-port")==0)
		{
			get_next_word(s, sizeof(s));
			i = atoi(s);
			set_serial_port((unsigned char) i);
			sprintf(s, "Setting serial port to %d", i);
			printline(s);
		}
		else if (strcasecmp(s, "-baud")==0)
		{
			get_next_word(s, sizeof(s));
			i = atoi(s);
			set_baud_rate(i);
			sprintf(s, "Setting baud rate to %d", i);
			printline(s);
		}

		// used to modify the dip switch settings of the game in question
		else if (strcasecmp(s, "-bank")==0)
		{
			get_next_word(s, sizeof(s));
			i = s[0] - '0';	// bank to modify.  We convert to a number this way to catch more errors
			
			get_next_word(s, sizeof(s));
			unsigned char value = (unsigned char) (strtol(s, NULL, 2));	// value to be set is in base 2 (binary)

			result = g_game->set_bank((unsigned char) i, (unsigned char) value);
		}

		else if (strcasecmp(s, "-latency")==0)
		{
			get_next_word(s, sizeof(s));
			i = atoi(s);
			g_ldp->set_search_latency(i);
			sprintf(s, "Setting Search Latency to %d milliseconds", i);
			printline(s);
		}
		else if (strcasecmp(s, "-cheat")==0)
		{
			g_game->enable_cheat();	// enable any cheat we have available :)
		}

		// enable keyboard LEDs for use with games such as Space Ace
		else if (strcasecmp(s, "-enable_leds")==0)
		{
			enable_leds();
		}

		// run daphne in fullscreen mode
		else if (strcasecmp(s, "-fullscreen")==0)
		{
			set_fullscreen(true);
		}
		
		// disable YUV hardware acceleration
		else if (strcasecmp(s, "-nohwaccel")==0)
		{
// MPO : I commented out the setenv command because unix should be able to also support putenv
//#ifdef WIN32
			putenv("SDL_VIDEO_YUV_HWACCEL=0");
//#else
//			setenv("SDL_VIDEO_YUV_HWACCEL", "0", 1);
//#endif
		}
		else if (strcasecmp(s, "-pal_dl")==0)
		{
			set_frame_modifier(MOD_PAL_DL);
			printline("Setting DAPHNE up for the PAL Dragon's Lair disc");
			cpu_change_irq(0, 0, LAIR_IRQ_PERIOD * (23.976/25.0)); // change IRQ 0 of CPU 0
			// DL PAL runs at a different speed so we need to overclock the IRQ slightly
		}
		else if (strcasecmp(s, "-pal_sa")==0)
		{
			set_frame_modifier(MOD_PAL_SA);
			printline("Setting DAPHNE up for the PAL Space Ace disc");
		}
		else if (strcasecmp(s, "-pal_dl_sc") == 0)
		{
			set_frame_modifier(MOD_PAL_DL_SC);
			printline("Setting DAPHNE up for the PAL Dragon's Lair Software Corner disc");
			cpu_change_irq(0, 0, LAIR_IRQ_PERIOD * (23.976/25.0));
			// DL Amiga runs at a different speed so we need to overclock the IRQ slightly
		}
		else if (strcasecmp(s, "-pal_sa_sc") == 0)
		{
			set_frame_modifier(MOD_PAL_SA_SC);
			printline("Setting DAPHNE up for the PAL Space Ace Software Corner disc");
		}
		else if (strcasecmp(s, "-spaceace91")==0)
		{
			set_frame_modifier(MOD_SA91);
			printline("Setting DAPHNE to play a Space Ace '91 disc");
		}
		else if (strcasecmp(s, "-preset")==0)
		{
			get_next_word(s, sizeof(s));
			i = atoi(s);
			g_game->set_preset(i);
		}
		else if (strcasecmp(s, "-version")==0)
		{
			get_next_word(s, sizeof(s));
			i = atoi(s);
			g_game->set_version(i);
		}
		else if (strcasecmp(s, "-x")==0)
		{
			get_next_word(s, sizeof(s));
			i = atoi(s);
			set_video_width((Uint16)i);
			sprintf(s, "Setting screen width to %d", i);
			printline(s);
		}
		else if (strcasecmp(s, "-y")==0)
		{
			get_next_word(s, sizeof(s));
			i = atoi(s);
			set_video_height((Uint16)i);
			sprintf(s, "Setting screen height to %d", i);
			printline(s);
		}
		else if (strcasecmp(s, "-trace")==0)
		{
#ifdef CPU_DEBUG
			printline("CPU tracing enabled");
			set_cpu_trace(1);
#else
			printline("DAPHNE needs to be compiled in debug mode for this to work");
			result = 0;
#endif
		}
		else if (strcasecmp(s, "-sbtext")==0)
		{
			get_next_word(s, sizeof(s));
			set_scoreboard_text(s);
			sprintf(s, "Sending text '%s' to scoreboard", s);
			printline(s);
		}

		// added by JFA for -idleexit
		else if (strcasecmp(s, "-idleexit")==0)
		{
			get_next_word(s, sizeof(s));
			i = atoi(s);
			set_idleexit(i*1000);
			sprintf(s, "Setting idleexit to %d ", i);
			printline(s);
		}
		// end edit

		// added by JFA for -startsilent
		else if (strcasecmp(s, "-startsilent")==0)
		{
			set_startsilent(1);
			printline("Starting silent...");
		}
		// end edit

		// if they are requesting to stop the laserdisc when the program exits
		else if (strcasecmp(s, "-stoponquit")==0)
		{
			g_ldp->set_stop_on_quit(true);
		}
        // this switch only supported by the ldp-vldp player class.
        else if (strcasecmp(s, "-useoverlaysb")==0)
        {
    		ldp_vldp *the_ldp = dynamic_cast<ldp_vldp *>(g_ldp);

            if (NULL == the_ldp)
                printline("-useoverlaysb: Only supported with LDP vldp -- ignoring switch...");
            else
            {
                lair *game_lair_or_sa = dynamic_cast<lair *>(g_game);
                thayers *game_thayers = dynamic_cast<thayers *>(g_game);

                if (NULL == game_lair_or_sa && NULL == game_thayers)
                    printline("-useoverlaysb: Not supported for this game -- ingnoring switch...");
                else
                {
                    if (game_lair_or_sa)
                        game_lair_or_sa->init_overlay_scoreboard();
                    else
                    // Must be Thayer's Quest
                        game_thayers->init_overlay_scoreboard();
                }
            }
        }
		else
		{
			printline("Unknown command line parameter:");
			printline(s);
			result = false;
		}
	  } // end for
	} // end if we know our game type
	
	// if game or ldp was unknown
	else
	{
		result = false;
	}

	return(result);

}


// copies a single word from the command line into result (a word is any group of characters separate on both sides by whitespace)
// if there are no more words to be copied, result will contain a null string
void get_next_word(char *result, int result_size)
{
	// make sure we still have command line left to parse
	if (g_arg_index < g_argc)
	{
		strncpy(result, g_argv[g_arg_index], result_size);
		result[result_size-1] = 0;	// terminate end of string just in case we hit the limit
		g_arg_index++;
	}

	// if we have no command line left to parse ...
	else
	{
		result[0] = 0;	// return a null string
	}
}
