/*
 * conout.cpp
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// CONOUT
// Provides console functionality in a window (replacement for printf, cout, etc)

#include <stdio.h>
#include <string.h>
#include <SDL.h>
#include "conout.h"
#include "../daphne.h"
#include "../video/video.h"
#include "../video/SDL_Console.h"
#include "../timer/timer.h"
#include "input.h"

char G_curline[CHARS_PER_LINE] = { 0 };	// current line of output

// Prints a "c string" to the screen
// Returns 1 on success, 0 on failure
unsigned char outstr(const char *s)
{
	unsigned char result = 1;

	if (get_console_initialized())
	{
		if (strlen(s) + strlen(G_curline) < CHARS_PER_LINE)
		{
			strcat(G_curline,s);
		}
	}
	addlog(s);	// add to our logfile

	return(result);
}

// Prints a single character to the screen
// Returns 1 on success, 0 on failure
unsigned char outchr(const char ch)
{

	unsigned char result = 1;
	char s[2] = { 0 };

	s[0] = ch;
	s[1] = 0;

	outstr(s);

	return(result);

}


// prints a null-terminated string to the screen
// and adds a line feed
// 1 = success
unsigned char printline(const char *s)
{

	unsigned char result = 1;

	outstr(s);
	newline();

	return(result);

}

// moves to the next line without printing anything
void newline()
{

	// Carriage return followed by line feed
	// So we can read the blasted thing in notepad
	static char newline_str[3] = { 13, 10, 0 };

	if (get_console_initialized())
	{
		ConOut(G_curline);
		G_curline[0] = 0;	// erase line
	}

	addlog(newline_str);

}

// prints the current line even if we haven't hit a linefeed yet
void con_flush()
{

	ConOutstr(G_curline);

}


// flood-safe printline
// it only prints every second and anything else is thrown out
void noflood_printline(char *s)
{
	static unsigned int old_timer = 0;
	
	if (elapsed_ms_time(old_timer) > 1000)
	{
		printline(s);
		old_timer = refresh_ms_time();
	}
}


// This is a safe version of ITOA in that prevents buffer overflow, but if your buffer size is too small,
// you will get an incorrect result.
void safe_itoa(int num, char *a, int sizeof_a)
{
	int i = 0,j = 0, n = 0;
	char c = 0;

	// safety check : size needs to be at least 3 to accomodate one number, a minus sign and a null character
	if (sizeof_a > 2)
	{
		n = num;
		
		// if number is negative, reverse its sign to make the math consistent with positive numbers
		if (num < 0)
		{
			n = -n;
		}
		
		do
		{
			a[i++]=n % 10 + '0';
			n = n / 10;
		} while ((n > 0) && (i < (sizeof_a - 2)));
		// sizeof_a - 2 to leave room for a null terminating character and a possible minus sign
		
		// if number was negative, then add a minus sign
		if (num < 0) a[i++]='-';
		
		a[i] = 0;	// terminate string
		
		// reverse string (we had to go in reverse order to do the calculation)
		for (i=0,j=strlen(a)-1; i<j; i++,j--)
		{
			c=a[i];
			a[i]=a[j];
			a[j]=c;
		}
	}
	
}

// adds a string to a log file (creates the logfile if it does not exist)
void addlog(const char *s)
{

	FILE *F = 0;
	unsigned int written = 0;	// # of unsigned chars written

	F = fopen(LOGNAME, "ab");	// open for appending
	if (F)
	{
		written = fprintf(F, "%s", s);
		if (written != strlen(s))
		{
#ifdef UNIX
			printf("Cannot write to logfile, do you have write permissions?\n");
#endif
		}
		fclose(F);	// close each time in case we get in an endless loop ...
	}
	// else directory is read-only so we will just ignore for now

	/*
	else
	{
		printline("Could not open log file!");
	}
	*/
}
