/*
 * error.cpp
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// error.cpp
// daphne error handling

#ifdef WIN32
#include <windows.h>
#endif

#include <SDL.h>
#include <string.h>
#include "../daphne.h"
#include "conout.h"
#include "conin.h"
#include "input.h"
#include "../game/game.h"
#include "../sound/sound.h"
#include "../video/video.h"
#include "../video/SDL_DrawText.h"

const char *instr = "Please read the daphne_log.txt file for more information";

// notifies the user of an error that has occurred
void printerror(const char *s)
{
	SDL_Rect region = { 0, 180, get_video_width(), (Uint16)(get_video_height()/2) };

	// if video has been initialized
	// FIXME: if bmp hasn't been loaded this might blow up
	if (get_console_initialized())
	{

		draw_othergfx_unsafe(B_DAPHNE_SAVEME, 0, 180);
		SDLDrawText(s, get_screen(), FONT_SMALL, (400-((strlen(s)/2)*6)), 240-13);
		SDLDrawText(instr, get_screen(), FONT_SMALL, (400-((strlen(instr)/2)*6)), 240);
		SDL_Flip(get_screen());

		// play a 'save me' sound if we've got sound
		if (get_sound_initialized())
		{
			sound_play_saveme();
		}
		printline(s);
		con_getkey();	// wait for keypress

		SDL_FillRect(get_screen(), &region, 0);	// erase message
		SDL_Flip(get_screen());

		// MATT : we can't call video_repaint because the video might not have been initialized
		// yet.  We can either comment this out, or add safety checks in video_repaint (which
		// is kind of a hassle right now since we have a bunch of redundant copies everywhere)

	} // end if video has been initialized

	// if video has not been initialized, print an error any way that we can
	else
	{
#ifdef WIN32
	MessageBox(NULL, s, "DAPHNE encountered an error", MB_OK | MB_ICONERROR);
#else
		printf("%s\n",s);
#endif
	}
}

// notifies user that the game does not work correctly and gives a reason
// this should be called after video has successfully been initialized
void printnowookin(const char *s)
{
//	SDL_Rect region = { 0, 180, get_video_width(), (Uint16)(get_video_height()/2) };

	draw_othergfx_unsafe(B_GAMENOWOOK, 0, 180);
	SDLDrawText(s, get_screen(), FONT_SMALL, (320-((strlen(s)/2)*6)), 240);
	SDL_Flip(get_screen());
	con_getkey();	// wait for keypress
	SDL_FillRect(get_screen(), NULL, 0);	// erase message
	SDL_Flip(get_screen());

	// repaint the disrupted overlay (ie Dragon's Lair Scoreboard when using a real LDP)
	g_game->video_force_blit();
}

// prints a notice to the screen
void printnotice(const char *s)
{
	SDL_Rect region = { 0, 180, get_video_width(), (Uint16)(get_video_height()/2) };
	char ch = 0;

	SDLDrawText(s, get_screen(), FONT_SMALL, (320-((strlen(s)/2)*6)), 240);
	SDL_Flip(get_screen());
	ch = con_getkey();	// wait for keypress
	SDL_FillRect(get_screen(), &region, 0);	// erase message
	SDL_Flip(get_screen());
	
	// if they pressed escape, quit
	if (ch == 27)
	{
		set_quitflag();
	}

	// repaint the disrupted overlay
	g_game->video_force_blit();
}
