/*
 * fileparse.cpp
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// fileparse.cpp
// by Matt Ownby

// some basic file-parsing functions which should be safe enough to prevent daphne from ever
// segfaulting, even if the file is totally corrupt

#include "fileparse.h"
#include <string.h>

// copies into buf everything from the current position in the file until the end of a
//  line and positions the file stream at the beginning of the next line.  Contents are also
//  null-terminated.
// Returns the # of bytes copied into the buf.
// buf_size is the total size of the buf to prevent buffer overflow
int read_line(FILE *F, char *buf, int buf_size)
{
	int index = 0;
	int ch = 0;

	// while we haven't overflowed the buffer
	// (we do -1 to leave space for null character at the end)
	while (index < (buf_size - 1))
	{
		ch = fgetc(F);

		if (ch == EOF) break;	// if we hit the end of file, then break

		// if we have hit the end of the line
		if ((ch == 10) || (ch == 13))
		{
			// keep reading until we get passed the end of line chars
			while (((ch == 10) || (ch == 13)) && (ch != EOF))
			{
				ch = getc(F);
			}

			// if we still have more of the file to read in, then move back one character
			// so that we are at the beginning of the next line
			if (ch != EOF)
			{
				fseek(F, -1, SEEK_CUR);
			}

			break;
		}

		// if the character is part of the current line
		else
		{
			buf[index++] = (unsigned char) ch;
		}
	}
	buf[index] = 0;

	return (index + 1);	// + 1 to indicate # of bytes actually copied
}

// takes a pathname (relative or absolute) + a filename and removes the filename itself
// therefore it returns just the path (stored in 'path')
// !!! We assume that the 'path' buffer's size is >= 'file_with_path' size
// returns false if there is an error (such as the file having no path)
bool get_path_of_file(char *file_with_path, char *path)
{
	bool success = false;
	int index = strlen(file_with_path) - 1;	// start on last character

	// make sure the file_with_path is at least 2 characters long
	// because otherwise our proceeding calculations could cause a segfault
	if (index > 0)
	{
		// locate the preceeding / or \ character
		while ((index >= 0) && (file_with_path[index] != '/') && (file_with_path[index] != '\\'))
		{
			index--;
		}

		// if we found a leading / or \ character
		if (index >= 0)
		{
			strcpy(path, file_with_path);
			path[index+1] = 0;	// chop off filename from string (we already know
			success = true;
		}
	}

	return success;
}
