/*
 * input.cpp
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// Handles SDL input functions (low-level keyboard/joystick input)

#include <time.h>
#include "input.h"
#include "conout.h"
#include "string.h"
#include "stdlib.h"
#include "../video/video.h"
#include "../video/SDL_Console.h"
#include "../daphne.h"
#include "../timer/timer.h"
#include "../game/game.h"
#include "../game/thayers.h"
#include "../ldp-out/ldp.h"

// Win32 doesn't use strcasecmp, it uses stricmp (lame)
#ifdef WIN32
#define strcasecmp stricmp
#endif

const int JOY_AXIS_MID = (int) (32768 * (0.75));		// how far they have to move the joystick before it 'grabs'

SDL_Joystick *G_joystick = NULL;	// pointer to joystick object
bool g_use_joystick = true;	// use a joystick by default
bool g_consoledown = false;			// whether the console is down or not
bool g_alt_pressed = false;	// whether the ALT key is presssed (for ALT-Enter combo)
unsigned int idle_timer; // added by JFA for -idleexit

// the ASCII key words that the parser looks at for the key values
// NOTE : these are in a specific order, corresponding to the enum in daphne.h
const char *g_key_names[] = 
{
	"KEY_UP", "KEY_LEFT", "KEY_DOWN", "KEY_RIGHT", "KEY_START1", "KEY_START2", "KEY_BUTTON1",
	"KEY_BUTTON2", "KEY_BUTTON3", "KEY_COIN1", "KEY_COIN2", "KEY_SKILL1", "KEY_SKILL2", 
	"KEY_SKILL3", "KEY_SERVICE", "KEY_TEST", "KEY_RESET", "KEY_SCREENSHOT", "KEY_QUIT"
};

// default key assignments, in case .ini file is missing
// Notice each switch can have two keys assigned to it
// NOTE : These are in a specific order, corresponding to the enum in daphne.h
int g_key_defs[SWITCH_COUNT][2] =
{
	{ SDLK_UP, SDLK_KP8 },	// up
	{ SDLK_LEFT, SDLK_KP4 },	// left
	{ SDLK_DOWN, SDLK_KP2 },	// down
	{ SDLK_RIGHT,  SDLK_KP6 },	// right
	{ SDLK_1,	0 }, // 1 player start
	{ SDLK_2,	0 }, // 2 player start
	{ SDLK_SPACE, SDLK_LCTRL }, // action button 1
	{ SDLK_LALT,	0 }, // action button 2
	{ SDLK_LSHIFT,	0 }, // action button 3
	{ SDLK_5, SDLK_c }, // coin chute left
	{ SDLK_6, 0 }, // coin chute right
	{ SDLK_KP_DIVIDE, 0 }, // skill easy
	{ SDLK_KP_MULTIPLY, 0 },	// skill medium
	{ SDLK_KP_MINUS, 0 },	// skill hard
	{ SDLK_9, 0 }, // service coin
	{ SDLK_F2, 0 },	// test mode
	{ SDLK_F3, 0 },	// reset cpu
	{ SDLK_F12, 0 },	// take screenshot
	{ SDLK_ESCAPE, SDLK_q }	// Quit DAPHNE
};

////////////

// added by Russ
// global button mapping array. just hardcoded room for 10 buttons max
int joystick_buttons_map[10] = {
	SWITCH_BUTTON1,	// button 1
	SWITCH_BUTTON2,	// button 2
	SWITCH_BUTTON3,	// button 3
	SWITCH_BUTTON1,	// button 4
	SWITCH_COIN1,		// button 5
	SWITCH_START1,		// button 6
	SWITCH_BUTTON1,	// button 7
	SWITCH_BUTTON1,	// button 8
	SWITCH_BUTTON1,	// button 9
	SWITCH_BUTTON1,	// button 10
};

////////////

void CFG_Keys()
{
	FILE *F = NULL;
	bool reached_end = false;
	bool reached_endl = false;
	int strIndex = 0;
	int keyCount = 0;
	bool foundVal = false;
	bool ignoreSpace = false;
	char strTemp[25] = {0};
	char thisChar = 0;

	F = fopen("dapinput.ini", "rt");

	// if we opened the file successfully
	if (F != NULL)
	{
		printline("Reading key remapping file ... (if the file is corrupt, daphne will crash, just so you know hehe)");
		char initype[25] = { 0 };	// which switch we are assigning a new value to
//		char iniscrap[5] = { 0 };	// gobbles up '=' character
		int ini1value = 0, ini2value = 0, ini3value = 0;	// SDLKey values to assign switches to
		char inivalue1str[25] = { 0 };	// primary switch assignment
		char inivalue2str[25] = { 0 };	// secondary switch assignment
		char inivalue3str[25] = { 0 };	// button switch assignment
		//The following do/while loop searches for the KEYBOARD
		//section of thi INI file. This allows for future INI file
		//use for other functions
		while (strcasecmp(initype,"[KEYBOARD]") != 0)
		{
			fscanf(F,"%s",initype);
		}

		// read lines until we hit END
		while (!reached_end)
		{
			strIndex = 0;
			keyCount = 0;
			reached_endl = false;
			ignoreSpace = false;

			strcpy(strTemp, "");
			strcpy(initype, "");
			strcpy(inivalue1str, "");
			strcpy(inivalue2str, "");
			strcpy(inivalue3str, "");

			// added by Russ
			// step through each line char by char and parse out vals by hand
			// kinda ugly, but effective and backwards compatible to old dapinput.ini files

			while (!reached_endl) {
				thisChar = (char) fgetc(F);
				foundVal = false;

				if (thisChar == ' ' || thisChar == '\t') {
					if (!ignoreSpace) {
						strTemp[strIndex++] = '\0';
						foundVal = true;
					}
					ignoreSpace = true;
				} else if (thisChar == '\n') {
					strTemp[strIndex++] = '\0';
					foundVal = true;
					reached_endl = true;
					ignoreSpace = true;
				} else {
					strTemp[strIndex++] = thisChar;
					ignoreSpace = false;
				} 

				if (foundVal) {
					if (keyCount == 0) {
						strcpy(initype, strTemp);
					} else if (keyCount == 2) {
						strcpy(inivalue1str, strTemp);
					} else if (keyCount == 3) {
						strcpy(inivalue2str, strTemp);
					} else if (keyCount == 4) {
						strcpy(inivalue3str, strTemp);
					}

					strcpy(strTemp, "");
					strIndex = 0;
					keyCount++;
				}
			}

			if (strlen(inivalue1str) > 0) ini1value = atoi(inivalue1str);
			if (strlen(inivalue2str) > 0) ini2value = atoi(inivalue2str);
			if (strlen(inivalue3str) > 0) ini3value = atoi(inivalue3str);

			// if we're not at the end of our switch definitions yet
			if (strcasecmp(initype,"END") != 0) {
				for (int i = 0; i < SWITCH_COUNT; i++) {
					// if we can match up a key name (see list above) then assign its two key values to the current key def
					if (strcasecmp(initype, g_key_names[i])==0) {
						g_key_defs[i][0] = ini1value;
						g_key_defs[i][1] = ini2value;

						// added by Russ.  Map the button here...
						if (ini3value > 0) {  // if zero then no mapping necessary, just use default, if any
							joystick_buttons_map[ini3value - 1] = i;
						}
					}
					// else it's foreign, so we do nothing
				}
			} else {
				// else we've hit the END so break out of the loop
				reached_end = true;
			}
		}
		fclose(F);
	} // end if we successfully opened the key init file
}


int SDL_input_init()
// initializes the keyboard (and joystick if one is present)
// returns 1 if successful, 0 on error
// NOTE: Video has to be initialized BEFORE this is or else it won't work
{

	int result = 0;

	if (SDL_InitSubSystem(SDL_INIT_JOYSTICK) >= 0)
	{
		// if joystick usage is enabled
		if (g_use_joystick)
		{
			// if there is at least 1 joystick and we are authorized to use the joystick for input
			if (SDL_NumJoysticks() > 0)
			{
				G_joystick = SDL_JoystickOpen(0);	// FIXME: right now we automatically choose the first joystick
				if (G_joystick != NULL)
				{
					printline("Joystick #0 was successfully opened");
				}
				else
				{
					printline("Error opening joystick!");
				}
			}
			else
			{
				printline("No joysticks detected");
			}
		}
		// notify user that their attempt to disable the joystick is successful
		else
		{
			printline("Joystick usage disabled");
		}
		
		CFG_Keys();	// NOTE : for some freak reason, this should not be done BEFORE the joystick is initialized, I don't know why!
		result = 1;
	}
	else
	{
		printline("Input initialization failed!");
	}

	idle_timer = refresh_ms_time(); // added by JFA for -idleexit

	return(result);

}

// does any shutting down necessary
// 1 = success, 0 = failure
int SDL_input_shutdown(void)
{
	SDL_QuitSubSystem(SDL_INIT_JOYSTICK);
	return(1);
}

// checks to see if there is incoming input, and acts on it
void SDL_check_input()
{

	SDL_Event event;

	while ((SDL_PollEvent (&event)) && (!get_quitflag()))
	{
		// if they press the tilda key to bring down the console
		// this is somewhat of a hacked if statement but I can't see
		// a better way based on the SDL_Console API ...
		if ((event.type == SDL_KEYDOWN) && (event.key.keysym.sym == SDLK_BACKQUOTE))
		{
			// we must not bring down the console if blitting is not allowed
			if (g_ldp->is_blitting_allowed())
			{
				toggle_console();
			}
		}
		// if we've got the console down, process events
		else if (g_consoledown)
		{
			ConsoleEvents(&event);
		}
		// else handle events normally
		else
		{
			process_event(&event);
		}
	}
	check_console_refresh();

	// added by JFA for -idleexit
	if (get_idleexit() > 0 && elapsed_ms_time(idle_timer) > get_idleexit()) set_quitflag();

}

void toggle_console()
{
	// if console is down, get rid of it
	if (g_consoledown)
	{
		g_consoledown = false;
		SDL_EnableUNICODE(0);
		display_repaint();
	}
	// if console is not down, display it
	else
	{
		g_consoledown = true;
		SDL_EnableUNICODE(1);
	}
}

// processes incoming input
void process_event(SDL_Event *event)
{
	static int i;

	switch (event->type)
	{
		case SDL_KEYDOWN:
			reset_idle(); // added by JFA for -idleexit
			if (g_game->get_game_type() != GAME_THAYERS)
			{
				process_keydown(event->key.keysym.sym);
			}
			// We use a special case for Thayers Quest
			else
			{
				thayers *l_thayers = dynamic_cast<thayers *>(g_game);
				// cast game class to a thayers class so we can call a thayers-specific function

				// make sure cast succeeded
				if (l_thayers)
				{
					l_thayers->process_keydown(event->key.keysym.sym);
				}
				// else cast failed, and we would crash if we tried to call process_keydown
				// cast would fail if g_game is not a thayers class
			}
			break;
		case SDL_KEYUP:
			reset_idle(); // added by JFA for -idleexit
			if (g_game->get_game_type() != GAME_THAYERS)
			{
				process_keyup(event->key.keysym.sym);
			}
			// We use a special case for Thayers Quest
			else
			{
				thayers *l_thayers = dynamic_cast<thayers *>(g_game);
				// cast game class to thayers class so we can call a thayers-specific function

				// make sure cast succeeded
				if (l_thayers)
				{
					l_thayers->process_keyup(event->key.keysym.sym);
				}
				// else cast failed and we would crash if we tried to call process_keyup
				// cast would fail if g_game is not a thayers class
			}
			break;
		case SDL_JOYAXISMOTION:
			//reset_idle(); // added by JFA for -idleexit
			// reset_idle removed here because the analog controls were registering
			// even when the joystick was not in use.  Joystick buttons still reset the idle.
			process_joystick_motion(event);
			break;
		case SDL_JOYBUTTONDOWN:
			reset_idle(); // added by JFA for -idleexit

			// added by Russ
			// loop through 10 buttons and look for a press
			for (i = 0; i < 10; i++) {
				if (event->jbutton.button == i) {
					input_enable((Uint8) joystick_buttons_map[i]);
					break;
				}
			}

			break;
		case SDL_JOYBUTTONUP:
			reset_idle(); // added by JFA for -idleexit

			// added by Russ
			for (i = 0; i < 10; i++) {
				if (event->jbutton.button == i) {
					input_disable((Uint8) joystick_buttons_map[i]);
					break;
				}
			}

			break;
		case SDL_QUIT:
			// if they are trying to close the window
			set_quitflag();
			break;
		default:
			break;
	}

	// added by JFA for -idleexit
	if (get_idleexit() > 0 && elapsed_ms_time(idle_timer) > get_idleexit()) set_quitflag();

}

// if a key is pressed, we go here
void process_keydown(SDLKey key)
{	
	// go through each key def (defined in enum in daphne.h) and check to see if the key entered matches
	// If we have a match, the switch to be used is the value of the index "move"
	for (Uint8 move = 0; move < SWITCH_COUNT; move++)
	{
		if ((key == g_key_defs[move][0]) || (key == g_key_defs[move][1]))
		{
			input_enable(move);
		}
	}
	
	// check for ALT-ENTER here
	if ((key == SDLK_LALT) || (key == SDLK_RALT))
	{
		g_alt_pressed = true;
	}
	else if ((key == SDLK_RETURN) && (g_alt_pressed))
	{
		vid_toggle_fullscreen();
	}
	// end ALT-ENTER check
}

// if a key is released, we go here
void process_keyup(SDLKey key)
{
	// go through each key def (defined in enum in daphne.h) and check to see if the key entered matches
	// If we have a match, the switch to be used is the value of the index "move"
	for (Uint8 move = 0; move < SWITCH_COUNT; move++)
	{
		if ((key == g_key_defs[move][0]) || (key == g_key_defs[move][1]))
		{
			input_disable(move);
		}
	}

	// if they are releasing an ALT key	
	if ((key == SDLK_LALT) || (key == SDLK_RALT))
	{
		g_alt_pressed = false;
	}
}

// processes movements of the joystick
void process_joystick_motion(SDL_Event *event)
{

	static int x_axis_in_use = 0;	// true if joystick is left or right
	static int y_axis_in_use = 0;	// true if joystick is up or down

	// if they are moving along the verticle axis
	if (event->jaxis.axis == 1)
	{
		// if they're moving up
		if (event->jaxis.value < -JOY_AXIS_MID)
		{
			input_enable(SWITCH_UP);
			y_axis_in_use = 1;
		}
		// if they're moving down
		else if (event->jaxis.value > JOY_AXIS_MID)
		{
			input_enable(SWITCH_DOWN);
			y_axis_in_use = 1;
		}

		// if they just barely stopped moving up or down
		else if (y_axis_in_use == 1)
		{
			input_disable(SWITCH_UP);
			input_disable(SWITCH_DOWN);
			y_axis_in_use = 0;
		}
	} // end verticle axis

	// horizontal axis
	else
	{
		// if they're moving right
		if (event->jaxis.value > JOY_AXIS_MID)
		{
			input_enable(SWITCH_RIGHT);
			x_axis_in_use = 1;
		}
		// if they're moving left
		else if (event->jaxis.value < -JOY_AXIS_MID)
		{
			input_enable(SWITCH_LEFT);
			x_axis_in_use = 1;
		}
		// if they just barely stopped moving right or left
		else if (x_axis_in_use == 1)
		{
			input_disable(SWITCH_RIGHT);
			input_disable(SWITCH_LEFT);
			x_axis_in_use = 0;
		}
	} // end horizontal axis
}

// functions to help us avoid 'extern' statements
bool get_consoledown()
{
	return (g_consoledown);
}

void set_consoledown (bool value)
{
	g_consoledown = value;
}

// draws console if it's down and if there's been enough of a delay
void check_console_refresh()
{

	static unsigned int console_refresh = 0;
	const unsigned int refresh_every = 125;	// refreshes console every (this many) ms

	if (g_consoledown)
	{
		if (elapsed_ms_time(console_refresh) > refresh_every)
		{
			DrawConsole();
			SDL_Flip(get_screen());
			console_refresh = refresh_ms_time();
		}
	}
}

// if user has pressed a key/moved the joystick/pressed a button
void input_enable(Uint8 move)
{
	// reset the game
	if (move == SWITCH_RESET)
	{
		g_game->reset();
	}
	else if (move == SWITCH_SCREENSHOT)
	{
		printline("Screenshot requested!");
		g_ldp->request_screenshot();
	}
	else if (move == SWITCH_QUIT)
	{
		set_quitflag();
	}
	else
	{
		g_game->input_enable(move);
	}
}

// if user has released a key/released a button/moved joystick back to center position
void input_disable(Uint8 move)
{
	// don't send reset or screenshots key-ups to the individual games because they will return warnings that will alarm users
	if ((move != SWITCH_RESET) && (move != SWITCH_SCREENSHOT) && (move != SWITCH_QUIT))
	{
		g_game->input_disable(move);
	}
	// else do nothing
}

// added by JFA for -idleexit
void reset_idle(void)
{
	static int soundon = 0;
	if (!soundon)
	{
		soundon = 1;
		reopen_sound();
	}
	idle_timer = refresh_ms_time();
}
// end edit

// primarily to disable joystick use if user wishes not to use one
void set_use_joystick(bool val)
{
	g_use_joystick = val;
}
