/*
 * sram.cpp
 *
 * Copyright (C) 2002 Warren Ondras
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// sram.cpp -- stores/loads saved static RAM to/from disk
// by Warren Ondras

#include <stdio.h>
#include <zlib.h>	// for compression

#ifdef WIN32
// for CreateDirectory
#include <windows.h>
#endif

#ifdef UNIX
// for mkdir
#include <sys/stat.h>
#include <sys/types.h>
#endif

#include "conout.h"

int sram_load(char * filename, unsigned char * mem, unsigned int size)
{
	char s[81];
	gzFile loadfile = NULL;
	int result = 0;

	sprintf(s, "ram/%s", filename);
	loadfile = gzopen(s, "rb");	// open compressed file

	// if loading the file succeeded
	if (loadfile)
	{
		// read compressed data
		if (gzread(loadfile, (voidp) mem, size) == (int) size)
		{
			sprintf(s, "Loaded %u bytes from %s", size, filename);
			printline(s);
			result = 1;
		}
		else
		{
			sprintf(s, "Error loading %d bytes from %s", size, filename);
			printline(s);
		}
		gzclose(loadfile);
	}

	// if loading the file failed
	else
	{
		sprintf(s, "NOTE : RAM file ram/%s was not found (it'll be created)", filename);
		printline(s);
	}
	
	return result;
}

int sram_save(char * filename, unsigned char * mem, unsigned int size)
{
	char s[81];
	gzFile savefile;
	int result = 0;

#ifdef WIN32
	CreateDirectory("ram", NULL);	// create directory if it does not already exist
#else
	unsigned int oldumask = umask(022);
	mkdir("ram", 0777);	// create directory if it does not already exist
	umask(oldumask);
#endif

	sprintf(s, "ram/%s", filename);
	savefile = gzopen(s, "wb");

	// if opening file was successful	
	if (savefile)
	{
		// set it to compress at maximum because it shouldn't take too long and
		// it'll save disk space
		gzsetparams(savefile, Z_BEST_COMPRESSION, Z_DEFAULT_STRATEGY);

		if (gzwrite(savefile, (voidp) mem, size) == (int) size)
		{
			sprintf(s, "Saved %d bytes to %s", size, filename);
			printline(s);
			result = 1;
		}
		else
		{
			sprintf(s, "Error saving %d bytes to %s", size, filename);
			printline(s);
		}
		gzclose(savefile);
	}

	else
	{
		sprintf(s, "Error saving RAM to file ram/%s", filename);
		printline(s);
	}
	
	return result;
}
