/*
 * ldp1000.cpp
 *
 * Copyright (C) 2001 Mark Broadhead
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// ldp1000.c
// part of the DAPHNE emulator
// written by Mark Broadhead
//
// This code emulates the Sony LDP-1000 laserdisc player which is used in
// Bega's Battle and Cobra Command (dedicated). The LDP-1450 in Dragon's Lair 2,
// ALG games, and Time Traveler also uses this same command set

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "../game/game.h"
#include "ldp1000.h"
#include "../daphne.h"
#include "../io/conout.h"
#include "../video/tms9128nl.h"
#include "../ldp-out/ldp.h"

#define LDP1000_STACKSIZE 6 // the largest response is 5 bytes
#define FRAME_SIZE 5

unsigned char ldp1000_output_stack[LDP1000_STACKSIZE];
int ldp1000_output_stack_pointer = 0;
unsigned int ldp1000_curframe;
unsigned int ldp1000_repeat_frame;
unsigned int ldp1000_repeat_start_frame;
bool ldp1000_repeat = false;
int ldp1000_times_to_repeat = 0;

char ldp1000_frame[FRAME_SIZE+1] = {0}; // holds the digits sent to the LDP1000
int ldp1000_frame_index = 0; 

// default status
static unsigned char enter_status = 0x00;
///////////////////////////////////////////

// retrieves the status from our virtual LDP1000
unsigned char read_ldp1000()
{
	unsigned char result = 0x00;

	if (ldp1000_output_stack_pointer < 1)
	{
		printline("ERROR: LDP1000 stack read when empty");
	}
	else 
	{
		ldp1000_output_stack_pointer--;
		result = ldp1000_output_stack[ldp1000_output_stack_pointer];
	}
	
	return(result);
}

// pushes a value on the ldp1000 stack, returns 1 if successful or 0 if stack is full
int ldp1000_stack_push(unsigned char value)
{
	int result = 0;

	// if we still have room to push
	if (ldp1000_output_stack_pointer < LDP1000_STACKSIZE-1)
	{
		ldp1000_output_stack[ldp1000_output_stack_pointer++] = value;
		result = 1;
	}
	else
	{
		printline("ERROR: LDP1000 stack overflow (increase its size)");
	}

	return(result);
}

// sends a byte to our virtual LDP1000
void write_ldp1000 (unsigned char value)
{
	char s[81] = { 0 };
	switch (value)
	{
	case 0x30: // '0'
	case 0x31: // '1'
	case 0x32: // '2'
	case 0x33: // '3'
	case 0x34: // '4'
	case 0x35: // '5'
	case 0x36: // '6'
	case 0x37: // '7'
	case 0x38: // '8'
	case 0x39: // '9'
		ldp1000_add_digit(value);
		ldp1000_stack_push(0x0a); // ack
		break;
	case 0x3a: // Play
		g_ldp->pre_play();
		ldp1000_stack_push(0x0a); // ack
		break;
	case 0x40: // Enter
		ldp1000_enter();
		break;
	case 0x43: // Search 
		enter_status = LDP1000_SEARCH;
		ldp1000_stack_push(0x0a); // ack
		break;
	case 0x44: // Repeat
		enter_status |= LDP1000_REPEAT;
		ldp1000_stack_push(0x0a); // ack
		break;
		break;
	case 0x56: // C.L. (reset)
		printline("C.L. (reset) recieved, ignored\n");
		ldp1000_stack_push(0x0a); // ack
		break;
	case 0x60: // Addr Inq (get current frame number)
		ldp1000_curframe = g_ldp->get_current_frame();
//		sprintf(s,"LDP1000: requesting frame %u",ldp1000_curframe);
//		printline(s);

		char f[6];
		sprintf(f, "%u", ldp1000_curframe);
		ldp1000_stack_push(f[4]); // M5
		ldp1000_stack_push(f[3]); // M4
		ldp1000_stack_push(f[2]); // M3
		ldp1000_stack_push(f[1]); // M2
		ldp1000_stack_push(f[0]); // M1
		break;
	default:
		sprintf(s,"Unsupported LDP1000 Command Received: %x", value);
		printline(s);
		ldp1000_stack_push(0x0a); // ack
		break;
	}
}

bool ldp1000_result_ready(void)
{
	bool result = false;

	if (ldp1000_repeat)
	{
		if (g_ldp->get_current_frame() >= ldp1000_repeat_frame)
		{
			// we finished one repeat
			ldp1000_times_to_repeat--;
			
			// start over if we still need to repeat
			if (ldp1000_times_to_repeat)
			{
				char f[6] = {0};
				sprintf(f, "%i", ldp1000_repeat_start_frame);
				g_ldp->pre_search(f);
				g_ldp->pre_play();
			}
			// or if we are done, pause
			else
			{
				g_ldp->pre_pause();
				ldp1000_repeat = false;
			}

			// we got to the desired frame
			ldp1000_stack_push(0x01); // completion
		}
	}
	
	if (ldp1000_output_stack_pointer)
	{
		result = true;
	}
	return result;
}

void ldp1000_enter(void)
{
	char s[81] = {0};
	
	if (enter_status == LDP1000_SEARCH)
	{
		ldp1000_repeat = false;
		
		ldp1000_frame[ldp1000_frame_index] = 0;
		
		// TODO: check for search failure
		g_ldp->pre_search(ldp1000_frame);

		ldp1000_frame_index = 0; // reset frame index

		ldp1000_stack_push(0x01); // completion
		ldp1000_stack_push(0x0a); // ack
		enter_status = 0x00;
	}

	else if (enter_status == LDP1000_REPEAT_NUMBER)
	{
		if (ldp1000_frame_index == 0)
		{
			ldp1000_times_to_repeat = 1;
		}
		// TODO: check for number of repeats other than 1
		sprintf(s, "LDP1000: Repeat %i times\n", ldp1000_times_to_repeat); 
		printline(s);

		g_ldp->pre_play();

		ldp1000_frame_index = 0; // reset frame index
		ldp1000_repeat = true;

		ldp1000_stack_push(0x0a); // ack
		enter_status = 0x00;
	}
	
	else if (enter_status == LDP1000_REPEAT)
	{
		ldp1000_frame[ldp1000_frame_index] = 0;
		
		sprintf(s, "LDP1000: Repeat at frame %i\n", atoi(ldp1000_frame)); 
		printline(s);

		ldp1000_repeat_frame = atoi(ldp1000_frame);
		ldp1000_repeat_start_frame = g_ldp->get_current_frame();

		ldp1000_frame_index = 0; // reset frame index
		
		ldp1000_stack_push(0x0a); // ack

		enter_status = LDP1000_REPEAT_NUMBER;
	}
}

// adds a digit to  the frame array that we will be seeking to
// digit should be in ASCII format
void ldp1000_add_digit(char digit)
{
	if (ldp1000_frame_index < FRAME_SIZE)
	{
		ldp1000_frame[ldp1000_frame_index] = digit;
		ldp1000_frame_index++;
	}
	else
	{
		char s[81] = { 0 };

		sprintf(s, "Too many digits received for frame! (over %d)", FRAME_SIZE);
		printline(s);
	}
}
