/*
 * ldv1000.cpp
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// LD-V1000.CPP
// part of the DAPHNE emulator
// started by Matt Ownby
// contributions made by Mark Broadhead

// This code emulates the Pioneer LD-V1000 laserdisc player which is used in these games:
// Dragon's Lair US
// Space Ace US
// Super Don Quixote
// Thayer's Quest
// Badlands
// Esh's Aurunmilla
// Interstellar
// Astron Belt (LD-V1000 roms)
// Starblazer

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "../game/game.h"
#include "ldv1000.h"
#include "../daphne.h"
#include "../game/game.h"
#include "../io/conout.h"
#include "../video/tms9128nl.h"
#include "../ldp-out/ldp.h"

#define LDV1000_STACKSIZE 10
unsigned char g_ldv1000_output_stack[LDV1000_STACKSIZE];
int g_ldv1000_output_stack_pointer = 0;
Uint16 g_ldv1000_autostop_frame = 0;	// which frame we need to stop on (if any)

bool g_instant_seeking = false;	// whether we can make the LD-V1000 complete searches instantly
								// Some games permit instantaneous seeking (Dragon's Lair) and
								// therefore should use it to improve performance.
								// Other games expect a little search lag (Esh's, Astron Belt)
								// and therefore should not use fast seeking.
								// Slow seeking is conservative and therefore, the default.

bool audio1 = true; // default audio status is on
bool audio2 = true;

char ldv1000_frame[FRAME_SIZE+1] = {0}; // holds the digits sent to the LD-V1000
int ldv1000_frame_index = 0; 

unsigned char g_ldv1000_output = 0xFC;	// LD-V1000 is PARK'd and READY

///////////////////////////////////////////

// retrieves the status from our virtual LD-V1000
unsigned char read_ldv1000()
{

	unsigned char result = 0;

	// if we don't have anything on the stack to return, then return current player status
	if (g_ldv1000_output_stack_pointer < 1)
	{
		// if autostop is active, we need to check to see if we need to stop
		if ((g_ldv1000_output & 0x7F) == 0x54)
		{
//			char f[81];
//			sprintf(f,"Watching Frame: %05u",g_ldp->get_current_frame());
//            tms9128nl_outcommand(f,43,23);

			// if we've hit the frame we need to stop on (or gone too far) then stop
			if (g_ldp->get_current_frame() >= g_ldv1000_autostop_frame)
			{
				g_ldp->pre_pause();
				g_ldv1000_output = (Uint8) ((g_ldv1000_output & 0x80) | 0x65);	// preserve ready bit and set status to paused
				g_ldv1000_autostop_frame = 0;
			}
		}

		//	If stack is empty return normal output
		result = g_ldv1000_output;
		g_ldv1000_output_stack_pointer = 0;
	}
	else 
	{
		//	or pop a value off the stack and return it
		g_ldv1000_output_stack_pointer--;
		result = g_ldv1000_output_stack[g_ldv1000_output_stack_pointer];
	}

	return(result);
}

// pushes a value on the ldv1000 stack, returns 1 if successful or 0 if stack is full
int ldv1000_stack_push(unsigned char value)
{
	int result = 0;

	// if we still have room to push
	if (g_ldv1000_output_stack_pointer < LDV1000_STACKSIZE-1)
	{
		g_ldv1000_output_stack[g_ldv1000_output_stack_pointer++] = value;
		result = 1;
	}
	else
	{
		printline("ERROR: LD-V1000 stack overflow (increase its size)");
	}

	return(result);
}

// sends a byte to our virtual LD-V1000
void write_ldv1000 (unsigned char value)
{

	char s[81] = { 0 };
	char f[81] = { 0 };
	Uint16 curframe = 0;	// current frame we're on

	// if high-bit is set, it means we are ready and so we accept input
	if (g_ldv1000_output & 0x80)
	{
		g_ldv1000_output &= 0x7F;	// clear high bit
		// because when we receive a non 0xFF command we are no longer ready

		switch (value)
		{
		case 0xBF:	// clear
			clear();
			g_ldv1000_output &= 0x7F; // Super Don expects highbit to get cleared when we send a Clear command
			break;
		case 0x3F:	// 0
			ldv1000_add_digit('0');
			break;
		case 0x0F:	// 1
			ldv1000_add_digit('1');
			break;
		case 0x8F:	// 2
			ldv1000_add_digit('2');
			break;
		case 0x4F:	// 3
			ldv1000_add_digit('3');
			break;
		case 0x2F:	// 4
			ldv1000_add_digit('4');
			break;
		case 0xAF:	// 5
			ldv1000_add_digit('5');
			break;
		case 0x6F:	// 6
			ldv1000_add_digit('6');
			break;
		case 0x1F:	// 7
			ldv1000_add_digit('7');
			break;
		case 0x9F:	// 8
			ldv1000_add_digit('8');
			break;
		case 0x5F:	// 9
			ldv1000_add_digit('9');
			break;
		case 0xF4:	// Audio 1
			pre_audio1();
			break;
		case 0xFC:	// Audio 2
			pre_audio2();
			break;
		case 0xA0:	// play at 0X (pause)
			g_ldp->pre_pause();
			break;
		case 0xA1:	// play at 1/4X
		case 0xA2:	// play at 1/2X
		case 0xA4:	// play at 2X
		case 0xA5:	// play at 3X
		case 0xA6:	// play at 4X
		case 0xA7:	// play at 5X
			printline("LDV1000: Unimplemented speed change command received!");
			break;
		case 0xA3:	// play at 1X
			// it is necessary to set the playspeed because otherwise the ld-v1000 ignores skip commands
			g_ldp->set_play_speed(1);
			break;
		case 0xF9:	// Reject - Stop the laserdisc player from playing
			printline("LDV1000: Reject received (ignored)");
			g_ldv1000_output = 0x7c;	// LD-V1000 is PARK'd and NOT READY
			// FIXME: laserdisc state should go into parked mode, but most people probably don't want this to happen (I sure don't)
			break;
		case 0xF3:	// Auto Stop (used by Esh's)
			// This command accepts a frame # as an argument and also begins playing the disc
			g_ldv1000_autostop_frame = get_buffered_frame();
			clear();
			g_ldp->pre_play();
			g_ldv1000_output = 0x54;	// autostop is active
			{
				char s[81];
				sprintf(s, "LDV1000 : Auto-Stop requested at frame %u", g_ldv1000_autostop_frame);
				printline(s);
			}
			break;
		case 0xFD:	// Play
			g_ldp->pre_play();
			g_ldv1000_output = 0x64;	// not ready and playing
			break;
		case 0xF7:	// Search
		
			// if we need to 'look busy' for the benefit of some games (Esh's, Astron Belt) then do so
			if (!g_instant_seeking)
			{
				ldv1000_stack_push(0x50);	// 0x50 means we are busy searching
				ldv1000_stack_push(0x50);
				ldv1000_stack_push(0x50);
				ldv1000_stack_push(0x50);
//				printline("ldv1000 : using slower seeking");
			}

			// if search failed
			if (g_ldp->pre_search(ldv1000_frame) == false)
			{
				printline("LDV1000 Error: Bad search from player!");
				// push search failure code for the LD-V1000 onto stack
				g_ldv1000_output = 0x90;
			}

			// if search succeeded
			else
			{
				// push search success code for the LD-V1000 onto stack
				g_ldv1000_output = 0xD0;
			}
						
			ldv1000_frame_index = 0; // clear frame buffer

			break;
		case 0xC2:	// get current frame
			curframe = g_ldp->get_current_frame();

			g_ldp->framenum_to_frame(curframe, s);
			ldv1000_stack_push(s[4]);
			ldv1000_stack_push(s[3]);
			ldv1000_stack_push(s[2]);
			ldv1000_stack_push(s[1]);
			ldv1000_stack_push(s[0]);	// high digit goes last
			
			// TEMPORARY HACK TO WATCH CURRENT FRAME
			sprintf(f,"Playing Frame: %s",s);
            tms9128nl_outcommand(f,43,23);
			break;
		case 0xB1:	// Skip Forward 10
		case 0xB2:	// Skip Forward 20
		case 0xB3:	// Skip Forward 30
		case 0xB4:	// Skip Forward 40
		case 0xB5:	// Skip Forward 50
		case 0xB6:	// Skip Forward 60
		case 0xB7:	// Skip Forward 70
		case 0xB8:	// Skip Forward 80
		case 0xB9:	// Skip Forward 90
		case 0xBa:	// Skip Forward 100
			// FIXME: ignore skip command if forward command has not been issued
			{
				Uint16 frames_to_skip = (Uint16) ((10 * (value & 0x0f)) + 1);	// LD-V1000 does add 1 when skipping
				g_ldp->pre_skip_forward(frames_to_skip);
			}
			break;
		case 0xCD:	// Display Disable
			pre_display_disable();
			break;
		case 0xCE:	// Display Enable
			pre_display_enable();
			break;
		case 0xFB:	// Stop - this actually just goes into still-frame mode, so we pause
			g_ldp->pre_pause();
			g_ldv1000_output = 0x65;	// stopped and not ready
			break;
		case 0x20:	// Badlands custom command (see below)
//			printline("LD-V1000 debug : Got a 0x20");
// UPDATE : This doesn't work so I am disabling it for now :)
//			g_ldp->pre_skip_backward(16);

			//WO: This should be a relative skip as Matt used, but since that doesn't work in pause mode,
			//I'm using an absolute search instead.  :)
			if (g_game->get_game_type() == GAME_BADLANDS)
			{
				g_ldp->framenum_to_frame((g_ldp->get_current_frame() - 16), s);
				if (g_ldp->pre_search(s) == false)
				{
					printline("LDV1000 Error on Badlands custom skip!");
					// push search failure code for the LD-V1000 onto stack
					g_ldv1000_output = 0x90;  //does this command really expect a response?
				}
				break;  //unsupported for all other games, so keep the break in here to get the default case
			}
			break;
		case 0x31:	// Badlands custom command
			// The Badlands programmers modified the LD-V1000 ROM (!!) and added a new
			// command which is 0x20 0x31.  This command skips backwards 16 frames.  It is
			// used on scenes such as the twirling axe and the red eyes (bats).
//			printline("LD-V1000 debug : Got a 0x31");
			break;
		case 0xFF:	// NO ENTRY
			// it's legal to send the LD-V1000 as many of these as you want, we just ignore 'em
			g_ldv1000_output |= 0x80;	// set highbit just in case
			break;
		default:	// Unsupported Command
			sprintf(s,"Unsupported LD-V1000 Command Received: %x", value);
			printline(s);
			break;
		}
	}

	// if we are not ready, we can become ready by receiving a 0xFF command
	else
	{
		// if we got 0xFF (NO ENTRY) as expected
		if (value == 0xFF)
		{
			g_ldv1000_output |= 0x80;	// set high bit, we are now ready
		}

		// if we got a non NO ENTRY, we just ignore it, only the first non-NO ENTRY matters
		else
		{
			g_ldv1000_output &= 0x7F;	// clear high bit, we are no longer ready
		}
	}

}

// As you can see we are currently not supporting display disable
void pre_display_disable()
{

	printline("Display disable received");

}

// We are not currently supporting display enable
void pre_display_enable()
{

	printline("Display enable received");

}

// Make the LD-V1000 appear to perform instantaneous searches
// NOTE : Doesn't work with all games
// Test this with each game and if it doesn't cause any harm, use it.
// If it breaks the game, don't use it.
void ldv1000_enable_instant_seeking()
{
	printline("LDV1000 : Instantaneous seeking enabled!");
	g_instant_seeking = true;
}

// adds a digit to  the frame array that we will be seeking to
// digit should be in ASCII format
void ldv1000_add_digit(char digit)
{
	// we need to set the high bit for Badlands - it might be caused by some emulation problem
	if (g_game->get_game_type() == GAME_BADLANDS)
	{
		g_ldv1000_output |= 0x80;	
	}

	if (ldv1000_frame_index < FRAME_SIZE)
	{
		ldv1000_frame[ldv1000_frame_index] = digit;
		ldv1000_frame_index++;
	}
	else
	{
		for (int count = 1; count < FRAME_SIZE; count++)
		{
			ldv1000_frame[count - 1] = ldv1000_frame[count];
			ldv1000_frame[ldv1000_frame_index - 1] = digit;
		}
		
		/*
		char s[81] = { 0 };

		sprintf(s, "Too many digits received for frame! (over %d)", FRAME_SIZE);
		printline(s);
		*/
	}
}

// Audio channel 1 on or off
void pre_audio1()
{
	// Check if we should just toggle
	if (ldv1000_frame_index == 0)
	{
		// Check status of audio and toggle accordingly
		if (audio1)
		{
			audio1 = false;
			g_ldp->disable_audio1();
		}
		else
		{
			audio1 = true;
			g_ldp->enable_audio1();
		}
	}
	// Or if we have an explicit audio command
	else 
	{
		switch (ldv1000_frame[0] % 2)
		{
		case 0:
			audio1 = false;
			g_ldp->disable_audio1();
			break;
		case 1:
			audio1 = true;
			g_ldp->enable_audio1();
			break;
		default:
			printline("pre_audio1: Ummm... you shouldn't get this");
		}

		ldv1000_frame_index = 0;
	}
}

// Audio channel 2 on or off
void pre_audio2()
{
	// Check if we should just toggle
	if (ldv1000_frame_index == 0)
	{
		// Check status of audio and toggle accordingly
		if (audio2)
		{
			audio2 = false;
			g_ldp->disable_audio2();
		}
		else
		{
			audio2 = true;
			g_ldp->enable_audio2();
		}
	}
	// Or if we have an explicit audio command
	else 
	{
		switch (ldv1000_frame[0] % 2)
		{
		case 0:
			audio2 = false;
			g_ldp->disable_audio2();
			break;
		case 1:
			audio2 = true;
			g_ldp->enable_audio2();
			break;
		default:
			printline("pre_audio2: Ummm... you shouldn't get this");
		}

		ldv1000_frame_index = 0;
	}
}

// returns the frame that has been entered in by add_digit thus far
Uint16 get_buffered_frame()
{
	ldv1000_frame[ldv1000_frame_index] = 0;	// terminate string
	return ((Uint16) atoi(ldv1000_frame));
}

// clears any received digits from the frame array
void clear(void)
{
	ldv1000_frame_index = 0;	
}

void reset_ldv1000()
{
	// anything else?
	g_ldv1000_output_stack_pointer = 0;
	g_ldv1000_autostop_frame = 0;	// which frame we need to stop on (if any)

	ldv1000_frame_index = 0; 

	g_ldv1000_output = 0xFC;	// LD-V1000 is PARK'd and READY
}
