/*
 * pr8210.cpp
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// pr8210.cpp
// by Matt Ownby
// translates PR-8210 blips into commands
// this should be useful for any game that uses the PR-8210

#include <stdio.h>
#include <string.h>	// for memcmp
#include "../daphne.h"
#include "../io/conout.h"
#include "pr8210.h"
#include "../ldp-out/ldp.h"

int g_pr8210_seek_received = 0;	// whether we've received a seek command
static char frame[6] = {0};

// processes 10 blips into a PR-8210 command
// The blips should be stored in the lowest 10 bits of the integer
// So for example, if my command is 0010000000 then my 32-bit integer would like like
// ???????? ???????? ??????00 10000000
void pr8210_command(unsigned int blips)
{
	char s[80];	// for printing error messages
	static unsigned int old_blips = 0xFFFF;	// the last command we received
	// initialied to 0xFFFF to make sure it is never equal to any new command we get

	blips &= 0x3FF;	// make sure any extra garbage is stripped off

//	printf("Command is ");
//	pr8210_print_binary(blips);
//	printf("\n");

	// if the new blips are not equal to the old ones, then accept command
	if (blips != old_blips)
	{
		// verify header and footer
		// So the format of a command is 001?????00
		if ((blips & 0x383) == 0x80)
		{
			unsigned char command = (unsigned char) ((blips & 0x7C) >> 2);
			// isolate the 5-bit command for comparison
			
			switch (command)
			{
				case 0x14:	// play (10100)
					g_ldp->pre_play();
					break;
				case 0x4:	// step forward (00100)
					printline("PR-8210 : step forward received (unsupported)");
					break;
				case 0xA:	// pause (01010)
					g_ldp->pre_pause();
					break;
				case 0xE:	// Audio 1/L (01110)
					pr8210_audio1();
					break;
				case 0x16:	// Audio 2/R (10110)
					pr8210_audio2();
					break;
				case 0x1A:	// Seek (11010)
					pr8210_seek();
					break;
				case 0x1:	// 0 (00001)
					pr8210_add_digit('0');
					break;
				case 0x11:	// 1 (10001)
					pr8210_add_digit('1');
					break;
				case 0x9:	// 2 (01001)
					pr8210_add_digit('2');
					break;
				case 0x19:	// 3 (11001)
					pr8210_add_digit('3');
					break;
				case 0x5:	// 4 (00101)
					pr8210_add_digit('4');
					break;
				case 0x15:	// 5 (10101)
					pr8210_add_digit('5');
					break;
				case 0xD:	// 6 (01101)
					pr8210_add_digit('6');
					break;
				case 0x1D:	// 7 (11101)
					pr8210_add_digit('7');
					break;
				case 0x3:	// 8 (00011)
					pr8210_add_digit('8');
					break;
				case 0x13:	// 9 (10011)
					pr8210_add_digit('9');
					break;
				case 0x0:	// filler to separate two of the same commands
					break;
				default:
					sprintf(s, "PR8210: Unknown command %x", command);
					printline(s);
					break;
			} // end switch
		} // end valid header and footer
		else
		{
			printline("PR8210 Error : Bad header or footer");
		}

		old_blips = blips;
	} // end if this is a new command	
}

// Audio 1/L
void pr8210_audio1()
{
	printline("PR8210 Audio 1/L (unsupported command!)");
}

// Audio 2/R
void pr8210_audio2()
{
	printline("PR8210 Audio 2/R (unsupported command!)");
}

void pr8210_seek()
{
//	printline("PR8210 Seek Received");
	g_pr8210_seek_received = 1;
}

// ch is the digit to add and it's in ASCII format
void pr8210_add_digit(char ch)
{
	static int digit_count = 0;

//	printf("PR8210 Digit Received : %c\n", ch);

	// make sure we are getting digits after we've gotten a seek
	if (g_pr8210_seek_received)
	{
		frame[digit_count] = ch;

		digit_count++;
		if (digit_count >= 5)
		{
			g_pr8210_seek_received = 0;
			digit_count = 0;
			g_ldp->pre_search(frame);	// perform seek
		}
	}
	
	// if we are getting digits without first getting a seek, something
	// is wrong
	else
	{
		printline("PR8210 error: digit received without seek command");
	}
}

// for debugging only
// prints (to stdout) a 10 bit PR-8210 command
void pr8210_print_binary(unsigned int num)
{
	int i = 0;
	
	for (i = 0; i < 10; i++)
	{
		printf("%d", (num & 0x200) >> 9);
		num = num << 1;	// print the next digit ...
	}
}
