/*
 * ldp-vldp-legacy.cpp
 *
 * Copyright (C) 2003 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// ldp-vldp-legacy.cpp
// by Matt Ownby

// Here is the old code that used the VLDP based on libmpeg2 v0.2.0
// I am keeping it around for comparison purposes, to ensure that the new VLDP2
// is better :)

#include <string.h>
#include <stdlib.h>
#include <time.h>
#include "../io/conout.h"
#include "../io/error.h"
#include "../video/video.h"
#include "../timer/timer.h"
#include "../daphne.h"
#include "../io/input.h"
#include "../io/fileparse.h"
#include <SDL_mixer.h>
#include "../game/game.h"
#include "../video/rgb2yuv.h"
#include "ldp-vldp-legacy.h"
#include "framemod.h"
#include "../vldp/include/vldp.h"	// to get the vldp info struct
#include "../video/palette.h"

#ifdef WIN32
#pragma warning (disable:4100)	// disable the warning about unreferenced formal parameters (MSVC++)
#endif

void ldp_vldp_audio_callback(void *unused, Uint8 *stream, int len);	// declaration for callback in other function

// video overlay stuff
static SDL_Surface *g_gamevid = NULL;	// pointer to the game's video overlay
static SDL_Color *g_gamecolors = NULL;	// pointer to the game's color palette
static Sint32 g_vertical_offset = 0;	// how many lines to shift the game video overlay
static Sint32 g_half_vertical_offset = 0;	// g_vertical_offset / 2 (used a lot, we only want to calc it once)

static bool g_take_screenshot = false;	// if true, a screenshot will be taken at the next available opportunity

/////////////////////////////////////////

// We have to dynamically load the .DLL/.so file due to incompatibilities between MSVC++ and mingw32 library files
// These pointers and typedefs assist us in doing so.

typedef int (*initproc)(SDL_Surface *);
typedef int (*openproc)(char *);
typedef int (*intproc)();
typedef int (*uint32proc)(Uint32);
typedef int (*searchproc)(Uint16);
typedef int (*skipproc)(Uint16, Uint32);
typedef void (*yuvproc)(void (*callback)(SDL_Overlay *cur_overlay));
typedef Uint16 (*frameproc)();
typedef struct vldp_info *(*infoproc)();
typedef void (*blankingproc)(int, int);

static initproc pvldp_init;
static openproc pvldp_open_mpeg;
static searchproc pvldp_search;
static skipproc pvldp_skip;
static uint32proc pvldp_play;
static intproc pvldp_pause, pvldp_shutdown;
static yuvproc pvldp_set_yuv_callback;
static frameproc pvldp_get_current_frame;
static infoproc pvldp_get_info;
static blankingproc pvldp_set_blanking;

/////////////////////////////////////////

ldp_vldp_legacy::ldp_vldp_legacy()
{
	blitting_allowed = false;	// Absolutely no video functions are allowed when using VLDP!  this is due to thread issues
	m_target_mpegframe = 0;	// mpeg frame # we are seeking to
	memset( (void *) &m_mpeg_path[0], 0, sizeof(m_mpeg_path));
	memset(m_cur_mpeg_filename, 0, sizeof(m_cur_mpeg_filename));
	m_file_index = 0; // # of mpeg files in our list
	strcpy(m_framefile, "undefined");	// user has to set the name of the framefile
	strcpy(m_altaudio_suffix, "");  //default is no suffix
	m_audio_file_opened = false;
	m_cur_ldframe_offset = 0;
	enable_audio1();	// both channels should be on by default
	enable_audio2();
	m_blank_on_searches = false;
	m_blank_on_skips = false;
}

// called when daphne starts up
bool ldp_vldp_legacy::init_player()
{
	bool result = false;
	bool need_to_parse = false;	// whether we need to parse all video

	// load the .DLL first in case we call any of its functions elsewhere
	if (load_vldp_lib())
	{
		if (read_frame_conversions())
		{
			// just a sanity check to make sure their frame file is correct
			if (first_video_file_exists())
			{

				// if the last video file has not been parsed, assume none of them have been
				// This is safe because if they have been parsed, it will just skip them
				if (!last_video_file_parsed())
				{
					printnotice("Press any key to parse your video file(s). This may take a while. Press ESC if you'd rather quit.");
					need_to_parse = true;
				}
			
				if (audio_init() && !get_quitflag())
				{
					if ((pvldp_init)(get_screen()))
					{
						char full_path[160];
						
						result = true;
						pvldp_set_blanking(m_blank_on_searches, m_blank_on_skips);
						// setup our blanking preferences
						
						// if we need to parse all the video
						if (need_to_parse)
						{
							parse_all_video();
						}

						strcpy(full_path, m_mpeg_path);
						strcat(full_path, m_mpeginfo[0].name);
						(pvldp_open_mpeg)(full_path);	// open first file so that
						// we can draw video overlay even if the disc is not playing
                        // Need to get/set MPEG size early in case deviation from NTSC defaults.
                        struct vldp_info *info = (pvldp_get_info)();

                        m_discvideo_width = info->w;
                        m_discvideo_height = info->h;

						// check to see if we need to draw a YUV video overlay	
						g_gamevid = g_game->get_active_video_overlay();
						g_vertical_offset = g_game->get_video_row_offset();
						g_half_vertical_offset = g_vertical_offset >> 1;

						// if this game uses a video overlay, then set the YUV callback.
						if (g_gamevid)
						{
							(pvldp_set_yuv_callback)(yuv_callback);
						}
						// else don't set it because performance is better without it

						// if user has sound enabled, then hook into SDL Mixer for our audio callback
						if (is_sound_enabled())
						{
							Mix_HookMusic(ldp_vldp_audio_callback, NULL); // hook into VLDP audio callback
						}
						
					} // end if reading the frame conversion file worked
					else
					{
						printline("Could not initialize VLDP!");
					}
				} // if audio init succeeded
				else
				{
					// only report an audio problem if there is one
					if (!get_quitflag())
					{
						printline("Could not initialize VLDP audio!");
					}

					// otherwise report that they quit
					else
					{
						printline("VLDP : Quit requested, shutting down!");
					}
				} // end if audio init failed or if user opted to quit instead of parse
			} // end if first file was present (sanity check)
			// else if first file was not found, we do just quit because an error is printed elsewhere
		} // end if framefile was read in properly
		else
		{
			printerror("Could not open frame file!");
			printline("You must specify a -framefile argument when using VLDP.");
			printline("If you didn't, then this is the problem. :)");
		}
	} // end if .DLL was loaded properly
	else
	{
		printline("Could not load VLDP dynamic library!!!");
	}


	return result;
}

void ldp_vldp_legacy::shutdown_player()
{
	if (pvldp_shutdown)
	{
		(pvldp_shutdown)();
	}

	if (is_sound_enabled())
	{
		Mix_HookMusic(NULL, NULL); // unhook from sdl mixer
		// add audio_shutdown in here too?
		// (audio_shutdown can be safely run if audio_init has never run so it's ok to leave it outside)
	}
	free_vldp_lib();
	audio_shutdown();
}

bool ldp_vldp_legacy::search(char *frame)
{
	
	bool result = false;
	char filename[81] = { 0 };
	char oggname[81] = { 0 };
	Uint16 target_ld_frame = (Uint16) atoi(frame);
	double fps = 0;
	Uint8 uses_fields = 0;
	double audio_target_pos = 0.0;	// position in audio to seek to

	audio_pause();	// pause the audio before we seek so we don't have overrun

	m_target_mpegframe = mpeg_info(filename, target_ld_frame); // try to get a filename
	audio_target_pos = m_target_mpegframe / g_game->get_disc_fps();	// # of seconds to seek to in the audio stream

	// if we can convert target frame into a filename, do it!
	if (filename[0] != 0)
	{
		result = true;	// now we assume success unless we fail

		// if the file to be opened is different from the one we have opened
		// OR if we don't yet have a file open ...
		// THEN open the file! :)
		if (strcmp(filename, m_cur_mpeg_filename) != 0)
		{
			// if we were able to successfully open the video file
			if ((pvldp_open_mpeg)(filename))
			{
				result = true;
				strcpy(m_cur_mpeg_filename, filename); // make video file our new current file

				// if sound is enabled, try to open an audio stream to match the video stream
				if (is_sound_enabled())
				{
					// try to open an optional audio file to go along with video
					oggize_path(oggname, filename);
					m_audio_file_opened = open_audio_stream(oggname);
				}
			}
			else
			{
				outstr("LDP-VLDP.CPP : Could not open video file ");
				printline(filename);
			}
		}

		// if we're ok so far, try the search
		if (result)
		{
			struct vldp_info *info = (pvldp_get_info)();
			fps = info->fps;
			uses_fields = info->uses_fields;
			m_discvideo_width = info->w;
			m_discvideo_height = info->h;

			// if we're using fields, we have to modify the mpeg target frame so it is the mpeg target field instead
			if (uses_fields)
			{
				m_target_mpegframe = m_target_mpegframe << 1;	// multiply by two since two fields == one frame
			}

			// if we are already doing a frame conversion elsewhere, we don't want to do it here again twice
			// but we do need to set the audio to the correct time
			if (need_frame_conversion())
			{
				audio_target_pos = m_target_mpegframe / get_frame_conversion_fps();
			}
			// if the mpeg's FPS and the disc's FPS differ, we need to adjust the mpeg frame
			// NOTE: AVOID this if you can because it makes seeking less accurate
			else if (g_game->get_disc_fps() != fps)
			{
				char s[160] = { 0 };
				sprintf(s, "NOTE: converting FPS from %f to %f. This may be less accurate.", g_game->get_disc_fps(), fps);
				printline(s);
				m_target_mpegframe = (Uint16) (((m_target_mpegframe / g_game->get_disc_fps()) * fps) + 0.5);
			}

			// try to search to the requested frame
			if ((pvldp_search)((Uint16) m_target_mpegframe))
			{
				// if we have an audio file opened, do an audio seek also
				if (m_audio_file_opened)
				{
					result = seek_audio(audio_target_pos);
				}
			}
			else
			{
				printline("LDP-VLDP.CPP : Search failed in video or audio file");
			}
		}
		// else opening the file failed
	}

	return(result);
}

unsigned int ldp_vldp_legacy::play()
{
	unsigned int result = 0;
	char full_path[320];
	char ogg_path[320];
	int ok = 1;	// whether it's ok to issue the play command
	
	// if we haven't opened any mpeg file yet, then do so now
	if (!m_cur_mpeg_filename[0])
	{
		strcpy(full_path, m_mpeg_path);
		strcat(full_path, m_mpeginfo[0].name);
		ok = (pvldp_open_mpeg)(full_path);	// get the first mpeg available in our list
		if (ok)
		{
			strcpy(m_cur_mpeg_filename, full_path);

			// if sound is enabled, try to load an audio stream to go with video stream ...
			if (is_sound_enabled())
			{			
				// try to open an optional audio file to go along with video
				oggize_path(ogg_path, full_path);
				m_audio_file_opened = open_audio_stream(ogg_path);
			}
			
		}
		else
		{
			outstr("LDP-VLDP.CPP : in play() function, could not open mpeg file ");
			printline(full_path);
		}
	} // end if we haven't opened a file yet

	// we need to keep this separate in case an mpeg is already opened
	if (ok)
	{
		Uint32 timer = SDL_GetTicks();
		audio_play(timer);
		if (pvldp_play(timer))
		{
			result = timer;
		}
	}

	if (!result)
	{
		printline("VLDP ERROR : play command failed!");
	}
	
	return result;
}

// skips forward a certain # of frames during playback without pausing
// Caveats: Does not work with an mpeg of the wrong framerate, does not work with an mpeg
// that uses fields, and does not skip across file boundaries.
// returns true if skip was successful
bool ldp_vldp_legacy::skip_forward(Uint16 frames_to_skip, Uint16 target_frame)
{
	bool result = false;
	struct vldp_info *info = (pvldp_get_info)();

	target_frame = (Uint16) (target_frame - m_cur_ldframe_offset);	// take offset into account
	// this is ok because we don't support skipping across files

	// we don't support skipping on mpegs that differ from the disc framerate
	if (g_game->get_disc_fps() == info->fps)
	{
		// make sure they're not using fields
		if (!info->uses_fields)
		{
			// if we have an audio file opened
			if (m_audio_file_opened)
			{
				double audio_target_pos = target_frame / g_game->get_disc_fps();	// # of seconds to seek to in the audio stream
				// seek and play if seeking was successful
				if (seek_audio(audio_target_pos))
				{
					audio_play(m_play_time);
				}
			}

			// if VLDP was able to skip successfully
			if ((pvldp_skip)(target_frame, m_play_time))
			{
				result = true;
			}
			else
			{
				printline("LDP-VLDP ERROR : video skip failed");
			}
		}
		else
		{
			printline("LDP-VLDP ERROR : Skipping not supported with mpegs that use fields (such as this one)");
		}
	}
	else
	{
		printline("LDP-VLDP ERROR : Skipping not supported when the mpeg's framerate differs from the disc's");
	}
	
	return result;
}

void ldp_vldp_legacy::pause()
{
	(pvldp_pause)();
	audio_pause();
}

/*
// NOTE : re-enable this for testing if you are suspicious of the generic get_current_frame's results
Uint16 ldp_vldp_legacy::get_current_frame()
{
	struct vldp_info *info = (pvldp_get_info)();
	Sint32 result = 0;
	
	// safety check, check to make sure we have an mpeg file loaded
	if (info)
	{
		double fps = info->fps;
		
		// since the mpeg's beginning does not correspond to the laserdisc's beginning, we add the offset
		result = m_cur_ldframe_offset + (pvldp_get_current_frame)();

		// if we're out of bounds, just set it to 0
		if (result <= 0)
		{
			result = 0;
		}
		
		// if we got a legitimate frame
		else
		{
			// FIXME : THIS CODE HAS BUGS IN IT, I HAVEN'T TRACKED THEM DOWN YET HEHE
			// if the mpeg's FPS and the disc's FPS differ, we need to adjust the frame that we return
			// NOTE: AVOID this if you can because it makes seeking less accurate
			if (g_game->get_disc_fps() != fps)
			{
				result = (Sint32) (((result / fps) * g_game->get_disc_fps()) + 0.5);
				// add 0.5 to round to nearest whole number when truncating
			}
		}
	} // if we have an .m2v file open

////////////

	Uint32 cpu_hz = get_cpu_hz(0);
	double elapsed_secs;

	if (cpu_hz)
	{
		Uint64 cur_total_cycles = get_total_cycles_executed(0);
		
		// check to make sure flush_cpu_timers has not been called
		if (cur_total_cycles > m_play_cycles)
		{
			// it seems kind of silly to cast to Uint32 then to double, but Uint64 can't go straight to double
			elapsed_secs = ((double)((Uint32)(cur_total_cycles - m_play_cycles))) / cpu_hz;

			Uint16 result2 = (Uint16) (m_last_frame + (Uint16) (elapsed_secs * g_game->get_disc_fps())); // do not add 0.5!!

			// if the difference between result2 and result1 is greater than 1, we've got a problem
			if (abs(result2 - result) > 1)
			{
				printf("NO! Real=%u Calc=%u, elapsed s is %f, last_frame is %u\n",
					result - m_cur_ldframe_offset, result2 - m_cur_ldframe_offset,
					elapsed_secs,
					m_last_frame);
				printf("result2 is %u, m_cur_ldframe_ofset is %u\n", result2, m_cur_ldframe_offset);
			}
		}
		// else flush timers interfered so we can't give any reliable information
	}

////////////

	return (Uint16) result;
}
*/

// takes a screenshot of the current frame + any video overlay
void ldp_vldp_legacy::request_screenshot()
{
	g_take_screenshot = true;
}

void ldp_vldp_legacy::set_search_blanking(bool enabled)
{
	m_blank_on_searches = enabled;
	
	// safety check, make sure DLL is already loaded
	if (pvldp_set_blanking)
	{
		pvldp_set_blanking(enabled, m_blank_on_skips);
	}
	// else this will take place when pvldp_init is called
}

void ldp_vldp_legacy::set_skip_blanking(bool enabled)
{
	m_blank_on_skips = enabled;

	// safety check, make sure DLL is already loaded
	if (pvldp_set_blanking)
	{
		pvldp_set_blanking(m_blank_on_searches, enabled);
	}
	// else this will take place when pvldp_init is called
}

// sets the name of the frame file
void ldp_vldp_legacy::set_framefile(char *filename)
{
	strcpy(m_framefile, filename);
}


// sets alternate soundtrack
void ldp_vldp_legacy::set_altaudio(char *audio_suffix)
{
	strcpy(m_altaudio_suffix, audio_suffix);
}




//////////////////////////////////

// loads the VLDP dynamic library, returning true on success
bool ldp_vldp_legacy::load_vldp_lib()
{
	bool result = false;

    m_dll_instance = M_LOAD_LIB(vldp);	// load VLDP.DLL or libvldp.so

    // If the handle is valid, try to get the function address. 
    if (m_dll_instance) 
    {
		pvldp_init = (initproc) M_GET_SYM(m_dll_instance, "vldp_init");
		pvldp_open_mpeg = (openproc) M_GET_SYM(m_dll_instance, "vldp_open_mpeg");
		pvldp_search = (searchproc) M_GET_SYM(m_dll_instance, "vldp_search");
		pvldp_skip = (skipproc) M_GET_SYM(m_dll_instance, "vldp_skip");
		pvldp_play = (uint32proc) M_GET_SYM(m_dll_instance, "vldp_play");
		pvldp_pause = (intproc) M_GET_SYM(m_dll_instance, "vldp_pause");
		pvldp_shutdown = (intproc) M_GET_SYM(m_dll_instance, "vldp_shutdown");
		pvldp_set_yuv_callback = (yuvproc) M_GET_SYM(m_dll_instance, "vldp_set_yuv_callback");
		pvldp_get_current_frame = (frameproc) M_GET_SYM(m_dll_instance, "vldp_get_current_frame");
		pvldp_get_info = (infoproc) M_GET_SYM(m_dll_instance, "vldp_get_info");
		pvldp_set_blanking = (blankingproc) M_GET_SYM(m_dll_instance, "vldp_set_blanking");

		// if all functions were found, then return success
		if (pvldp_init && pvldp_open_mpeg && pvldp_search && pvldp_play && pvldp_pause && pvldp_shutdown
			&& pvldp_set_yuv_callback && pvldp_get_current_frame && pvldp_get_info && pvldp_skip
			&& pvldp_set_blanking)
		{
			result = true;
		}
		else
		{
			printline("VLDP LOAD ERROR : some DLL functions could not be loaded");
		}
	}
	else
	{
		printline("ERROR: could not open the VLDP dynamic library");
	}

	return result;
}

// frees the VLDP dynamic library if we loaded it in
void ldp_vldp_legacy::free_vldp_lib()
{
	// don't free if the library was never opened
	if (m_dll_instance)
	{
		M_FREE_LIB(m_dll_instance);
	}
}




// read frame conversions in from LD-frame to mpeg-frame data file
bool ldp_vldp_legacy::read_frame_conversions()
{

	FILE *fileConvert = NULL;
	char s[81] = { 0 };
	int cAssigned = 0;
	char frame_string[81] = { 0 };
	Sint32 frame = 0;
	bool result = false;
	char framefile_path[160];	// FIXME : don't assume 160 bytes is always sufficient

	strcpy(framefile_path, m_framefile);
	fileConvert = fopen(framefile_path, "r");

	// if the file was not found in relative directory, try looking for it in a framefile directory
	if (!fileConvert)
	{
		strcpy(framefile_path, "framefile/");
		strcat(framefile_path, m_framefile);
		fileConvert = fopen(framefile_path, "r");
	}

	// if the framefile was opened successfully
	if (fileConvert)
	{
		char ch = 0;	// for temporary use

		read_line(fileConvert, m_mpeg_path, sizeof(m_mpeg_path));	// read in path

		ch = m_mpeg_path[strlen(m_mpeg_path)-1];

		// Clean up after the user if they goofed up on the path
		if ((ch != '\\') && (ch != '/'))
		{
			strcat(m_mpeg_path, "/");	// windows will accept / as well as \, so we're ok here
		}

		// if m_mpeg_path is NOT an absolute path
		// then we want to use the framefile's path for convenience purposes
		// (this should be win32 and unix compatible)
		if ((m_mpeg_path[0] != '/') && (m_mpeg_path[0] != '\\') && (m_mpeg_path[1] != ':'))
		{
			char path[sizeof(framefile_path)] = { 0 };

			// try to isolate the path of the framefile
			if (get_path_of_file(framefile_path, path))
			{
				char bigbuf[320] = { 0 };

				// put the path of the framefile in front of the relative path of the mpeg files
				// This will allow people to move the location of their mpegs around to
				// different directories without changing the framefile at all.
				// For example, if the framefile and the mpegs are in the same directory,
				// then the framefile's first line could be "./"
				strcpy(bigbuf, path);
				strcat(bigbuf,  m_mpeg_path);
				strcpy(m_mpeg_path, bigbuf);
				// NOTE : doing this is very dangerous with respect to buffer overflow
				// but I don't want to deal with it right now :)
			}
		}

		// read entire file
		while(!feof(fileConvert))
		{

			cAssigned = fscanf(fileConvert, "%s %s", frame_string, s);
			frame = atoi(frame_string);
				               
			// if we read 2 entries like we expected, store them
			if(cAssigned == 2)
			{
				m_mpeginfo[m_file_index].frame = (Sint32) frame;
				strcpy(m_mpeginfo[m_file_index].name, s);
				m_file_index++;
			}

			// check for overflow
			if (m_file_index >= MAX_MPEG_FILES)
			{
				printline("ERROR : Your framefile has too many entries in it.");
				printline("You need to increase the value of MAX_MPEG_FILES and recompile.");
				break;
			}
		}

		result = true;	
		fclose(fileConvert);
	}
	else
	{
		sprintf(s, "Could not open mpeg frame conversion file : %s", m_framefile);
		printline(s);
	}
	
	return result;
}

// if file does not exist, we print an error message
bool ldp_vldp_legacy::first_video_file_exists()
{
	char full_path[160] = { 0 };
	char s[160] = { 0 };
	FILE *F = NULL;
	bool result = false;
	
	// if we have at least one file
	if (m_file_index)
	{
		strcpy(full_path, m_mpeg_path);
		strcat(full_path, m_mpeginfo[0].name);	// first video file
		F = fopen(full_path, "rb");	// test to see if file exists
		
		// if we could open the file, then file exists
		if (F)
		{
			result = true;
			fclose(F);
		}
		else
		{
			sprintf(s, "Could not open file : %s", full_path);
			printerror(s);
		}
	}

	return result;
}

// returns true if the last video file has been parsed
// This is so we don't parse_all_video if all files are already parsed
bool ldp_vldp_legacy::last_video_file_parsed()
{
	char full_path[160] = { 0 };
	FILE *F = NULL;
	bool result = false;
	
	// if we have at least one file
	if (m_file_index)
	{
		strcpy(full_path, m_mpeg_path);
		strcat(full_path, m_mpeginfo[m_file_index-1].name);
		strcpy(full_path + strlen(full_path) - 3, "dat");	// change suffix
		F = fopen(full_path, "rb");	// test to see if file exists
		
		// if we could open the file, then file exists
		if (F)
		{
			result = true;
			fclose(F);
		}
	}
	// else there is a problem with the frame file so return false
	
	return result;
}

// opens (and closes) all video files, forcing any unparsed video files to get parsed
void ldp_vldp_legacy::parse_all_video()
{
	int i = 0;
	char full_path[160] = { 0 };
	
	for (i = 0; i < m_file_index; i++)
	{
		strcpy(full_path, m_mpeg_path);	// create a full pathname to file
		strcat(full_path, m_mpeginfo[i].name);	// add current filename
//		printf("Opening file %s ...\n", full_path);
		(pvldp_open_mpeg)(full_path);
		(pvldp_search)(0);	// search to frame 0 to render it so the user has something to watch while he/she waits
	}
}

// returns # of frames to seek into file, and mpeg filename
// if there's an error, filename is NULL
// WARNING: This assumes the mpeg and disc are running at exactly the same FPS
// If they aren't, you will need to calculate the actual mpeg frame to seek to
// The reason I don't return time here instead of frames is because it is more accurate to
//  return frames if they are at the same FPS (which hopefully they are hehe)
Uint16 ldp_vldp_legacy::mpeg_info (char *filename, Uint16 ld_frame)
{
	int index = 0;
	Uint16 mpeg_frame = 0;	// which mpeg frame to seek (assuming mpeg and disc have same FPS)

	// find the mpeg file that has the LD frame inside of it
	while ((index+1 < m_file_index) && (ld_frame >= m_mpeginfo[index+1].frame))
	{
		index = index + 1;
	}

	// make sure we have a filename at this spot
	if (m_mpeginfo[index].name[0])
	{
		strcpy(filename, m_mpeg_path);
		strcat(filename, m_mpeginfo[index].name);
		mpeg_frame = (Uint16) (ld_frame - m_mpeginfo[index].frame);
		m_cur_ldframe_offset = m_mpeginfo[index].frame;
//		printf("setting m_cur_ldframe_offset %d\n", m_cur_ldframe_offset);
	}
	else
	{
		printline("VLDP error, no filename found");
		filename[0] = 0;	// ERROR
	}
	
	return(mpeg_frame);
}

////////////////////////////////////////////////////////////////////////////////////////

static bool mutex_lock_request = false;
static bool mutex_lock_acknowledge = false;

// puts the yuv_callback into a blocking state
// This is necessary if the g_gamevid ever becomes invalid for a period of time (ie it gets free'd and re-allocated in seektest)
// timeout is how many ms to wait before giving up
// returns true if it got the lock or false if it couldn't get a lock
bool ldp_vldp_legacy::lock_overlay(Uint32 timeout)
{
	Uint32 time = refresh_ms_time();

	mutex_lock_request = true;

	// sleep until the yuv callback acknowledges that we have control
	while (!mutex_lock_acknowledge)
	{
		SDL_Delay(0);
		if (elapsed_ms_time(time) > timeout)
		{
			printline("LDP_VLDP : lock_overlay timed out without getting a lock");
			mutex_lock_request = false;	// just to be safe, we want to default to non-locked mode
			break;
		}
	}

	return (mutex_lock_acknowledge);
}

// releases the yuv_callback from its blocking state
bool ldp_vldp_legacy::unlock_overlay(Uint32 timeout)
{
	Uint32 time = refresh_ms_time();

	mutex_lock_request = false;

	// sleep until the yuv callback acknowledges that it has control once again
	while (mutex_lock_acknowledge)
	{
		SDL_Delay(0);

		// if we've timed out
		if (elapsed_ms_time(time) > timeout)
		{
			printline("LDP_VLDP : unlock_overlay timed out while trying to unlock");
			break;
		}
	}

	return (!mutex_lock_acknowledge);
}

// Draws game video overlay on top of laserdisc YUV video
// It is assumed the the SDL_Overlay is safely writable
// NOTE : this function could be optimized--it's called every frame on games w/ video overlay
// NOTE2 : this function is called from a separate thread
void yuv_callback(SDL_Overlay *dst)
{
	// we assume that g_vertical_offset and g_gamecolors are all valid at this point (and also constant!!!)

//	Uint8 transparent_val = g_game->get_transparent_color();	// transparent value can change at any time
	Uint8 *Y_line_begin = dst->pixels[0];	// points to beginning of Y channel data on current line
	Uint8 *V_line_begin = dst->pixels[1];	// points to beginning of V channel data on current line
	Uint8 *U_line_begin = dst->pixels[2];	// points to beginning of U channel data on current line
	Uint8 *Y = NULL, *V = NULL, *U = NULL;	// roaming pointers

	int row = 0;
	int col = 0;

	// if another thread has requested that we block, then do so.
	// DANGER : This is dangerous because it has endless loop potential
	// However, it is also fast and therefore we just need to be conscious coders and not let endless loops happen
	if (mutex_lock_request)
	{
		mutex_lock_acknowledge = 1;
		// wait until other thread has unlocked (endless loop potential)
		while (mutex_lock_request)
		{
			SDL_Delay(0);
		}
		mutex_lock_acknowledge = 0;
	}

	g_gamevid = g_game->get_active_video_overlay();	// This could change at any time (double buffering, for example)
	// so we are forced to query it every time we run this function.  If someone has a better idea, let me know

	// sanity check.  Make sure the game video is the proper width.  If it's not return
	if ((g_gamevid->w << 1) != dst->w)
	{
		static bool warned = false;

		// newbies might appreciate knowing why their video overlay isn't working, so
		// let's give them some instruction in the logfile.  We only want to print this once
		// to avoid spamming, hence the 'warned' boolean
		if (!warned)
		{
            if (!g_game->is_overlay_size_dynamic())
            {
			    char s[81];

			    printline("CRITICAL ERROR : Your MPEG is the wrong resolution.  Video overlay will not work!");
			    sprintf(s, "Your MPEG needs to be %d x %d pixels!", dst->w, dst->h);
			    printline(s);
            }
			// else, there is no problem at all, so don't alarm the user
    	    warned = true;
		}

        if (g_game->is_overlay_size_dynamic())
        {
            // MPEG size has changed, so drop a hint to any game
            // that resizes dynamically.
            g_game->set_video_overlay_needs_update(true);
        }

		return;
	}

	Uint8 *gamevid_pixels = (Uint8 *) g_gamevid->pixels - (g_gamevid->w * g_half_vertical_offset);

	// this could be global, any benefit?
	t_yuv_color* yuv_palette = get_yuv_palette();

	Uint8 *temp = NULL;	// temporary pointer
	Uint8 cur_color = 0;
//	int last_color = -1;	// the last color we converted to YUV
	// made an integer so I could set it to -1 and not risk overlapping
	// any legitimate 8-bit color

	// go through each row of the overlay and copy it to the YUV stream
	for (row = 0; (row < g_gamevid->h) && ((row << 1) < dst->h); row ++)
	{
		// starting a new line, so update our roaming pointers
		Y = Y_line_begin;
		V = V_line_begin;
		U = U_line_begin;

		// do each pixel in the current line
		for (col = 0; col < g_gamevid->w; col++)
		{
			cur_color = *gamevid_pixels;
			
			// if current game video overlay pixel is not transparent
			// then write it to the YUV stream
			if ((!yuv_palette[cur_color].transparent) && ((row - g_half_vertical_offset) > 0)) 
			{
				// we double size of pixel so we need to write
				// the value in 4 places

				// this plots a 2x2 square (a 1x1 pixel doubled in size)
				*Y = yuv_palette[cur_color].y;
				*(Y+1) = yuv_palette[cur_color].y;
				temp = Y + dst->pitches[0];
				*temp = yuv_palette[cur_color].y;
				*(temp+1) = yuv_palette[cur_color].y;

				// U and V are half the size of Y so we do not need to enlarge
				*V = yuv_palette[cur_color].v;
				*U = yuv_palette[cur_color].u;
			} // end if it's not a transparent color

			gamevid_pixels++;	// advance to next pixel
			Y += 2;	// advance two pixels over since game video is doubled
			U++;
			V++;
		} // end columns for loop
		
		Y_line_begin += (dst->pitches[0] << 1);	// skip the next line because we've already done it
		V_line_begin += dst->pitches[1];
		U_line_begin += dst->pitches[2];
	}

	// if we've been instructed to take a screenshot, do so now that the overlay is in place
	if (g_take_screenshot)
	{
		g_take_screenshot = false;
		take_screenshot(dst);
	}
}

