/*
 * ldp-vldp.cpp
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// ldp-vldp.c
// by Matt Ownby

// pretends to be an LDP, but uses the VLDP library for output
// (for people who have no laserdisc player)

// If you want to experiment with a YUY2 overlay, uncomment this ...
// My tests show that it is surprisingly slow, I'm not sure why ...
//#define TRY_YUY2 1

#include <string.h>
#include <stdlib.h>
#include <time.h>
#include "../io/conout.h"
#include "../io/error.h"
#include "../video/video.h"
#include "../timer/timer.h"
#include "../daphne.h"
#include "../io/input.h"
#include "../io/fileparse.h"
#include <SDL_mixer.h>
#include "../game/game.h"
#include "../video/rgb2yuv.h"
#include "ldp-vldp.h"
#include "framemod.h"
#include "../vldp2/vldp/vldp.h"	// to get the vldp structs
#include "../video/palette.h"
#include "../video/SDL_DrawText.h"

#ifdef WIN32
#pragma warning (disable:4100)	// disable the warning about unreferenced formal parameters (MSVC++)
#endif

void ldp_vldp_audio_callback(void *unused, Uint8 *stream, int len);	// declaration for callback in other function

// video overlay stuff
Sint32 g_half_vertical_offset = 0;	// (used a lot, we only want to calc it once)

bool g_take_screenshot = false;	// if true, a screenshot will be taken at the next available opportunity

/////////////////////////////////////////

// We have to dynamically load the .DLL/.so file due to incompatibilities between MSVC++ and mingw32 library files
// These pointers and typedefs assist us in doing so.

typedef const struct vldp_out_info *(*initproc)(const struct vldp_in_info *in_info);
initproc pvldp_init;	// pointer to the init proc ...

// pointer to all functions the VLDP exposes to us ...
const struct vldp_out_info *g_vldp_info = NULL;

// info that we provide to the VLDP DLL
struct vldp_in_info g_local_info;

/////////////////////////////////////////

ldp_vldp::ldp_vldp()
{
	blitting_allowed = false;	// Absolutely no video functions are allowed when using VLDP!  this is due to thread issues
	m_target_mpegframe = 0;	// mpeg frame # we are seeking to
	memset( (void *) &m_mpeg_path[0], 0, sizeof(m_mpeg_path));
	memset(m_cur_mpeg_filename, 0, sizeof(m_cur_mpeg_filename));
	m_file_index = 0; // # of mpeg files in our list
	MPO_STRNCPY(m_framefile, "undefined", sizeof(m_framefile));	// user has to set the name of the framefile
	MPO_STRNCPY(m_altaudio_suffix, "", sizeof(m_altaudio_suffix));  //default is no suffix
	m_audio_file_opened = false;
	m_cur_ldframe_offset = 0;
	enable_audio1();	// both channels should be on by default
	enable_audio2();
	m_blank_on_searches = false;
	m_blank_on_skips = false;
}

// called when daphne starts up
bool ldp_vldp::init_player()
{
	bool result = false;
	bool need_to_parse = false;	// whether we need to parse all video

	// load the .DLL first in case we call any of its functions elsewhere
	if (load_vldp_lib())
	{
		if (read_frame_conversions())
		{
			// just a sanity check to make sure their frame file is correct
			if (first_video_file_exists())
			{

				// if the last video file has not been parsed, assume none of them have been
				// This is safe because if they have been parsed, it will just skip them
				if (!last_video_file_parsed())
				{
					printnotice("Press any key to parse your video file(s). This may take a while. Press ESC if you'd rather quit.");
					need_to_parse = true;
				}
			
				if (audio_init() && !get_quitflag())
				{
					// if our game is using video overlay, we'll use our slower callback
					if (g_game->get_active_video_overlay())
					{
						g_local_info.prepare_frame = prepare_frame_callback_with_overlay;
					}
					
					// otherwise we can draw the frame much faster w/o worrying about
					// video overlay
					else
					{
						g_local_info.prepare_frame = prepare_frame_callback_without_overlay;
					}
					g_local_info.display_frame = display_frame_callback;
					g_local_info.report_parse_progress = report_parse_progress_callback;
					g_local_info.report_mpeg_dimensions = report_mpeg_dimensions_callback;
					g_local_info.render_blank_frame = blank_overlay;
					g_local_info.blank_during_searches = m_blank_on_searches;
					g_local_info.blank_during_skips = m_blank_on_skips;
					g_vldp_info = pvldp_init(&g_local_info);

					// if we successfully made contact with VLDP ...
					if (g_vldp_info != NULL)
					{
						char full_path[160];
						
						result = true;
						
						// if we need to parse all the video
						if (need_to_parse)
						{
							parse_all_video();
						}

						MPO_STRNCPY(full_path, m_mpeg_path, sizeof(full_path));
						MPO_STRNCAT(full_path, m_mpeginfo[0].name, sizeof(full_path));
						reset_parse_meter();	// must come before the open_and_block call
						g_vldp_info->open_and_block(full_path);	// open first file so that
						// we can draw video overlay even if the disc is not playing

						// check to see if we need to draw a YUV video overlay
						g_half_vertical_offset = g_game->get_video_row_offset() >> 1;

                        // set MPEG size ASAP in case different from NTSC default
			            m_discvideo_width = g_vldp_info->w;
			            m_discvideo_height = g_vldp_info->h;

						// if user has sound enabled, then hook into SDL Mixer for our audio callback
						if (is_sound_enabled())
						{
							Mix_HookMusic(ldp_vldp_audio_callback, NULL); // hook into VLDP audio callback
						}
						
					} // end if reading the frame conversion file worked
					else
					{
						printline("Could not initialize VLDP!");
					}
				} // if audio init succeeded
				else
				{
					// only report an audio problem if there is one
					if (!get_quitflag())
					{
						printline("Could not initialize VLDP audio!");
					}

					// otherwise report that they quit
					else
					{
						printline("VLDP : Quit requested, shutting down!");
					}
				} // end if audio init failed or if user opted to quit instead of parse
			} // end if first file was present (sanity check)
			// else if first file was not found, we do just quit because an error is printed elsewhere
		} // end if framefile was read in properly
		else
		{
			printerror("Could not open frame file!");
			printline("You must specify a -framefile argument when using VLDP.");
			printline("If you didn't, then this is the problem. :)");
		}
	} // end if .DLL was loaded properly
	else
	{
		printline("Could not load VLDP dynamic library!!!");
	}


	return result;
}

void ldp_vldp::shutdown_player()
{
	// if VLDP has been loaded
	if (g_vldp_info)
	{
		g_vldp_info->shutdown();
	}

	if (is_sound_enabled())
	{
		Mix_HookMusic(NULL, NULL); // unhook from sdl mixer
		// add audio_shutdown in here too?
		// (audio_shutdown can be safely run if audio_init has never run so it's ok to leave it outside)
	}
	free_vldp_lib();
	audio_shutdown();

	free_yuv_overlay();	// de-allocate overlay if we have one allocated ...
}

bool ldp_vldp::search(char *frame)
{
	
	bool result = false;
	char filename[VLDP_STR_SIZE] = { 0 };
	char oggname[VLDP_STR_SIZE] = { 0 };
	Uint16 target_ld_frame = (Uint16) atoi(frame);
	double fps = 0;
	Uint8 uses_fields = 0;
	double audio_target_pos = 0.0;	// position in audio to seek to

	audio_pause();	// pause the audio before we seek so we don't have overrun

	m_target_mpegframe = mpeg_info(filename, target_ld_frame, sizeof(filename)); // try to get a filename
	audio_target_pos = m_target_mpegframe / g_game->get_disc_fps();	// # of seconds to seek to in the audio stream

	// if we can convert target frame into a filename, do it!
	if (filename[0] != 0)
	{
		result = true;	// now we assume success unless we fail

		// if the file to be opened is different from the one we have opened
		// OR if we don't yet have a file open ...
		// THEN open the file! :)
		if (strcmp(filename, m_cur_mpeg_filename) != 0)
		{
			reset_parse_meter(); // must come before the open_and_block call
			// if we were able to successfully open the video file
			if (g_vldp_info->open_and_block(filename))
			{
				result = true;
				MPO_STRNCPY(m_cur_mpeg_filename, filename, sizeof(m_cur_mpeg_filename)); // make video file our new current file

				// if sound is enabled, try to open an audio stream to match the video stream
				if (is_sound_enabled())
				{
					// try to open an optional audio file to go along with video
					oggize_path(oggname, filename);
					m_audio_file_opened = open_audio_stream(oggname);
				}
			}
			else
			{
				outstr("LDP-VLDP.CPP : Could not open video file ");
				printline(filename);
				result = false;
			}
		}

		// if we're ok so far, try the search
		if (result)
		{
			fps = g_vldp_info->fps;
			uses_fields = g_vldp_info->uses_fields;
			m_discvideo_width = g_vldp_info->w;
			m_discvideo_height = g_vldp_info->h;

			// if we're using fields, we have to modify the mpeg target frame so it is the mpeg target field instead
			if (uses_fields)
			{
				m_target_mpegframe = m_target_mpegframe << 1;	// multiply by two since two fields == one frame
			}

			// if we are already doing a frame conversion elsewhere, we don't want to do it here again twice
			// but we do need to set the audio to the correct time
			if (need_frame_conversion())
			{
				audio_target_pos = m_target_mpegframe / get_frame_conversion_fps();
			}
			// if the mpeg's FPS and the disc's FPS differ, we need to adjust the mpeg frame
			// NOTE: AVOID this if you can because it makes seeking less accurate
			else if (g_game->get_disc_fps() != fps)
			{
				char s[160] = { 0 };
				sprintf(s, "NOTE: converting FPS from %f to %f. This may be less accurate.", g_game->get_disc_fps(), fps);
				printline(s);
				m_target_mpegframe = (Uint16) (((m_target_mpegframe / g_game->get_disc_fps()) * fps) + 0.5);
			}

			// try to search to the requested frame
			if (g_vldp_info->search_and_block((Uint16) m_target_mpegframe))
			{
				// if we have an audio file opened, do an audio seek also
				if (m_audio_file_opened)
				{
					result = seek_audio(audio_target_pos);
				}
			}
			else
			{
				printline("LDP-VLDP.CPP : Search failed in video or audio file");
			}
		}
		// else opening the file failed
	}

	return(result);
}

unsigned int ldp_vldp::play()
{
	unsigned int result = 0;
	char full_path[VLDP_STR_SIZE];
	char ogg_path[VLDP_STR_SIZE];
	int ok = 1;	// whether it's ok to issue the play command
	
	// if we haven't opened any mpeg file yet, then do so now
	if (!m_cur_mpeg_filename[0])
	{
		MPO_STRNCPY(full_path, m_mpeg_path, sizeof(full_path));
		MPO_STRNCAT(full_path, m_mpeginfo[0].name, sizeof(full_path));
		reset_parse_meter(); // must come before the open_and_block call
		ok = g_vldp_info->open_and_block(full_path);	// get the first mpeg available in our list
		if (ok)
		{
			MPO_STRNCPY(m_cur_mpeg_filename, full_path, sizeof(m_cur_mpeg_filename));

			// if sound is enabled, try to load an audio stream to go with video stream ...
			if (is_sound_enabled())
			{			
				// try to open an optional audio file to go along with video
				oggize_path(ogg_path, full_path);
				m_audio_file_opened = open_audio_stream(ogg_path);
			}
			
		}
		else
		{
			outstr("LDP-VLDP.CPP : in play() function, could not open mpeg file ");
			printline(full_path);
		}
	} // end if we haven't opened a file yet

	// we need to keep this separate in case an mpeg is already opened
	if (ok)
	{
		Uint32 timer = SDL_GetTicks();
		audio_play(timer);
		if (g_vldp_info->play(timer))
		{
			result = timer;
		}
	}

	if (!result)
	{
		printline("VLDP ERROR : play command failed!");
	}
	
	return result;
}

// skips forward a certain # of frames during playback without pausing
// Caveats: Does not work with an mpeg of the wrong framerate, does not work with an mpeg
// that uses fields, and does not skip across file boundaries.
// returns true if skip was successful
bool ldp_vldp::skip_forward(Uint16 frames_to_skip, Uint16 target_frame)
{
	bool result = false;

	target_frame = (Uint16) (target_frame - m_cur_ldframe_offset);	// take offset into account
	// this is ok because we don't support skipping across files

	// we don't support skipping on mpegs that differ from the disc framerate
	if (g_game->get_disc_fps() == g_vldp_info->fps)
	{
		// make sure they're not using fields
		if (!g_vldp_info->uses_fields)
		{
			// if we have an audio file opened
			if (m_audio_file_opened)
			{
				double audio_target_pos = target_frame / g_game->get_disc_fps();	// # of seconds to seek to in the audio stream
				// seek and play if seeking was successful
				if (seek_audio(audio_target_pos))
				{
					audio_play(m_play_time);
				}
			}

			// if VLDP was able to skip successfully
			if (g_vldp_info->skip(target_frame, m_play_time))
			{
				result = true;
			}
			else
			{
				printline("LDP-VLDP ERROR : video skip failed");
			}
		}
		else
		{
			printline("LDP-VLDP ERROR : Skipping not supported with mpegs that use fields (such as this one)");
		}
	}
	else
	{
		printline("LDP-VLDP ERROR : Skipping not supported when the mpeg's framerate differs from the disc's");
	}
	
	return result;
}

void ldp_vldp::pause()
{
	g_vldp_info->pause();
	audio_pause();
}

/*
// NOTE : re-enable this for testing if you are suspicious of the generic get_current_frame's results
Uint16 ldp_vldp::get_current_frame()
{
	struct vldp_info *info = (pvldp_get_info)();
	Sint32 result = 0;
	
	// safety check, check to make sure we have an mpeg file loaded
	if (info)
	{
		double fps = info->fps;
		
		// since the mpeg's beginning does not correspond to the laserdisc's beginning, we add the offset
		result = m_cur_ldframe_offset + (pvldp_get_current_frame)();

		// if we're out of bounds, just set it to 0
		if (result <= 0)
		{
			result = 0;
		}
		
		// if we got a legitimate frame
		else
		{
			// FIXME : THIS CODE HAS BUGS IN IT, I HAVEN'T TRACKED THEM DOWN YET HEHE
			// if the mpeg's FPS and the disc's FPS differ, we need to adjust the frame that we return
			// NOTE: AVOID this if you can because it makes seeking less accurate
			if (g_game->get_disc_fps() != fps)
			{
				result = (Sint32) (((result / fps) * g_game->get_disc_fps()) + 0.5);
				// add 0.5 to round to nearest whole number when truncating
			}
		}
	} // if we have an .m2v file open

////////////

	Uint32 cpu_hz = get_cpu_hz(0);
	double elapsed_secs;

	if (cpu_hz)
	{
		Uint64 cur_total_cycles = get_total_cycles_executed(0);
		
		// check to make sure flush_cpu_timers has not been called
		if (cur_total_cycles > m_play_cycles)
		{
			// it seems kind of silly to cast to Uint32 then to double, but Uint64 can't go straight to double
			elapsed_secs = ((double)((Uint32)(cur_total_cycles - m_play_cycles))) / cpu_hz;

			Uint16 result2 = (Uint16) (m_last_frame + (Uint16) (elapsed_secs * g_game->get_disc_fps())); // do not add 0.5!!

			// if the difference between result2 and result1 is greater than 1, we've got a problem
			if (abs(result2 - result) > 1)
			{
				printf("NO! Real=%u Calc=%u, elapsed s is %f, last_frame is %u\n",
					result - m_cur_ldframe_offset, result2 - m_cur_ldframe_offset,
					elapsed_secs,
					m_last_frame);
				printf("result2 is %u, m_cur_ldframe_ofset is %u\n", result2, m_cur_ldframe_offset);
			}
		}
		// else flush timers interfered so we can't give any reliable information
	}

////////////

	return (Uint16) result;
}
*/

// takes a screenshot of the current frame + any video overlay
void ldp_vldp::request_screenshot()
{
	g_take_screenshot = true;
}

void ldp_vldp::set_search_blanking(bool enabled)
{
	m_blank_on_searches = enabled;
	// ADD ME
}

void ldp_vldp::set_skip_blanking(bool enabled)
{
	m_blank_on_skips = enabled;
	// ADD ME
}

// sets the name of the frame file
void ldp_vldp::set_framefile(char *filename)
{
	MPO_STRNCPY(m_framefile, filename, sizeof(m_framefile));
}

// sets alternate soundtrack
void ldp_vldp::set_altaudio(char *audio_suffix)
{
	MPO_STRNCPY(m_altaudio_suffix, audio_suffix, sizeof(m_altaudio_suffix));
}




//////////////////////////////////

// loads the VLDP dynamic library, returning true on success
bool ldp_vldp::load_vldp_lib()
{
	bool result = false;

    m_dll_instance = M_LOAD_LIB(vldp2);	// load VLDP2.DLL or libvldp2.so

    // If the handle is valid, try to get the function address. 
    if (m_dll_instance)
    {
		pvldp_init = (initproc) M_GET_SYM(m_dll_instance, "vldp_init");

		// if all functions were found, then return success
		if (pvldp_init)
		{
			result = true;
		}
		else
		{
			printerror("VLDP LOAD ERROR : vldp_init could not be loaded");
		}
	}
	else
	{
		printerror("ERROR: could not open the VLDP2 dynamic library (file not found maybe?)");
	}

	return result;
}

// frees the VLDP dynamic library if we loaded it in
void ldp_vldp::free_vldp_lib()
{
	// don't free if the library was never opened
	if (m_dll_instance)
	{
		M_FREE_LIB(m_dll_instance);
	}
}




// read frame conversions in from LD-frame to mpeg-frame data file
bool ldp_vldp::read_frame_conversions()
{

	FILE *fileConvert = NULL;
	int cAssigned = 0;
	char frame_string[81] = { 0 };
	Sint32 frame = 0;
	bool result = false;
	char framefile_path[VLDP_STR_SIZE];

	MPO_STRNCPY(framefile_path, m_framefile, sizeof(framefile_path));
	fileConvert = fopen(framefile_path, "r");

	// if the file was not found in relative directory, try looking for it in a framefile directory
	if (!fileConvert)
	{
		MPO_STRNCPY(framefile_path, "framefile/", sizeof(framefile_path));
		MPO_STRNCAT(framefile_path, m_framefile, sizeof(framefile_path));
		fileConvert = fopen(framefile_path, "r");
	}

	// if the framefile was opened successfully
	if (fileConvert)
	{
		char ch = 0;	// for temporary use

		read_line(fileConvert, m_mpeg_path, sizeof(m_mpeg_path));	// read in path

		ch = m_mpeg_path[strlen(m_mpeg_path)-1];

		// Clean up after the user if they goofed up on the path
		if ((ch != '\\') && (ch != '/'))
		{
			MPO_STRNCAT(m_mpeg_path, "/", sizeof(m_mpeg_path));	// windows will accept / as well as \, so we're ok here
		}

		// if m_mpeg_path is NOT an absolute path
		// then we want to use the framefile's path for convenience purposes
		// (this should be win32 and unix compatible)
		if ((m_mpeg_path[0] != '/') && (m_mpeg_path[0] != '\\') && (m_mpeg_path[1] != ':'))
		{
			char path[sizeof(framefile_path)] = { 0 };

			// try to isolate the path of the framefile
			if (get_path_of_file(framefile_path, path))
			{
				char bigbuf[VLDP_STR_SIZE*2] = { 0 };

				// put the path of the framefile in front of the relative path of the mpeg files
				// This will allow people to move the location of their mpegs around to
				// different directories without changing the framefile at all.
				// For example, if the framefile and the mpegs are in the same directory,
				// then the framefile's first line could be "./"
				MPO_STRNCPY(bigbuf, path, sizeof(bigbuf));
				MPO_STRNCAT(bigbuf, m_mpeg_path, sizeof(bigbuf));
				MPO_STRNCPY(m_mpeg_path, bigbuf, sizeof(m_mpeg_path));
			}
		}

		result = true;	

		// read entire file
		while(!feof(fileConvert))
		{
			char s[VLDP_STR_SIZE];
			cAssigned = fscanf(fileConvert, "%s %s", frame_string, s);	// NOTE : this could overflow 's' so it's still not entirely safe
			frame = atoi(frame_string);
				               
			// if we read 2 entries like we expected, store them
			if(cAssigned == 2)
			{
				m_mpeginfo[m_file_index].frame = (Sint32) frame;
				MPO_STRNCPY(m_mpeginfo[m_file_index].name, s, sizeof(m_mpeginfo[m_file_index].name));
				m_file_index++;
			}
			// if this not a blank line, but has some bogus data on i
			else if ((cAssigned != 0) && (cAssigned != EOF))
			{
				// This error has been stumping self-proclaimed "experienced emu'ers"
				// so I am going to extra effort to make it clear to them what the
				// problem is.
				printerror("Framefile is invalid! See docs to learn how to make framefiles.");
				printline("Framefile ERROR : Expected a number followed by a string, but didn't find it!");
				printline("Read the documentation on how to make framefiles.");
				result = false;
				break;
			}
			// else it is probably just an empty line or EOF, so we can ignore it

			// check for overflow
			if (m_file_index >= MAX_MPEG_FILES)
			{
				printline("ERROR : Your framefile has too many entries in it.");
				printline("You need to increase the value of MAX_MPEG_FILES and recompile.");
				result = false;
				break;
			}
		}

		fclose(fileConvert);
	}
	else
	{
		outstr("Could not open mpeg frame conversion file : ");
		printline(m_framefile);
	}
	
	return result;
}

// if file does not exist, we print an error message
bool ldp_vldp::first_video_file_exists()
{
	char full_path[VLDP_STR_SIZE] = { 0 };
	char s[1000] = { 0 };	// to minimize possibility of buffer overflow
	FILE *F = NULL;
	bool result = false;
	
	// if we have at least one file
	if (m_file_index)
	{
		MPO_STRNCPY(full_path, m_mpeg_path, sizeof(full_path));
		MPO_STRNCAT(full_path, m_mpeginfo[0].name, sizeof(full_path));	// first video file
		F = fopen(full_path, "rb");	// test to see if file exists
		
		// if we could open the file, then file exists
		if (F)
		{
			result = true;
			fclose(F);
		}
		else
		{
			sprintf(s, "Could not open file : %s", full_path);
			printerror(s);
		}
	}
	else
	{
		printerror("ERROR : Framefile seems empty, it's probably invalid");
		printline("Read the documentation to learn how to create framefiles.");
	}

	return result;
}

// returns true if the last video file has been parsed
// This is so we don't parse_all_video if all files are already parsed
bool ldp_vldp::last_video_file_parsed()
{
	char full_path[VLDP_STR_SIZE] = { 0 };
	FILE *F = NULL;
	bool result = false;
	
	// if we have at least one file
	if (m_file_index)
	{
		MPO_STRNCPY(full_path, m_mpeg_path, sizeof(full_path));
		MPO_STRNCAT(full_path, m_mpeginfo[m_file_index-1].name, sizeof(full_path));
		// it's ok to use strcpy here as long as we verify that the overall length is sufficient
		if (strlen(full_path) > 4)
		{
			strcpy(full_path + strlen(full_path) - 3, "dat");	// change suffix
		}
		F = fopen(full_path, "rb");	// test to see if file exists
		
		// if we could open the file, then file exists
		if (F)
		{
			result = true;
			fclose(F);
		}
	}
	// else there is a problem with the frame file so return false
	
	return result;
}

// opens (and closes) all video files, forcing any unparsed video files to get parsed
void ldp_vldp::parse_all_video()
{
	int i = 0;
	char full_path[VLDP_STR_SIZE] = { 0 };
	
	for (i = 0; i < m_file_index; i++)
	{
		MPO_STRNCPY(full_path, m_mpeg_path, sizeof(full_path));	// create a full pathname to file
		MPO_STRNCAT(full_path, m_mpeginfo[i].name, sizeof(full_path));	// add current filename
		reset_parse_meter(); // must come before the open_and_block call

		// if the file can be opened
		if (g_vldp_info->open_and_block(full_path) == 1)
		{
			g_vldp_info->search_and_block(0);	// search to frame 0 to render it so the user has something to watch while he/she waits
		}
		else
		{
			outstr("ERROR: LDP-VLDP: Could not parse video because file ");
			outstr(full_path);
			printline(" could not be opened.");
			break;
		}
	}
}

// returns # of frames to seek into file, and mpeg filename
// if there's an error, filename is NULL
// WARNING: This assumes the mpeg and disc are running at exactly the same FPS
// If they aren't, you will need to calculate the actual mpeg frame to seek to
// The reason I don't return time here instead of frames is because it is more accurate to
//  return frames if they are at the same FPS (which hopefully they are hehe)
Uint16 ldp_vldp::mpeg_info (char *filename, Uint16 ld_frame, unsigned int filename_size)
{
	int index = 0;
	Uint16 mpeg_frame = 0;	// which mpeg frame to seek (assuming mpeg and disc have same FPS)

	// find the mpeg file that has the LD frame inside of it
	while ((index+1 < m_file_index) && (ld_frame >= m_mpeginfo[index+1].frame))
	{
		index = index + 1;
	}

	// make sure we have a filename at this spot
	if (m_mpeginfo[index].name[0])
	{
		MPO_STRNCPY(filename, m_mpeg_path, filename_size);
		MPO_STRNCAT(filename, m_mpeginfo[index].name, filename_size);
		mpeg_frame = (Uint16) (ld_frame - m_mpeginfo[index].frame);
		m_cur_ldframe_offset = m_mpeginfo[index].frame;
//		printf("setting m_cur_ldframe_offset %d\n", m_cur_ldframe_offset);
	}
	else
	{
		printline("VLDP error, no filename found");
		filename[0] = 0;	// ERROR
	}
	
	return(mpeg_frame);
}

////////////////////////////////////////////////////////////////////////////////////////

bool mutex_lock_request = false;
bool mutex_lock_acknowledge = false;

// puts the yuv_callback into a blocking state
// This is necessary if the gamevid ever becomes invalid for a period of time (ie it gets free'd and re-allocated in seektest)
// timeout is how many ms to wait before giving up
// returns true if it got the lock or false if it couldn't get a lock
bool ldp_vldp::lock_overlay(Uint32 timeout)
{
	Uint32 time = refresh_ms_time();

	mutex_lock_request = true;

	// sleep until the yuv callback acknowledges that we have control
	while (!mutex_lock_acknowledge)
	{
		SDL_Delay(0);
		if (elapsed_ms_time(time) > timeout)
		{
			printline("LDP_VLDP : lock_overlay timed out without getting a lock");
			mutex_lock_request = false;	// just to be safe, we want to default to non-locked mode
			break;
		}
	}

	return (mutex_lock_acknowledge);
}

// releases the yuv_callback from its blocking state
bool ldp_vldp::unlock_overlay(Uint32 timeout)
{
	Uint32 time = refresh_ms_time();

	mutex_lock_request = false;

	// sleep until the yuv callback acknowledges that it has control once again
	while (mutex_lock_acknowledge)
	{
		SDL_Delay(0);

		// if we've timed out
		if (elapsed_ms_time(time) > timeout)
		{
			printline("LDP_VLDP : unlock_overlay timed out while trying to unlock");
			break;
		}
	}

	return (!mutex_lock_acknowledge);
}

//////////////////////////////////////////////////////////////////////
SDL_Rect *g_screen_clip_rect = NULL;	// used a lot, we only want to calulcate once
SDL_Overlay *g_hw_overlay = NULL;
struct yuv_buf g_blank_yuv_buf;	// this will contain a blank YUV overlay suitable for search/seek blanking

// returns VLDP_TRUE on success, VLDP_FALSE on failure
int prepare_frame_callback_with_overlay(struct yuv_buf *buf)
{
	int result = VLDP_FALSE;

	// if another thread has requested that we block, then do so.
	// DANGER : This is dangerous because it has endless loop potential
	// However, it is also fast and therefore we just need to be conscious coders and not let endless loops happen
	if (mutex_lock_request)
	{
		mutex_lock_acknowledge = 1;
		// wait until other thread has unlocked (endless loop potential)
		while (mutex_lock_request)
		{
			SDL_Delay(0);
		}
		mutex_lock_acknowledge = 0;
	}

	if (SDL_LockYUVOverlay(g_hw_overlay) == 0)
	{
#ifdef TRY_YUY2
		buf2overlay_YUY2(g_hw_overlay, buf);
#else
		buf2overlay(g_hw_overlay, buf);	// copy buffer
#endif

		Uint8 *dst_Y_line_begin = (Uint8 *) g_hw_overlay->pixels[0];
		Uint8 *dst_V_line_begin = (Uint8 *) g_hw_overlay->pixels[1];
		Uint8 *dst_U_line_begin = (Uint8 *) g_hw_overlay->pixels[2];
		Uint8 *dst_Y = NULL;
		Uint8 *dst_V = NULL;
		Uint8 *dst_U = NULL;

		int row = 0;
		int col = 0;

		SDL_Surface *gamevid = g_game->get_finished_video_overlay();	// This could change at any time (double buffering, for example)
		// so we are forced to query it every time we run this function.  If someone has a better idea, let me know

		// sanity check.  Make sure the game video is the proper width.  If it's not return
		if ((gamevid->w << 1) != g_hw_overlay->w)
		{
			static bool warned = false;
		
			// newbies might appreciate knowing why their video overlay isn't working, so
			// let's give them some instruction in the logfile.  We only want to print this once
			// to avoid spamming, hence the 'warned' boolean
			if (!warned)
			{
				// if the game does not dynamically reallocate its overlay size (most of them don't)
				if (!g_game->is_overlay_size_dynamic())
				{
					char s[81];
					printline("WARNING : Your MPEG doesn't match your video overlay's resolution.");
					printline("Video overlay will not work!");
					sprintf(s, "Your MPEG's size is %d x %d, and needs to be %d x %d", g_hw_overlay->w, g_hw_overlay->h, (gamevid->w << 1), (gamevid->h << 1));
					printline(s);
				}
				// else, there is no problem at all, so don't alarm the user
				warned = true;
			}

			if (g_game->is_overlay_size_dynamic())
			{
				// MPEG size has changed, so drop a hint to any game
				// that resizes dynamically.
				g_game->set_video_overlay_needs_update(true);
			}
		} // end sanity check

		// if sanity check passed
		else
		{

			// adjust for vertical offset
			// We use _half_ of the requested vertical offset because the mpeg video is twice
			// the size of the overlay
			Uint8 *gamevid_pixels = (Uint8 *) gamevid->pixels - (gamevid->w * g_half_vertical_offset);
			
			// this could be global, any benefit?
			t_yuv_color* yuv_palette = get_yuv_palette();
			
			Uint8 *temp = NULL;	// temporary pointer
			Uint8 cur_color = 0;

#ifdef TRY_YUY2
			unsigned int pitch = g_hw_overlay->pitches[0];
			Uint8 *dst_line_ptr = (Uint8 *) g_hw_overlay->pixels[0];
			Uint8 *dst_ptr = NULL;

			for (row = 0; (row < gamevid->h) && ((row << 1) < g_hw_overlay->h); row++)
			{
				dst_ptr = dst_line_ptr;

				for (col = 0; col < gamevid->w; col++)
				{
					cur_color = *gamevid_pixels;

					// safety check
					if ((row - g_half_vertical_offset) >= 0)
					{
						// if current pixel is not transparent, then draw it
						if (!yuv_palette[cur_color].transparent)
						{
							// upper left
							*dst_ptr = yuv_palette[cur_color].y;
							*(dst_ptr+1) = yuv_palette[cur_color].u;

							// upper right
							*(dst_ptr+2) = yuv_palette[cur_color].y;
							*(dst_ptr+3) = yuv_palette[cur_color].v;

							// lower left
							*(dst_ptr + pitch) = yuv_palette[cur_color].y;
							*(dst_ptr + pitch + 1) = yuv_palette[cur_color].u;

							// lower right
							*(dst_ptr + pitch + 2) = yuv_palette[cur_color].y;
							*(dst_ptr + pitch + 3) = yuv_palette[cur_color].v;
						}
					}

					gamevid_pixels++;	// advance to the next pixel
					dst_ptr += 4;	// we've done 2 horizontal pixels, so move over 2 pixels (4 bytes)
				}
				dst_line_ptr += (pitch << 1);	// we've done 2 vertical pixels, so skip a row
			}
#else
			// go through each row of the overlay and copy it to the YUV stream
			for (row = 0; (row < gamevid->h) && ((row << 1) < g_hw_overlay->h); row ++)
			{
				dst_Y = dst_Y_line_begin;
				dst_U = dst_U_line_begin;
				dst_V = dst_V_line_begin;
				
				// do each pixel in the current line
				for (col = 0; col < gamevid->w; col++)
				{
					cur_color = *gamevid_pixels;
			
					// safety check
					if ((row - g_half_vertical_offset) >= 0)
					{
						// if current pixel is not transparent ...
						if (!yuv_palette[cur_color].transparent)
						{
							// we double size of pixel so we need to write
							// the value in 4 places
			
							// this plots a 2x2 square (a 1x1 pixel doubled in size)
							*dst_Y = yuv_palette[cur_color].y;
							*(dst_Y+1) = yuv_palette[cur_color].y;
							temp = dst_Y + g_hw_overlay->pitches[0];	// move down exactly 1 line
							*temp = yuv_palette[cur_color].y;
							*(temp+1) = yuv_palette[cur_color].y;
			
							// U and V are half the size of Y so we do not need to enlarge
							*dst_V = yuv_palette[cur_color].v;
							*dst_U = yuv_palette[cur_color].u;
						} // end if it's not a transparent color
					}
			
					gamevid_pixels++;	// advance to next pixel
					dst_Y += 2;	// advance two pixels over since game video is doubled
					dst_U++;
					dst_V++;
				} // end columns for loop

				// move the three channels down 1 line
				dst_Y_line_begin += (g_hw_overlay->pitches[0] << 1); // skip a full line on the Y channel because we've already done it
				dst_V_line_begin += g_hw_overlay->pitches[1];
				dst_U_line_begin += g_hw_overlay->pitches[2];
			} // end rows for loop

			// if we've been instructed to take a screenshot, do so now that the overlay is in place
			if (g_take_screenshot)
			{
				g_take_screenshot = false;
				take_screenshot(g_hw_overlay);
			}
#endif

			result = VLDP_TRUE;	// we were successful

		} // end if sanity check passed

		SDL_UnlockYUVOverlay(g_hw_overlay);
	} // end if locking the overlay was successful

	// else we are trying to feed info to the overlay too quickly, so we'll just have to wait

	return result;
}

// faster callback because it assumes we have no overlay
// returns VLDP_TRUE on success, VLDP_FALSE on failure
int prepare_frame_callback_without_overlay(struct yuv_buf *buf)
{
	int result = VLDP_FALSE;

	// if locking the video overlay is successful
	if (SDL_LockYUVOverlay(g_hw_overlay) == 0)
	{
#ifdef TRY_YUY2
		buf2overlay_YUY2(g_hw_overlay, buf);
#else
		buf2overlay(g_hw_overlay, buf);

		// if we've been instructed to take a screenshot, do so now that the overlay is in place
		if (g_take_screenshot)
		{
			g_take_screenshot = false;
			take_screenshot(g_hw_overlay);
		}
#endif

		SDL_UnlockYUVOverlay(g_hw_overlay);

		result = VLDP_TRUE;
	}

	// else just ignore

	return result;
}

// displays the frame as fast as possible
void display_frame_callback(struct yuv_buf *buf)
{
	SDL_DisplayYUVOverlay(g_hw_overlay, g_screen_clip_rect);
}

// copies the contents of src into dst
// assumes destination overlay is locked and *IMPORTANT* assumes src and dst are the same resolution
// thx to Benoit Miller for this fix
// Optimizing this function would help significantly, so try if you can!
void buf2overlay(SDL_Overlay *dst, struct yuv_buf *src)
{
	Uint8 *src_ptr = (Uint8 *) src->Y;   // start with Y
	Uint8 *dst_ptr = dst->pixels[0];
	Uint32 i;
	Uint32 w_half = dst->w >> 1;	// half of the overlay width, to avoid calculating this more than once
	Uint32 h_half = dst->h >> 1;	// half of the overlay height, to avoid calculating this more than once

	// copy Y layer
	for (i = 0; i < (Uint32) dst->h; i++)
	{
		memcpy(dst_ptr, src_ptr, dst->w);
		src_ptr += dst->w;
		dst_ptr += dst->pitches[0];
	}

	src_ptr = (Uint8 *) src->V;   // V
	dst_ptr = dst->pixels[1];
	// copy V layer
	for (i = 0; i < h_half; i++)
	{
		memcpy(dst_ptr, src_ptr, w_half);
		src_ptr += w_half;
		dst_ptr += dst->pitches[1];
	}

	src_ptr = (Uint8 *) src->U;   // U
	dst_ptr = dst->pixels[2];
	// copy U layer
	for (i = 0; i < h_half; i++)
	{
		memcpy(dst_ptr, src_ptr, w_half);
		src_ptr += w_half;
		dst_ptr += dst->pitches[2];
	}
}

// IMPORTANT : This function is not intended to be used!  It's just here as an experiment ...
// This function converts the YV12-formatted 'src' to a YUY2-formatted overlay (which Xbox-Daphne may be using)
// copies the contents of src into dst
// assumes destination overlay is locked and *IMPORTANT* assumes src and dst are the same resolution
void buf2overlay_YUY2(SDL_Overlay *dst, struct yuv_buf *src)
{
	unsigned int pitch = dst->pitches[0];
	Uint8 *dst_line_ptr = (unsigned char *) dst->pixels[0];
	Uint8 *dst_ptr = NULL;
	Uint8 *Y = (Uint8 *) src->Y;
	Uint8 *U = (Uint8 *) src->U;
	Uint8 *V = (Uint8 *) src->V;

	for (int row = 0; row < (dst->h >> 1); row++)
	{
		dst_ptr = dst_line_ptr;
		for (int col = 0; col < (dst->w >> 1); col++)
		{
/*
			// upper left
			*dst_ptr = *Y;
			*(dst_ptr+1) = *U;

			// upper right
			*(dst_ptr+2) = *(Y+1);
			*(dst_ptr+3) = *V;

			// lower left
			*(dst_ptr + pitch) = *(Y + dst->w);
			*(dst_ptr + pitch + 1) = *U;

			// lower right
			*(dst_ptr + pitch + 2) = *(Y + dst->w + 1);
			*(dst_ptr + pitch + 3) = *V;

			dst_ptr += 4;
			Y += 2;	// we've done 2 horiz Y pixels, so more over 2 pixels (2 bytes)
			U++;	// we've done 2 horiz U pixels so move over 2 pixels (1 byte)
			V++;	// we've done 2 horiz V pixels so move over 2 pixels (1 byte)
*/
			*dst_ptr = *Y;
			*(dst_ptr + pitch) = *(Y + dst->w);
			dst_ptr++;

			*dst_ptr = *U;
			*(dst_ptr + pitch) = *(U++);
			dst_ptr++;

			Y++;

			*dst_ptr = *Y;
			*(dst_ptr + pitch) = *(Y + dst->w);
			dst_ptr++;

			*dst_ptr = *V;
			*(dst_ptr + pitch) = *(V++);
			dst_ptr++;

			Y++;

		}
		dst_line_ptr += (pitch << 1);	// we've done 2 vertical pixels, so skip a row
		Y += dst->w;	// we've done 2 vertical Y pixels, so skip a row
	}	
}

///////////////////

Uint32 g_parse_start_time = 0;	// when mpeg parsing began approximately ...
double g_parse_start_percentage = 0.0;	// the first percentage report we received ...
bool g_parse_started = false;	// whether we've received any data at all ...

// called just before an mpeg is opened in case it starts to get parsed
void reset_parse_meter()
{
	g_parse_started = false;
}

// percent_complete is between 0 and 1
void report_parse_progress_callback(double percent_complete_01)
{
	double elapsed_s = 0;	// how many seconds have elapsed since we began this ...
	double total_s = 0;	// how many seconds the entire operation is likely to take
	double remaining_s = 0;	// how many seconds are remaining

	double percent_complete = percent_complete_01 * 100.0;	// switch it from [0-1] to [0-100]

	// if this is the first data we've got since the parsing began ...
	if (!g_parse_started)
	{
		free_yuv_overlay();	// because we will be blitting to the screen
		g_parse_start_time = refresh_ms_time();
		g_parse_start_percentage = percent_complete;
		g_parse_started = true;
	}

	// if we already have some data collected ...
	else
	{
		elapsed_s = (elapsed_ms_time(g_parse_start_time)) * 0.001;	// compute elapsed seconds
		double percentage_accomplished = percent_complete - g_parse_start_percentage;	// how much 'percentage' points we've accomplished

		total_s = (elapsed_s * 100.0) / percentage_accomplished;	// 100 signifies 100 percent (I got this equation by doing some algebra on paper)

		// as long as percent_complete is always 100 or lower, total_s will always be >= elapsed_s, so no checking necessary here
		remaining_s = total_s - elapsed_s;
	}

	SDL_Surface *screen = get_screen();	// the main screen that we can draw on ...
	SDL_FillRect(screen, NULL, 0);	// erase screen

	// if we have some progress to report ...
	if (remaining_s > 0)
	{
		char s[160];
		int half_h = screen->h >> 1;	// calculations to center message on screen ...
		int half_w = screen->w >> 1;
		sprintf(s, "Video parsing is %02.f percent complete, %02.f seconds remaining.\n", percent_complete, remaining_s);
		SDLDrawText(s, screen, FONT_SMALL, (half_w-((strlen(s)/2)*FONT_SMALL_W)), half_h-FONT_SMALL_H);

		// now draw a little graph thing ...
		SDL_Rect clip = screen->clip_rect;
		const int THICKNESS = 10;	// how thick our little status bar will be
		clip.y = (clip.h - THICKNESS) / 2;	// where to start our little status bar
		clip.h = THICKNESS;
		clip.y += FONT_SMALL_H + 5;	// give us some padding

		SDL_FillRect(screen, &clip, SDL_MapRGB(screen->format, 255, 255, 255));	// draw a white bar across the screen ...

		clip.x++;	// move left boundary in 1 pixel
		clip.y++;	// move upper boundary down 1 pixel
		clip.w -= 2;	// move right boundary in 1 pixel
		clip.h -= 2;	// move lower boundary in 1 pixel

		SDL_FillRect(screen, &clip, SDL_MapRGB(screen->format, 0, 0, 0));	// fill inside with black

		clip.w = (Uint16) ((screen->w * percent_complete_01) + 0.5)-1;	// compute how wide our progress bar should be (-1 to take into account left pixel border)

		// go from full red (hardly complete) to full green (fully complete)
		SDL_FillRect(screen, &clip, SDL_MapRGB(screen->format,
			(Uint8) (255 * (1.0 - percent_complete_01)),
			(Uint8) (255 * percent_complete_01),
			0));
	}

	SDL_Flip(screen);	// display the danged thing ...
	SDL_check_input();	// this will allow windows to repaint the screen just in case it needs to ...

}

///////////////////

// this always gets called before the draw_callback ...
void report_mpeg_dimensions_callback(int width, int height)
{
	g_screen_clip_rect = &get_screen()->clip_rect;	// used a lot, we only want to calculate it once
	
	// if an overlay exists, but its dimensions are wrong, we need to de-allocate it
	if (g_hw_overlay && ((g_hw_overlay->w != width) || (g_hw_overlay->h != height)))
	{
		free_yuv_overlay();		
	}

	// if our overlay has been de-allocated, or if we never had one to begin with ... then allocate it now
	if (!g_hw_overlay)
	{
#ifdef TRY_YUY2
		g_hw_overlay = SDL_CreateYUVOverlay (width, height, SDL_YUY2_OVERLAY, get_screen());
#else
		g_hw_overlay = SDL_CreateYUVOverlay (width, height, SDL_YV12_OVERLAY, get_screen());
#endif

		// safety check
		if (!g_hw_overlay)
		{
			printline("ldp-vldp.cpp : YUV overlay creation failed!");
			set_quitflag();
		}

		// we don't need to check whether these buffers have been allocated or not because this is checked for earlier
		// when we check to see if g_hw_overlay has been allocated
		g_blank_yuv_buf.Y_size = width*height;
		g_blank_yuv_buf.Y = malloc(g_blank_yuv_buf.Y_size);
		memset(g_blank_yuv_buf.Y, 0, g_blank_yuv_buf.Y_size);	// blank Y color
		g_blank_yuv_buf.UV_size = g_blank_yuv_buf.Y_size >> 2;
		g_blank_yuv_buf.U = malloc(g_blank_yuv_buf.UV_size);
		memset(g_blank_yuv_buf.U, 127, g_blank_yuv_buf.UV_size);	// blank U color
		g_blank_yuv_buf.V = malloc(g_blank_yuv_buf.UV_size);
		memset(g_blank_yuv_buf.V, 127, g_blank_yuv_buf.UV_size);	// blank V color
	}
}

void free_yuv_overlay()
{
	if (g_hw_overlay)
	{
		SDL_FreeYUVOverlay(g_hw_overlay);
	}
	g_hw_overlay = NULL;

	// free blank buf ...
	free(g_blank_yuv_buf.Y);
	free(g_blank_yuv_buf.U);
	free(g_blank_yuv_buf.V);
	memset(&g_blank_yuv_buf, 0, sizeof(g_blank_yuv_buf));
}

// makes the laserdisc video black while drawing game's video overlay on top
void blank_overlay()
{
	// only do this if the HW overlay has already been allocated
	if (g_hw_overlay)
	{
		g_local_info.prepare_frame(&g_blank_yuv_buf);
		g_local_info.display_frame(&g_blank_yuv_buf);
	}
}
