/*
 * ldp-vldp.h
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef LDP_VLDP_H
#define LDP_VLDP_H

#include <SDL.h>

// maximum # of mpeg files that we will handle
#define MAX_MPEG_FILES 500

// using arrays will be replaced with STL strings in the next release, but for now we'll
// use big arrays :)
#define VLDP_STR_SIZE 320

// safe strncpy with null termination
#define MPO_STRNCPY(dst,src,size) strncpy(dst, src, size); dst[size-1] = 0;

// safe strncat with null termination
#define MPO_STRNCAT(dst,src,size) if (strlen(dst) < size) { strncat(dst, src, size-strlen(dst)); dst[size-1] = 0; }

struct fileframes1
{
	char name[VLDP_STR_SIZE];	// name of mpeg file
	Sint32 frame;	// ldp frame that mpeg file starts on
};

#include "ldp.h"
#include "../io/dll.h"

class ldp_vldp : public ldp
{
public:
	ldp_vldp();

	// all set virtual for ldp-vldp-legacy's benefit
	virtual bool init_player();
	virtual void shutdown_player();
	virtual bool search(char *);
	virtual unsigned int play();
	virtual bool skip_forward(Uint16 frames_to_skip, Uint16 target_frame);
	virtual void pause();
//	virtual Uint16 get_current_frame();	// enable for accuracy testing only
	virtual void request_screenshot();
	virtual void set_search_blanking(bool);
	virtual void set_skip_blanking(bool);

	virtual void set_framefile(char *filename);
	virtual void set_altaudio(char *audio_suffix);
	virtual bool lock_overlay(Uint32);
	virtual bool unlock_overlay(Uint32);

private:
	bool load_vldp_lib();
	void free_vldp_lib();
	bool read_frame_conversions();
	bool first_video_file_exists();
	bool last_video_file_parsed();
	void parse_all_video();

	// filename_size indicates the size of the array (in bytes) that filename points to
	Uint16 mpeg_info (char *filename, Uint16 ld_frame, unsigned int filename_size);
	
	Sint32 m_target_mpegframe;	// mpeg frame # we are seeking to
	Sint32 m_cur_ldframe_offset;	// which laserdisc frame corresponds to the first frame in current mpeg file
	char m_cur_mpeg_filename[VLDP_STR_SIZE];	// name of the mpeg file we currently have open
	char m_mpeg_path[VLDP_STR_SIZE*2]; // location of mpeg file(s) to play
	char m_framefile[VLDP_STR_SIZE];	// name of the framefile we load to get the names of the mpeg files and their frame #'s
	char m_altaudio_suffix[32]; //adds a suffix to the ogg filename, to support alternate soundtracks
	struct fileframes1 m_mpeginfo[MAX_MPEG_FILES]; // names of mpeg files
	int m_file_index; // # of mpeg files in our list
	DLL_INSTANCE m_dll_instance;	// pointer to DLL we load

	bool m_audio_file_opened;	// whether we have audio to accompany the video
	bool m_blank_on_searches;	// should we blank while searching?
	bool m_blank_on_skips;		// should we blank while skipping?

//////////////////////////////////////////////////

	// stuff inside ldp-vldp-audio.cpp
public:
	void enable_audio1();
	void enable_audio2();
	void disable_audio1();
	void disable_audio2();
// this is protected so the ldp-vldp-legacy class can access these audio functions
protected:
	void oggize_path(char *, char *);
	bool audio_init();
	void audio_shutdown();
	void close_audio_stream();
	bool open_audio_stream(char *filename);
	bool seek_audio(double time);
	void audio_play(Uint32);
	void audio_pause();
};

int prepare_frame_callback_with_overlay(struct yuv_buf *buf);
int prepare_frame_callback_without_overlay(struct yuv_buf *buf);
void display_frame_callback(struct yuv_buf *buf);
void buf2overlay(SDL_Overlay *dst, struct yuv_buf *src);
void buf2overlay_YUY2(SDL_Overlay *dst, struct yuv_buf *src);
void reset_parse_meter();
void report_parse_progress_callback(double percent_complete);
void report_mpeg_dimensions_callback(int, int);
void free_yuv_overlay();
void blank_overlay();

#endif
