
/*
 * ldp.cpp
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// LDP.CPP
// by Matt Ownby

// The code in this file translates general LDP functions into LDP-specific functions
// Part of the Daphne emulator

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "../daphne.h"
#include "../io/serial.h"
#include "ldp.h"
#include "../timer/timer.h"
#include "../io/conout.h"
#include "framemod.h"
#include "../game/game.h"
#include "../game/boardinfo.h"
#include "../cpu/cpu.h"
#include "../cpu/generic_z80.h"

#ifdef WIN32
#pragma warning (disable:4100)	// disable the warning about unreferenced parameters
#endif

// generic ldp constructor
ldp::ldp()
{
	blitting_allowed = true;
	need_serial = false;
	serial_initialized = false;	// assume we don't need serial
	player_initialized = false;
	m_last_frame = 0;
	m_play_cycles = 0;
	m_play_time = 0;
	m_status = LDP_STOPPED;	// this is a safe assumption.. if the disc is already playing, no harm done
	search_latency = 0;
	m_stop_on_quit = false;	// do not stop the laserdisc player when daphne quits (by default)
	skipping_supported = false;
	skip_instead_of_search = false;
	max_skippable_frames = 0;
	m_discvideo_width = 640;	// NTSC default, overwritten only by VLDP
	m_discvideo_height = 480;	// NTSC default, overwritten only by VLDP
}

// Initializes and "pings" the LDP/VLDP and returns true if the player is online and detected
// or false if the LDP could not be initialized
bool ldp::init ()
{

	bool result = true;	// assume everything works until we find out otherwise
	bool temp = true;	// needed to make && work the way we want below

	// If we are controlling a real LDP,
	// or if we are controlling a combo and DVD initialized properly,
	// then initialize the serial port here
	if (need_serial)
	{
		serial_initialized = serial_init(get_serial_port(), get_baud_rate());
		temp = serial_initialized;
	}
	
	player_initialized = init_player();
	result = temp && player_initialized;
	
	return(result);

}

// initializes the player after serial has been initialized
// this is so that the main init function can initialize the serial without having each LDP
// initialize its own serial
bool ldp::init_player()
{

	return(true);

}

// STOPs the player, shuts down the serial if it's open, then calls shutdown_player for player-specific stuff
void ldp::shutdown()
{
	// if player has been initialized, stop the player from playing
	if (player_initialized && m_stop_on_quit)
	{
		pre_stop();
	}
	
	// if serial has been initialized, shut it down now
	if (serial_initialized)
	{
		serial_close();
		serial_initialized = false;
	}

	shutdown_player();
	player_initialized = false;
}

// does any player specific stuff  
void ldp::shutdown_player()
{
}

// prepares to search
// searches to the frame inside the "frame" array
// frame_index should be set to FRAME_SIZE before this function is called
// (this happens naturally if add_digit is called 5 times)
bool ldp::pre_search(char* frame)
{

	Uint16 frame_number = 0;	// frame to search to
	bool result = false;
	char s1[81] = { 0 };

	m_status = LDP_BUSY;	// searching can take a while

	frame[5] = 0;	// terminate the string ...
	frame_number = (Uint16) atoi(frame);

	// If we need to alter the frame # before searching
	if (need_frame_conversion())
	{
		Uint16 unadjusted_frame = frame_number;
		frame_number = (Uint16) do_frame_conversion(frame_number);
		framenum_to_frame(frame_number, frame);
		sprintf(s1, "Search to %d (formerly %d) received", frame_number, unadjusted_frame);
	}
	else
	{
		sprintf(s1, "Search to %d received", frame_number);
	}

	outstr(s1);

	// if it's Dragon's Lair/Space Ace, print the board we are on
	if ((g_game->get_game_type() >= GAME_LAIR) && (g_game->get_game_type() <= GAME_ACE))
	{
		Uint8 *cpumem = get_cpu_mem(0);	// get the memory for the first (and only)
		outstr(" - ");
		print_board_info(cpumem[0xA00E], cpumem[0xA00F], cpumem[Z80_GET_IY]);
	}
	else
	{
		newline();
	}

	// If the user requested a delay before seeking, make it now
	if (search_latency > 0)
	{
		make_delay(get_search_latency());
	}

	// HERE IS WHERE THE SEARCH ACTUALLY TAKES PLACE

	// if the player supports skipping AND user has requested the we skip instead of searching when possible
	if (skipping_supported & skip_instead_of_search)
	{
		int difference = frame_number - get_current_frame();	// how many frames we'd have to skip

		// if we can skip forward instead of seeking, do it!
		if ((difference <= max_skippable_frames) && (difference > 1))
		{
			result = pre_skip_forward((Uint16) difference);
		}
		// if it's too far to skip, search instead
		else
		{
			result = search(frame);
		}
	}
	// otherwise do a regular search
	else
	{
		result = search(frame);
	}

	if (result)
	{
		m_last_frame = (Uint16) atoi(frame);
	}

	flush_cpu_timers();	// for now, we make all seeking instantaneous by flushing the timers
							// we probably will want to change this later but that will require some rewriting
	m_status = LDP_PAUSED;	// after a seek, the laserdisc player always goes into a paused state

	return(result);
}

// does the actual search
// frame is in ASCII format, and is always 5 digits
bool ldp::search(char *new_frame)
{
	// I did this to get rid of compiler warnings
	if (new_frame)
	{
	}

	return(true);
}

// prepare to skip forward a certain # of frames and continue playing
// Call this function, do not call skip_forward directly
bool ldp::pre_skip_forward(Uint16 frames_to_skip)
{
	bool result = false;

	// only skip if the LDP is playing
	if (m_status == LDP_PLAYING)
	{
		char s[81] = { 0 };
		Uint32 ms_to_skip = 0;
		Uint32 completed_frames = 0;	// # of frames that have been displayed since we last played
		Uint16 target_frame = (Uint16) (get_current_frame() + frames_to_skip);

		completed_frames = (Uint32) (elapsed_ms_time(m_play_time) / g_game->get_disc_ms_per_frame());
		// find out how many frames we've completed up to this point, yes we want result truncated so we don't add +0.5
		completed_frames++;	// increment by 1 to account for the current frame we're on
		ms_to_skip = (Uint32) ((completed_frames * g_game->get_disc_ms_per_frame())+0.5);
		// compute what to add to our play_timer

		Uint32 cpu_timer = get_cpu_timer();
		m_play_time = m_play_time + ms_to_skip;	// adjust book-keeping
		m_last_frame = target_frame;	// adjust book-keeping

		// if cpu timers were not reset, we can do a perfect re-calculation of the cycles
		if (m_play_time > cpu_timer)
		{
			m_play_cycles = (Uint64) ((m_play_time - cpu_timer) * get_cpu_cycles_per_ms(0));
		}
		// otherwise we'll have to settle for a good approximation (this shouldn't ever happen)
		else
		{
			printline("LDP.CPP approximating m_play_cycles, you should try to avoid this");
			m_play_cycles = m_play_cycles + (Uint64) ((ms_to_skip * get_cpu_cycles_per_ms(0))+0.5);
		}

		result = skip_forward(frames_to_skip, target_frame);

		sprintf(s, "Skipped forward %d frames", frames_to_skip);
		printline(s);
	}
	else
	{
		printline("LDP ERROR: Skip forward command was called when the disc wasn't playing");
	}

	return(result);
}

// prepare to skip forward a certain # of frames backward and continue playing forward
// do not call skip_backward directly
bool ldp::pre_skip_backward(Uint16 frames_to_skip)
{
	bool result = false;

	// only skip if the LDP is playing
	if (m_status == LDP_PLAYING)
	{
		result = skip_backward(frames_to_skip, (Uint16) (get_current_frame() - frames_to_skip));

		char s[81];
		sprintf(s, "Skipped backward %d frames", frames_to_skip);
		printline(s);
	}
	else
	{
		printline("LDP ERROR: Skip backward command was called when the disc wasn't playing");
	}

	return(result);
}


// skips forward a certain number of frames and continues playing
// Do not call this function directly!  Call pre_skip_forward instead
bool ldp::skip_forward(Uint16 frames_to_skip, Uint16 target_frame)
{
	bool result = false;
	
	char frame[6] = {0};
	sprintf(frame, "%05d", target_frame);

	result = pre_search(frame);
	pre_play();

	return(result);
}

// DO NOT CALL THIS FUNCTION DIRECTLY
// call pre_skip_backward instead
bool ldp::skip_backward(Uint16 frames_to_skip, Uint16 target_frame)
{
	bool result = false;
	
	char frame[6] = {0};
	sprintf(frame, "%05d", target_frame);

	result = pre_search(frame);
	pre_play();

	return(result);
}


// prepares to play the disc
void ldp::pre_play()
{
	bool flush = false;	// whether to flush the timers
	Uint32 cpu_hz;	// used to calculate elapsed cycles

	// if the disc is stopped, then spinning up could take quite a long time, so we should flush the timers
	// otherwise we don't want to flush the timers because it will lose accuracy
	if (m_status == LDP_STOPPED)
	{
		flush = true;
	}

	// we only want to refresh the frame calculation stuff if the disc is
	// not playing
	if (m_status != LDP_PLAYING)
	{
		m_play_time = play();
		cpu_hz = get_cpu_hz(0);

		// if the game is using a cpu, then we want to calculate current LDP frame based
		// off of elapsed cpu cycles
		if (cpu_hz)
		{
			Uint32 cpu_timer = get_cpu_timer();

			// aside from when the disc was first played, this should always be true
			// and we need it to be true in order to get an accurate
			// calculation of how many cpu cycles should have elapsed
			if (m_play_time >= cpu_timer)
			{
				m_play_cycles = (Uint64) ((m_play_time - cpu_timer) * get_cpu_cycles_per_ms(0));
			}
			// this should only happen when the disc is first played
			else
			{
				printline("LDP.CPP WARNING : play() less accurate because timers were reset");
				m_play_cycles = 0;
			}
		} // end if we're using cpu cycles to calculate current laserdisc frame
		
		m_status = LDP_PLAYING;
		if (flush)
		{
			printline("LDP : Since disc was stopped, we are flushing the timers (remove me later)");
			flush_cpu_timers();
		}
	}
	else
	{
		printline("LDP : disc is already playing, play command ignored");
	}
	
	printline("Play");	// moved to the end of the function so as to not cause lag before play command could be issued

}

// starts playing the laserdisc
// returns the timer value that indicates when the playback actually started (used to calculate current frame)
unsigned int ldp::play()
{
	return refresh_ms_time();
}

// sets the speed at which we playback at (1 is standard speed)
void ldp::set_play_speed(Uint8 speed)
{
	char s[81] = { 0 };

	sprintf(s, "LDP: Playback speed of %d was requested (ignored)", speed);
	printline(s);
}

// prepares to pause
void ldp::pre_pause()
{
	// only send pause command if disc is playing
	// some games (Super Don) repeatedly flood with a pause command and this doesn't work well with the Hitachi
	if (m_status == LDP_PLAYING)
	{
		m_last_frame = get_current_frame();
		pause();
		m_status = LDP_PAUSED;
		printline("Pause");
	}
	else
	{
		printline("LDP : Received pause while disc was not playing, ignoring");
	}
}

// pauses the disc
void ldp::pause()
{
}

// prepares to stop the disc
// "stop" is defined as going to frame 0 and stopping the motor so that
// the player has to spin up again to begin playing
void ldp::pre_stop()
{
	m_last_frame = 0;
	stop();
	m_status = LDP_STOPPED;
	printline("Stop");
}

// stops the disc
void ldp::stop()
{
}

// returns the current frame number that the disc is on
// this is a generic function which computes the current frame number using the elapsed time
// and the framerate of the disc.  Obviously querying the laserdisc player would be preferable
// if possible (some laserdisc players don't like to be queried too often)
Uint16 ldp::get_current_frame()
{
	Uint16 result = 0;
	Uint32 cpu_hz = get_cpu_hz(0);
	double elapsed_secs = 0.0;	// how many seconds have elapsed since the disc began playing

	// if our game has a cpu, then we use its elapsed cycles to compute the elapsed time.
	// This prevents major problems with games that depend on their cpu/irq's to sync with the laserdisc
	// player, such as Super Don Quix-ote
	if (cpu_hz)
	{
		Uint64 cur_total_cycles = get_total_cycles_executed(0);
		
		// check to make sure flush_cpu_timers has not been called and that cpu is caught up
		if (cur_total_cycles > m_play_cycles)
		{
			// it seems kind of silly to cast to Uint32 then to double, but Uint64 can't go straight to double
			elapsed_secs = ((double)((Uint32)(cur_total_cycles - m_play_cycles))) / cpu_hz;
		}
		// else the CPU is momentarily lagged or else flush_cpu_timers was foolishly called
		// In either event, we have to calculate current frame based on elapsed time for now
		else
		{
			// FIXME : We probably should never flush cpu timers
			// and print a warning here at all times
			elapsed_secs = elapsed_ms_time(m_play_time) * 0.001; // compute elapsed seconds
		}
	}
	// if our game has no cpu (seektest for example) then we cannot compute the current frame based on
	// cpu cycles and must the less safe (but more accurate) method of elapsed time instead
	else
	{
		elapsed_secs = elapsed_ms_time(m_play_time) * 0.001; // compute elapsed seconds
	}

	// if the disc is playing, compute which frame # we're on
	if (m_status == LDP_PLAYING)
	{
		result = (Uint16) (m_last_frame + (Uint16) (elapsed_secs * g_game->get_disc_fps()));
		// do not add 0.5, we do NOT want to round to the nearest number in this case
	}
	// otherwise the disc is idle, so return the frame that we last seeked to
	else
	{
		result = m_last_frame;
	}

	return(result);
}

// steps froward 1 frame
// (I added this for debugging purposes)
void ldp::pre_step_forward()
{
	// this assumes the player is paused
	m_last_frame ++;
}

// returns value of blitting_allowed.  VLDP does not allow blitting!
bool ldp::is_blitting_allowed()
{
	return (blitting_allowed);
}

// returns the current LDP status
int ldp::get_status()
{
	return m_status;
}

// converts an integer frame number into ASCII (with leading zeroes)
void ldp::framenum_to_frame(Uint16 num, char *f)
{
    sprintf(f, "%05d", num);
}

Uint32 ldp::get_search_latency()
{
	return(search_latency);
}

void ldp::set_search_latency(Uint32 value)
{
	search_latency = value;
}

void ldp::set_stop_on_quit(bool value)
{
	m_stop_on_quit = value;
}

// causes LDP to blank video while searching
void ldp::set_search_blanking(bool enabled)
{
	printline("NOTE : Search blanking cannot be modified with this laserdisc player!");
}

// causes LDP to blank video while skipping
void ldp::set_skip_blanking(bool enabled)
{
	printline("NOTE : Skip blanking cannot be modified with this laserdisc player!");
}

void ldp::enable_audio1()
{
	printline("Audio1 enable received (ignored)");
}

void ldp::enable_audio2()
{
	printline("Audio2 enable received (ignored)");
}

void ldp::disable_audio1()
{
	printline("Audio1 disable received (ignored)");
}

void ldp::disable_audio2()
{
	printline("Audio2 disable received (ignored)");
}

// asks LDP to take a screenshot if that's possible
// it's only possible with VLDP as of this time
void ldp::request_screenshot()
{
	printline("NOTE: current laserdisc player does not support taking screenshots, sorry");
}

// returns the width of the laserdisc video (only meaningful with mpeg)
Uint32 ldp::get_discvideo_width()
{
	return m_discvideo_width;
}

// returns the height of the laserdisc video (only meaningful with mpeg)
Uint32 ldp::get_discvideo_height()
{
	return m_discvideo_height;
}

// ordinarily does nothing unless we're VLDP
bool ldp::lock_overlay(Uint32 timeout)
{
	return true;
}

// does nothing unless we're VLDP
bool ldp::unlock_overlay(Uint32 timeout)
{
	return true;
}
