/*
 * ldp.h
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifndef LDP_PRE_H
#define LDP_PRE_H

// different states that the laserdisc player could be in
enum
{
	LDP_ERROR, LDP_BUSY, LDP_STOPPED, LDP_PLAYING, LDP_PAUSED
};

#include <SDL.h>	// needed for datatypes

#define FRAME_SIZE 5	

class ldp
{
public:
	ldp();
	bool init();
	virtual bool init_player();
	void shutdown();
	virtual void shutdown_player();
	void clear();	// clears any received frames
	bool pre_search(char *);
	virtual bool search(char *);
	bool pre_skip_forward(Uint16);
	bool pre_skip_backward(Uint16);

	virtual bool skip_forward(Uint16 frames_to_skip, Uint16 target_frame);
	// NOTE : frames_to_skip and target_frame are both included for your convenience
	// use frames_to_skip only if you cannot use target_frame (which is more accurate)

	virtual bool skip_backward(Uint16 frames_to_skip, Uint16 target_frame);
	// NOTE : frames_to_skip and target_frame are both included for your convenience
	// use frames_to_skip only if you cannot use target_frame (which is more accurate)
	
	void pre_play();
	virtual unsigned int play();
	virtual void set_play_speed(Uint8);	// sets the speed we playback at
	void pre_pause();
	virtual void pause();
	void pre_stop();
	virtual void stop();
	virtual Uint16 get_current_frame();
	void pre_step_forward();
	virtual void enable_audio1();
	virtual void enable_audio2();
	virtual void disable_audio1();
	virtual void disable_audio2();
	virtual void request_screenshot();
	virtual void set_search_blanking(bool enabled);
	virtual void set_skip_blanking(bool enabled);

	bool is_blitting_allowed();	// returns value of blitting_allowed
	int get_status();	// returns status of laserdisc player
	void framenum_to_frame(Uint16, char *);	// converts int to 5-digit string
	Uint32 get_search_latency();
	void set_search_latency(Uint32);
	void set_stop_on_quit(bool);	// enables the stop_on_quit bool flag

	Uint32 get_discvideo_height();	// gets the height of the laserdisc video (only meaningful with mpeg)
	Uint32 get_discvideo_width();	// gets the width of the laserdisc video (only meaningful with mpeg)
	virtual bool lock_overlay(Uint32);	// prevents yuv callback from being called (only meaningful with mpeg)
	virtual bool unlock_overlay(Uint32);

protected:
	bool need_serial;	// whether this LDP driver needs the serial port initialized
	bool serial_initialized; // whether serial has been initialized
	bool player_initialized; // whether the LDP has been properly initialized
	bool blitting_allowed;	// whether it's ok to blit directly to the screen (SMPEG forbids this)
	bool skipping_supported;	// whether the laserdisc player supports skipping
	bool skip_instead_of_search;	// whether we should skip instead of search if searching forward a short distance
	Uint16 max_skippable_frames;	// maximum # of frames that player can skip (if skipping is supported)
	Uint16 m_last_frame;	// the last frame we seeked to
	Uint64 m_play_cycles;	// # of elapsed cpu cycles from when we last issued a play command
	Uint32 m_play_time;	// current time when we last issued a play command
	int m_status;	// the current status of the laserdisc player
	Uint32 search_latency;	// how many ms to stall before searching (to simulate slow laserdisc players)
	bool m_stop_on_quit;	// should the LDP stop when it quits?
	Uint32 m_discvideo_width;	// width of laserdisc video (only meaningful with mpeg)
	Uint32 m_discvideo_height;	// height of laserdisc video (only meaningful with mpeg)
};

extern ldp *g_ldp;	// our global ldp class.  Defined here so that every .cpp file doesn't have to define it.

#endif
