/*
 * sound.cpp
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// DAPHNE
// The sound code in this file uses SDL

#include <stdio.h>
#include <stdlib.h>

#include "SDL.h"
#include "SDL_audio.h"
#include "SDL_mixer.h"
#include "sound.h"
#include "../io/conout.h"
#include "../game/game.h"
#include "../daphne.h"
#include "../ldp-out/ldp-vldp.h" // added by JFA for -startsilent

//#define BUF_SIZE	2048	

Mix_Chunk *mixwave[MAX_NUM_SOUNDS] = { NULL };
Mix_Chunk *mixwave_saveme = NULL;	// the special saveme wav which is loaded independently of any game

bool g_sound_enabled = true;	// whether sound is enabled

Uint16 buf_size = 2048;  // Matt prefers 1024 but some people (cough Warren) can't handle it haha
						// but now it can be changed from the command line

int cur_wave = 0;	// the current wave being played (0 to NUM_DL_BEEPS-1)
bool g_sound_initialized = false;	// whether the sound will work if we try to play it

// added by JFA for -startsilent
void reopen_sound()
{
	// only proceed if sound has been initialized
	if (g_sound_initialized)
	{
		sound_volume(0, 64);	// disable volume for both channels
		sound_volume(1, 64);

		ldp_vldp *cur_ldp = dynamic_cast<ldp_vldp *>(g_ldp);	// see if the currently selected LDP is VLDP
		if (cur_ldp)
		{
			cur_ldp->enable_audio1();
			cur_ldp->enable_audio2();
		}
	}
}
// end edit

void set_soundbuf_size(Uint16 newbufsize)
{
	buf_size = newbufsize;
}

bool sound_init()
// returns a true on success, false on failure
{

	bool result = false;
	int audio_rate = 44100; // rate to mix audio at.  This cannot be changed without resampling all .wav's and all .ogg's

	Uint16 audio_format = AUDIO_S16;
	int audio_channels = 2; // 2 channels

	printline("Initializing sound system ... ");

	// if the user has not disabled sound from the command line
	if (is_sound_enabled())
	{
	  // if SDL audio initialization was successful
	  if (SDL_InitSubSystem(SDL_INIT_AUDIO) >= 0)
	  {
		// if we can open the audio device
		if (Mix_OpenAudio(audio_rate, audio_format, audio_channels, buf_size) >= 0)
		{
			// if we can load all our waves, we're set
			if (load_waves())
			{
				if (get_startsilent())
				{ // added by JFA for -startsilent
					sound_volume(0, 0);	// disable volume for both channels
					sound_volume(1, 0);

					ldp_vldp *cur_ldp = dynamic_cast<ldp_vldp *>(g_ldp);	// see if the currently selected LDP is VLDP
					if (cur_ldp)
					{
						cur_ldp->disable_audio1();
						cur_ldp->disable_audio2();
					}
				}
				else {
					sound_volume(0, 64);	// set max volume for both channels
					sound_volume(1, 64);
				}
				result = true;
				g_sound_initialized = true;
			}
		} // end if audio device could be opened ...
		
		// if audio device could not be opened (ie no sound card)
		else
		{
			printline("WARNING: Audio device could not be opened, DAPHNE will run without sound");
			g_sound_enabled = false;
		}
	  } // end if sound initializtion worked	  
	} // end if sound is enabled

	// if sound isn't enabled, then we act is if sound initialization worked so daphne doesn't quit
	if (!is_sound_enabled())
	{
		return true;
	}

	return(result);

}

// shuts down the sound subsystem
void sound_shutdown()
{
	// shutdown sound only if we previously initialized it
	if (g_sound_initialized)
	{
		printline("Shutting down sound system...");
		Mix_CloseAudio();
		free_waves();
		g_sound_initialized = 0;
		SDL_QuitSubSystem(SDL_INIT_AUDIO);
	}
}

bool sound_play(Uint32 whichone)
// Plays one of our DL beeps
// 1 on success, 0 on failure
{

	bool result = false;

	// only play a sound if sound has been initialized (and if whichone points to a valid wav)
	if (is_sound_enabled() && (whichone < g_game->get_num_sounds()))
	{
//		Mix_PlayChannel(1, mixwave[whichone], 0);   //all sounds locked to single channel - #1
		Mix_PlayChannel(-1, mixwave[whichone], 0);  // plays multiple sounds, each on first available channel
		result = true;
	}

	return(result);

}

// plays the 'saveme' sound
bool sound_play_saveme()
{
	bool result = false;

	if (is_sound_enabled())
	{
		Mix_PlayChannel(1, mixwave_saveme, 0);
		result = true;
	}

	return result;
}

// sets the volume for channel 0 or 1
void sound_volume(Uint8 which_channel, Uint8 volume)
{
	Mix_Volume(which_channel, volume);
}

// loads the wave files into the wave structure
// returns 0 if failure, or non-zero if success
int load_waves()
{

	Uint32 i = 0;
	int result = 1;
	char filename[81] = { 0 };

	for (; i < g_game->get_num_sounds(); i++)
	{
		sprintf(filename, "sound/%s", g_game->get_sound_name(i));

		// if loading of the wave failed...
		mixwave[i] = Mix_LoadWAV(filename);
		if (!mixwave[i])
		{
			fprintf(stderr, "Can't find file %s\n", filename);
			result = 0;
			free_waves();	// De-allocate any waves we've allocated thus far
			break;
		}
	}

	// load "saveme" sound in
	mixwave_saveme = Mix_LoadWAV("sound/saveme.wav");
	if (!mixwave_saveme)
	{
		result = 0;
	}

	return(result);

}

// free all allocated waves
void free_waves()
{

	Uint32 i = 0;

	for (; i < g_game->get_num_sounds(); i++)
	{
		// don't de-allocate a sound if it hasn't been allocated
		if (mixwave[i])
		{
			Mix_FreeChunk(mixwave[i]);
		}
	}
}

int get_sound_initialized()
{
	return(g_sound_initialized);
}

void set_sound_enabled_status (bool value)
{
	g_sound_enabled = value;
}

bool is_sound_enabled()
{
	return g_sound_enabled;
}
