/*
 * ssi263.cpp
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// SSI263.CPP
// by Matt Ownby

// This is a _PARTIAL_ SSI263 speech chip emulator designed in order to emulate the speech used
// in the arcade game Thayer's Quest.
// At the time of this writing, we have not been able to find much documentation on how to program
// this chip, so we have had to make certain assumptions which likely won't hold true for other
// applications that used the SSI263.

#include "ssi263.h"

#ifdef SSI_DEBUG
#include <stdio.h>
#include "../io/conout.h"
#endif

// This code attempts to emulate the SSI263 speech chip used in the game Thayer's Quest

bool g_ssi_active = false;	// whether we are actively requesting phonemes or not
bool g_ssi_control = false;	// whether we are controlling the SSI or sending data to it

// Duration/Phoneme
// Working theory: top 2 bits are for duration, the rest is for the phoneme
// a duration of 0x0 is the slowest, 0x3 (both bits set) is the fastest
void ssi263_reg0(unsigned char value)
{
#ifdef SSI_DEBUG
	char s[81] = { 0 };
#endif

	// if data receive to this register is for controlling the SSI263
	if (g_ssi_control)
	{
		// as far as I can tell, C0 starts the speech chip requesting phonemes in control mode
		if (value == 0xC0)
		{
#ifdef SSI_DEBUG
			printline("SSI263 enabled");
#endif
			g_ssi_active = true;
		}
		// as far as I can tell, 0 stops the speech chip requesting phonemes in control mode
		else if (value == 0)
		{
#ifdef SSI_DEBUG
			printline("SSI263 disabled");
#endif
			g_ssi_active = false;
		}
	}

	// if we are receiving phoneme/duration data
	else
	{
#ifdef SSI_DEBUG
		unsigned char duration = static_cast<unsigned char>((value & 0xC0) >> 6);
#endif
		unsigned char phoneme = static_cast<unsigned char>(value & 0x3F);

		switch (phoneme)
		{
			// some form of pause
		case 0:
	#ifdef SSI_DEBUG
			sprintf(s, "SSI263 paused with a duration of %x", duration);
			printline(s);
	#endif
			break;
		default:
	#ifdef SSI_DEBUG
			sprintf(s, "Unsupported value %x received: phoneme %x, duration %x", value, phoneme, duration);
			printline(s);
	#endif
			break;
		}
	}
}

// INFLECT
void ssi263_reg1(unsigned char value)
{
#ifdef SSI_DEBUG
	char s[81] = { 0 };
#endif

	switch (value)
	{
	case 0x46:
	case 0x5E:
	case 0x76:
	case 0x8E:
#ifdef SSI_DEBUG
//		sprintf(s, "Inflection byte received in reg 1: %x", value);
//		printline(s);
#endif
		break;
	default:
#ifdef SSI_DEBUG
		sprintf(s, "Unknown inflection byte: %x", value);
		printline(s);
#endif
		break;
	}
}

// Speech Rate
void ssi263_reg2(unsigned char value)
{
	switch (value)
	{
	case 0x98:
#ifdef SSI_DEBUG
//		printline("Speech rate set to low");
#endif
		break;
	case 0xA8:
#ifdef SSI_DEBUG
//		printline("Speech rate set to high");
#endif
		break;
	default:
#ifdef SSI_DEBUG
		char s[81] = { 0 };
		sprintf(s, "Unknown speech rate byte sent to reg 2: %x", value);
		printline(s);
#endif
		break;
	}
}

// CTTRAMP
void ssi263_reg3(unsigned char value)
{
	// if high bit is set, I believe that means control is being enabled
	if (value & 0x80)
	{
		g_ssi_control = true;
#ifdef SSI_DEBUG
//		printline("Control bit in register 3 is being raised");
#endif
	}
	else
	{
		g_ssi_control = false;

		switch (value)
		{
		case 0x6C:
		case 0x6F:
			break;
		default:
#ifdef SSI_DEBUG
			char s[81] = { 0 };
			sprintf(s, "Unsupported amplitude code received: %x", value);
			printline(s);
#endif
			break;
		}
	}
}

// Filter Frequency
void ssi263_reg4(unsigned char value)
{
	switch (value)
	{
	case 0xE6:
#ifdef SSI_DEBUG
//		printline("Filter frequency set to low");
#endif
		break;
	case 0xE7:
#ifdef SSI_DEBUG
//		printline("Filter frequency set to high");
#endif
		break;
	default:
#ifdef SSI_DEBUG
		printline("Unknown filter frequency value");
#endif
		break;
	}
}

// returns true if the SSI263 is requesting a phoneme
// This should be checked frequently and if true, a phoneme should be supplied ASAP
bool ssi263_phoneme_requested()
{
	return g_ssi_active;
}

