/*
 * palette.cpp
 *
 * Copyright (C) 2002 Mark Broadhead
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "../game/game.h"
#include "../io/conout.h" // for printline
#include "palette.h"
#include "rgb2yuv.h"

int g_palette_size = 0;
SDL_Color *g_rgb_palette = NULL;
t_yuv_color *g_yuv_palette = NULL;

bool g_palette_modified = true;
SDL_Color g_transparent_color;
int g_transparent_number = -1;


// call this function once to set size of game palette
bool palette_initialize (int num_colors)
{
	bool result = true;

	// set the transparent color to black (default)
	g_transparent_color.r = 0;
	g_transparent_color.g = 0;
	g_transparent_color.b = 0;

	g_palette_size = num_colors;
	// we can only have 256 max since all our surfaces are 8-bit
	if (num_colors > 256)
	{
		printline("palette_initialize error: Too many colors > 256!");
		result = false;
	}

	if (result)
	{
		g_rgb_palette = new SDL_Color[num_colors];
		g_yuv_palette = new t_yuv_color[num_colors];
	}

	if (!((g_rgb_palette) && (g_yuv_palette)
		))
	{
		printline("palette_initialize error: Could not allocate palette arrays!");
		palette_shutdown();
		result = false;
	}
	else
	{
		// set all colors to unmodified and black
		for (int x = 0; x < g_palette_size; x++)
		{
			SDL_Color temp_color;
			temp_color.r = 0;
			temp_color.g = 0;
			temp_color.b = 0;

			palette_set_color(x, temp_color);
		}
	}

	return result;
}

// call this before you call palette_set_color if you need to change the transparent color value from the default black
void palette_set_transparent_color(SDL_Color color_value)
{
	g_transparent_color = color_value;
}

// some games use a specific color number, instead of color value to be transparent (bega's battle, gp world)
// call this before you call palette_set_color to set which color number is transparent
void palette_set_transparent_number(int color_number)
{
	if (color_number < g_palette_size)
	{
		g_transparent_number = color_number;
	}
	else
	{
		printline("error: palette_set_transparent_number out of range");
	}
}

// call this function when a color has changed
void palette_set_color (int color_num, SDL_Color color_value)
{
	// put the rgb color value into our array
	if (color_num < g_palette_size)
	{
		// are we using a transparent_number instead of value?
		if (g_transparent_number == -1)
		{
			// no, so check if this color is transparent
			if ((g_transparent_color.r == color_value.r) && (g_transparent_color.g == color_value.g) && (g_transparent_color.b == color_value.b))
			{
				g_yuv_palette[color_num].transparent = true;
			}
			else
			{
				g_yuv_palette[color_num].transparent = false;
			}
		}
		else
		{
			if (color_num == g_transparent_number)
			{
				g_yuv_palette[color_num].transparent = true;
			}
			else
			{
				g_yuv_palette[color_num].transparent = false;
			}
		}

		// make sure the color has really been modified because the RGB2YUV calculations are expensive
		if (!((g_rgb_palette[color_num].r == color_value.r) && (g_rgb_palette[color_num].g == color_value.g) && (g_rgb_palette[color_num].b == color_value.b)))
		{
			g_rgb_palette[color_num] = color_value;
			g_palette_modified = true;	

			// MATT : seems to make more sense to calculate the YUV value of the color here
			rgb2yuv_input[0] = g_rgb_palette[color_num].r;
			rgb2yuv_input[1] = g_rgb_palette[color_num].g;
			rgb2yuv_input[2] = g_rgb_palette[color_num].b;
			rgb2yuv();
			g_yuv_palette[color_num].y = rgb2yuv_result_y;
			g_yuv_palette[color_num].v = rgb2yuv_result_v;
			g_yuv_palette[color_num].u = rgb2yuv_result_u;
		}
	}
	else
	{
		printline("palette_set_color error: color number out of bounds!");
	}
}

// call this function right before drawing the current overlay
void palette_finalize()
{
	if (g_palette_modified)
	{
		// update color palette for all the surfaces that we are using
		for (int i = 0;; i++)
		{
			SDL_Surface *video_overlay = g_game->get_video_overlay(i);

			// if we have a video overlay to set the colors no ...
			if (video_overlay)
			{
				SDL_SetColors(video_overlay, g_rgb_palette, 0, g_palette_size);
			}
			else
			{
				break;
			}
		}
	}

	g_palette_modified = false;
}

// call this function on video shutdown
void palette_shutdown (void)
{
	if (g_rgb_palette)
	{
		delete [] g_rgb_palette;
		g_rgb_palette = NULL;
	}
	if (g_yuv_palette)
	{
		delete [] g_yuv_palette;
		g_yuv_palette = NULL;
	}

}

// this function is here temporarily while the game drivers are switch to this new method
t_yuv_color *get_yuv_palette(void)
{
	return g_yuv_palette;
}

