/*
 * scoreboard.cpp
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


// Handles the scoreboard for Dragon's Lair / Space Ace

#include <stdio.h>
#include <stdlib.h>
#include "../daphne.h"
#include "video.h"
#include "../io/parallel.h"
#include "../io/conout.h"
#include "../timer/timer.h"
#include "../io/error.h"
#include "../game/game.h"

// Arrays that hold the current value of the LED's so we don't display the same number over and over again
// (Dragon's Lair will constantly refresh the credits LED's, this just wastes CPU power to display it constantly)
// This is also used when the LED's need to be redrawn

int player1_score_values[6] = { 0xf, 0xf, 0xf, 0xf, 0xf, 0xf };	// 0xF is a blank LED
int player1_lives_value = 0xf;
int player2_score_values[6] = { 0xf, 0xf, 0xf, 0xf, 0xf, 0xf };
int player2_lives_value = 0xf;
int credits_values[2] = { 0xf, 0xf };

// Whether or not to overlay scoreboard onto video (DL/SA with vldp only).
bool show_overlay_scoreboard = false;

unsigned char rsb_value = 0;	// real scoreboard value

// Graphical coordinates of where all these LED's should go

#define LED_WIDTH 40
#define LED_HEIGHT 53
#define TITLE_HEIGHT 20

const int player_title_x = 217;
const int player_score_x = 200;
const int player1_score_y = 73;
const int player1_title_y = player1_score_y - 27;
const int player1_lives_y = player1_score_y + LED_HEIGHT + 14;
const int player_lives_x = 320 - LED_WIDTH + 15;
const int player1_lives_title_y = player1_lives_y + (LED_HEIGHT/2) - (TITLE_HEIGHT/2);	// centered with LED
const int player_lives_title_x = player_lives_x + LED_WIDTH + 21;
const int player2_score_y = player1_lives_y + LED_HEIGHT + 45;
const int player2_title_y = player2_score_y - 27;
const int player2_lives_y = player2_score_y + LED_HEIGHT + 14;
const int player2_lives_title_y = player2_lives_y + (LED_HEIGHT/2) - (TITLE_HEIGHT/2);
const int credits_x = player_lives_x - LED_WIDTH;
const int credits_y = player2_lives_y + LED_HEIGHT + 23;
const int credits_title_y = credits_y + 18;

// Use video overlay to display score if this option enabled.
void update_overlay_player_score (int player, int start_digit, int values[], int num_digits)
{
    int result = 0;
    SDL_Surface *overlay;

    if (NULL != (overlay = g_game->get_active_video_overlay()))
    {
        int x = start_digit * OVERLAY_LED_WIDTH;

        // Player1 position is static, but Player2 will be affected
        // by different MPEG widths (640x480, 720x480 known so far).
        x += (player == 0 ? 65 : overlay->w - 7 * OVERLAY_LED_WIDTH);

        draw_overlay_leds(values, num_digits, x, 0, overlay);
    }
}

// digit = which LED to update (range 0-5)
// LED 0 is the left-most LED
// value = which value to put there (range 0-F)
// returns 1 on success, 0 on failure
// player = which player, 0 is player 1, anything else is player 2
int update_player_score(int digit, int value, int player)
{
	int success = 0;
	int coord_x = 0;
	int coord_y = 0;
	int already_displayed = 1;	// whether the number is already displayed on the scoreboard
	int rsb_port = get_scoreboard_port();

	// make sure all parameters are within the proper range and we aren't in debug mode
	if (((digit >=0) && (digit <= 5))
		&& ((value >=0) && (value <= 0x0F)))
	{
		coord_x = (digit * LED_WIDTH) + player_score_x;

		if (player == 0)
		{
			// if number is not already displayed
			if (player1_score_values[digit] != value)
			{
				coord_y = player1_score_y;
				player1_score_values[digit] = value;

    		    already_displayed = 0;  // Update scoreboard (virtual or overlay).

				// if we are driving a real scoreboard
				if (get_scoreboard())
				{
					rsb_value = (unsigned char) ((digit << 4) | value);
					par_base2(rsb_port, 0);
					par_base0(rsb_port, rsb_value);
					par_base2(rsb_port, 2);
				}
			}
		}
		else
		{
			// if number is not already displayed
			if (player2_score_values[digit] != value)
			{
				coord_y = player2_score_y;
				
				if (g_game->get_game_type() == GAME_SAE) //needs special handler
				{
					//check for continued cycling H/P (this makes an 'A' for SAE)
					if ((player2_score_values[digit] == 0x10) && ((value == 0x0C) || (value==0x0E)))  
					{
						already_displayed = 1;
						//leave player2_score_values[digit] at 0x10
					}
					//check for new cycling H/P
					else if (((value == 0x0C) && (player2_score_values[digit] == 0x0E)) 
						|| ((value == 0x0E) && (player2_score_values[digit] == 0x0C)))
					{
						player2_score_values[digit] = 0x10; //'A' for SAE
						value = 0x10;
					    already_displayed = 0;  // Update scoreboard (virtual or overlay).
					}
					//normal write
					else
					{
						player2_score_values[digit] = value;
						already_displayed = 0;
					}
				}
				else  //normal stuff for everything but SAE
				{
					player2_score_values[digit] = value;
			        already_displayed = 0;  // Update scoreboard (virtual or overlay).
				}


				// if we are driving a real scoreboard
				if (get_scoreboard())
				{
					rsb_value = (unsigned char) ((digit << 4) | value);
					par_base2(rsb_port, 0);
					par_base0(rsb_port, rsb_value);
					par_base2(rsb_port, 8);
				}
			}
		}

		// if the number we want to draw is not already being displayed
		if (! already_displayed)
		{
            if (show_overlay_scoreboard)
            {
                update_overlay_player_score (player, digit, &value, 1);
                success++;
            }
            else
			    success = draw_led(value, coord_x, coord_y);
		}
	}
	else
	{
    	// The parameters are out of range
		char s[81] = { 0 };

		sprintf(s, "SCOREBOARD: Unsupported write to scoreboard: Digit %x Value %x ", digit, value);
		printline(s);
	}

	return(success);
}

// Use video overlay to display lives left if this option enabled
void update_overlay_player_lives (int player, int lives)
{
    SDL_Surface *overlay;

    if (NULL != (overlay = g_game->get_active_video_overlay()))
    {
        // Value of lives was validated in caller, so charge right ahead.
        draw_overlay_leds(&lives, 1,
                          player == 0 ? 48 : overlay->w - 2 * OVERLAY_LED_WIDTH,
                          OVERLAY_LED_HEIGHT, overlay);
    }
}

// value = which value to put in the single LED (range 0-F)
// player = which player, 0 is player 1, anything else is player 2
// returns 1 on success
int update_player_lives(int value, int player)
{
	int success = 0;
	int coord_x = 0, coord_y = 0;
	int already_displayed = 1;
	int rsb_port = get_scoreboard_port();

	if ((value >= 0) && (value <= 0x0F))
	{
		coord_x = player_lives_x;

		if (player == 0)
		{
			if (player1_lives_value != value)
			{
				player1_lives_value = value;

                if (show_overlay_scoreboard)
                    update_overlay_player_lives (player, player1_lives_value);
                else
                {
    				coord_y = player1_lives_y;
				    already_displayed = 0;  // Update virtual scoreboard?
                }

                if (get_scoreboard())
				{
					rsb_value=6; //p1 lives
					rsb_value=(unsigned char) (rsb_value<<4); //shift to address space
					rsb_value |= value; //turn on digit value
					par_base2(rsb_port, 0);
					par_base0(rsb_port, rsb_value);
					par_base2(rsb_port, 2);
				}
			}
		}
		else
		{
			if (player2_lives_value != value)
			{
				player2_lives_value = value;

                if (show_overlay_scoreboard)
                    update_overlay_player_lives (player, player2_lives_value);
                else
                {
    				coord_y = player2_lives_y;
				    already_displayed = 0;  // Update virtual scoreboard.
                }

				if (get_scoreboard())
				{
					rsb_value = 7; //p2 lives
					rsb_value = (unsigned char) (rsb_value << 4); //shift to address space
					rsb_value |= value; //turn on digit value
					par_base2(rsb_port, 0);
					par_base0(rsb_port, rsb_value);
					par_base2(rsb_port, 2);
				}
			}
		}

		if (! already_displayed)
		{
			success = draw_led(value, coord_x, coord_y);
		}
	}

	return (success);
}

// Use video overlay to display credits left if this option enabled.
void update_overlay_credits()
{
    SDL_Surface *overlay;

    if (NULL != (overlay = g_game->get_active_video_overlay()))
    {
        int fudge;

        // need to shift a bit to look exactly centered
        if (GAME_THAYERS == g_game->get_game_type())
            fudge = (overlay->w == 360 ? 4 : 3);
        else
            fudge = (overlay->w == 360 ? 2 : 0);


        draw_overlay_leds(credits_values, 2,
                          overlay->w / 2 - (OVERLAY_LED_WIDTH + fudge),
                          OVERLAY_LED_HEIGHT, overlay);
    }
}

// digit = which LED to update (0 or 1)
// value = which value to put there (range 0-F)
// returns 1 on success
int update_credits(int digit, int value)
{
	int success = 0;
	int coord_x = 0;
	int rsb_port = get_scoreboard_port();

	if (((digit >= 0) && (digit <= 1))
		&& ((value >= 0) && (value <= 0x0F)))
	{
		// if the number to be displayed is not already displayed
		if (credits_values[digit] != value)
		{
			credits_values[digit] = value;

            if (show_overlay_scoreboard)
                update_overlay_credits();
            else
            {
    			coord_x = (LED_WIDTH * digit) + credits_x;
			    success = draw_led(value, coord_x, credits_y);
            }

			if (get_scoreboard())
			{
				if (digit == 0)
				{
					rsb_value = 6;
				}
				else
				{
					rsb_value = 7;
				}

				rsb_value = (unsigned char) (rsb_value << 4);
				rsb_value |= value;
				par_base2(rsb_port, 0);
				par_base0(rsb_port, rsb_value);
				par_base2(rsb_port, 8);
			} // end real scoreboard
		}
	}

	return (success);
}

// Clear or redraw the overlay scoreboard (only with vldp and -useoverlaysb
// command line switch). Not too efficient, but won't be called often unless
// user switches the overlay on/off a lot...
void draw_overlay_scoreboard(SDL_Surface *overlay, bool show)
{
    if (show)
    {
        // Draw all overlay scoreboard data. If the console font width
        // ever changes, this will break.
        if (GAME_THAYERS == g_game->get_game_type())
        {
            // Thayer's Quest only uses "Credits" portion of the DL/SA
            // scoreboard.
            draw_string("Time", overlay->w / 12 - 2, 0, overlay);

            // Refresh the time units value.
            update_overlay_credits();
        }
        else
        {
            // Draw all DL/SA scoreboard labels.
            // Credits Label
            draw_string("Credits", overlay->w / 12 - (overlay->w == 360 ? 4 : 3), 0, overlay);

            // Player labels.
            draw_string("Player 1: ", 1, 0, overlay);
            draw_string("Player 2: ", overlay->w / 6 - 19, 0, overlay);

            // Lives labels.
            draw_string("Lives: ", 1, 1, overlay);
            draw_string("Lives: ", overlay->w / 6 - 10, 1, overlay);

            // Update all data.
            update_overlay_credits();

            // Player 1.
            update_overlay_player_score(0, 0, player1_score_values, 6);
            update_overlay_player_lives(0, player1_lives_value);

            // Player 2.
            update_overlay_player_score(1, 0, player2_score_values, 6);
            update_overlay_player_lives(1, player2_lives_value);
        }
    }
    else
    {
        SDL_Rect dest = {0};

        // Erase overlay scoreboard.
        dest.w = overlay->w;
        dest.h = OVERLAY_LED_HEIGHT * 2;

	    SDL_FillRect(overlay, &dest, 0);
	    SDL_UpdateRects(overlay, 1, &dest);	
    }
}

// repaints the scoreboard including all of the words such as "PLAYER 1 SCORE"
// returns 1 if successful, 0 if failed
void scoreboard_repaint()
{
	int digit_index = 0;
	int value1 = 0, value2 = 0;	// values to be displayed in each LED

	draw_othergfx(B_DL_PLAYER1, player_title_x, player1_title_y);
	draw_othergfx(B_DL_PLAYER2, player_title_x, player2_title_y);
	draw_othergfx(B_DL_LIVES, player_lives_title_x, player1_lives_title_y);
	draw_othergfx(B_DL_LIVES, player_lives_title_x, player2_lives_title_y);
	draw_othergfx(B_DL_CREDITS, player_lives_title_x, credits_title_y);

	// Redraw each player's score
	for (digit_index = 0; digit_index < 6; digit_index ++)
	{
		value1 = player1_score_values[digit_index];	// grab the value before we alter it
		value2 = player2_score_values[digit_index];
		player1_score_values[digit_index] ^= 0x0F;	// arbitrary number that forces a redraw because it is different
		player2_score_values[digit_index] ^= 0x0F;
		update_player_score(digit_index, value1, 0);
		update_player_score(digit_index, value2, 1);
	}

	// Redraw each player's lives
	value1 = player1_lives_value;
	value2 = player2_lives_value;
	player1_lives_value ^= 0x0F;	// arbitrary number that forces a redraw because it is different
	player2_lives_value ^= 0x0F;
	update_player_lives(value1, 0);
	update_player_lives(value2, 1);

	// Redraw the credits
	value1 = credits_values[0];
	value2 = credits_values[1];
	credits_values[0] ^= 0x0F;	// arbitrary number that forces a redraw because it is different
	credits_values[1] ^= 0x0F;
	update_credits(0, value1);
	update_credits(1, value2);
}

// initializes the scoreboard, including
//  setting up parallel port, if need be
// returns 1 if successful or 0 if not successful
bool scoreboard_init()
{
	bool result = true;

	// for now we're just doing a quick hack!
	if (get_scoreboard())
	{
		result = par_init(get_scoreboard_port());
	}

    if (! show_overlay_scoreboard)
	    scoreboard_repaint();

	if (!result)
	{
		printerror("Scoreboard initialization failed!");
	}

	return(result);
}

void scoreboard_shutdown()
{

	if (get_scoreboard())
	{
		par_close(get_scoreboard_port());
	}

}

void set_overlay_scoreboard(bool state)
{
    show_overlay_scoreboard = state;
}

bool get_overlay_scoreboard()
{
    return show_overlay_scoreboard;
}

int convert_scoreboard(char letter)
{
	if (letter=='D' || letter=='d')
		return 10;

	if (letter=='E' || letter=='e')
		return 11;

	if (letter=='H' || letter=='h')
		return 12;

	if (letter=='L' || letter=='l')
		return 13;

	if (letter=='P' || letter=='p')
		return 14;

	if (letter=='S' || letter=='s')
		return 15;
	
	return 15;
}

void send_scoreboard_text(char sbtext[])
{
	int result=0, sbnum=0, i=0, x=0;

	// update player 1 score
	for (i=0 ; i < 7; i++)
	{
		sbnum = sbtext[i] - '0';

		if ((sbnum >= 0) && (sbnum <= 10))
		{
    		result = update_player_score(i, sbnum, 0);
		}
		else
		{
    		result = update_player_score(i, convert_scoreboard(sbtext[i]), 0);
		}
	}

	// update player 1 lives

	sbnum = sbtext[6] - '0';

	if ((sbnum >= 0) && (sbnum <= 10))
		{
		result = update_player_lives(sbnum, 0);
		}
		else
		{
		result = update_player_lives(convert_scoreboard(sbtext[6]), 0);
		}

	x = 0;

	// update player 2 score

	for (i=7 ; i < 13; i++)
	{
		sbnum = sbtext[i] - '0';

		if ((sbnum >= 0) && (sbnum <= 10))
		{
		result = update_player_score(x, sbnum, 1);
		}
		else
		{
		result = update_player_score(x, convert_scoreboard(sbtext[i]), 1);
		}
	x++;
	}

	// update player 2 lives

	sbnum = sbtext[13] - '0';

	if ((sbnum >= 0) && (sbnum <= 10))
		{
		result = update_player_lives(sbnum, 1);
		}
		else
		{
		result = update_player_lives(convert_scoreboard(sbtext[13]), 1);
		}

	x = 0;

	// update credits

	for (i=14 ; i < 17; i++)
	{
		sbnum = sbtext[i] - '0';

		if ((sbnum >= 0) && (sbnum <= 10))
		{
		result = update_credits(x, sbnum);
		}
		else
		{
		result = update_credits(x, convert_scoreboard(sbtext[i]));
		}
	x++;
	}
}
