/*
 * video.cpp
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of DAPHNE, a laserdisc arcade game emulator
 *
 * DAPHNE is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DAPHNE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// video.cpp
// Part of the DAPHNE emulator
// This code started by Matt Ownby, May 2000

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "video.h"
#include "SDL_DrawText.h"
#include "../daphne.h"
#include "scoreboard.h"
#include "../io/conout.h"
#include "../io/error.h"
#include "SDL_Console.h"
#include "../game/game.h"
#include "../ldp-out/ldp.h"

Uint16 g_vid_width = 640, g_vid_height = 480;	// default video width and video height
SDL_Surface *g_led_bmps[LED_RANGE] = { 0 };
SDL_Surface *g_other_bmps[B_EMPTY] = { 0 };
SDL_Surface *g_screen = 0;	// the window that we initialize to hold the bmp's
bool g_console_initialized = false;	// 1 once console is initialized
bool g_fullscreen = false;	// whether we should initialize video in fullscreen mode or not
const Uint16 cg_normalwidths[] = { 640, 800, 1024, 1280, 1280, 1600 };
const Uint16 cg_normalheights[]= { 480, 600, 768, 960, 1024, 1200 };

////////////////////////////////////////////////////////////////////////////////////////////////////

// initializes the window in which we will draw our BMP's
// returns true if successful, false if failure
bool init_display()
{

	bool result = false;	// whether video initialization is successful or not
	bool abnormalscreensize = true; // assume abnormal
	const SDL_VideoInfo *vidinfo = NULL;
	Uint8 suggested_bpp = 0;
	Uint32 sdl_flags = 0;	// the default for this depends on whether we are using HW accelerated YUV overlays or not

	char *hw_env = getenv("SDL_VIDEO_YUV_HWACCEL");

	// if HW acceleration has been disabled, we need to use a SW surface due to some oddities with crashing and fullscreen
	if (hw_env && (hw_env[0] == '0'))
	{
		sdl_flags = SDL_SWSURFACE;
	}

	// else if HW acceleration hasn't been explicitely disabled ...
	else
	{

		sdl_flags = SDL_HWSURFACE;
		// Win32 notes (may not apply to linux) :
		// After digging around in the SDL source code, I've discovered a few things ...
		// When using fullscreen mode, you should always use SDL_HWSURFACE because otherwise
		// you can't use YUV hardware overlays due to SDL creating an extra surface without
		// your consent (which seems retarded to me hehe).
		// SDL_SWSURFACE + hw accelerated YUV overlays will work in windowed mode, but might
		// as well just use SDL_HWSURFACE for everything.
	}

	char s[250] = { 0 };
	Uint32 x = 0;	// temporary index

	// if we were able to initialize the video properly
	if ( SDL_InitSubSystem(SDL_INIT_VIDEO) >=0 )
	{

		vidinfo = SDL_GetVideoInfo();
		suggested_bpp = vidinfo->vfmt->BitsPerPixel;

		// if we were in 8-bit mode, try to get at least 16-bit color
		// because we definitely use more than 256 colors in daphne
		if (suggested_bpp < 16)
		{
			suggested_bpp = 32;
		}

		if (g_fullscreen)
		{
			SDL_ShowCursor(SDL_DISABLE);	// hide mouse in fullscreen mode
			sdl_flags |= SDL_FULLSCREEN;
		}

		// go through each standard resolution size to see if we are using a standard resolution
		for (x=0; x < (sizeof(cg_normalwidths) / sizeof(Uint16)); x++)
		{
			// if we find a match, we know we're using a standard res
			if ((g_vid_width == cg_normalwidths[x]) && (g_vid_height == cg_normalheights[x]))
			{
				abnormalscreensize = false;
			}
		}

		// if we're using a non-standard resolution
		if (abnormalscreensize)
		{
			printline("WARNING : You have specified an abnormal screen resolution! Normal screen resolutions are:");
			// print each standard resolution
			for (x=0; x < (sizeof(cg_normalwidths) / sizeof(Uint16)); x++)
			{	
				sprintf(s,"%d x %d", cg_normalwidths[x], cg_normalheights[x]);
				printline(s);
			}
			newline();
		}

		g_screen = SDL_SetVideoMode(g_vid_width, g_vid_height, suggested_bpp, sdl_flags);

		if (g_screen)
		{
			sprintf(s, "Set %dx%d at %d bpp with flags: %x", g_screen->w, g_screen->h, g_screen->format->BitsPerPixel, g_screen->flags);
			printline(s);
			SDL_WM_SetCaption("DAPHNE: First Ever Multiple Arcade Laserdisc Emulator =]", "daphne");
			
			// initialize SDL console in the background
			if (ConsoleInit("pics/ConsoleFont.bmp", g_screen, 100)==0)
			{
				AddCommand(g_cpu_break, "break");
				g_console_initialized = true;
				result = true;

			}
			else
			{
				printerror("Error initializing SDL console =(");			
			}
		}
	}

	if (result == 0)
	{
		sprintf(s, "Could not initialize video display: %s", SDL_GetError());
		printerror(s);
	}

	return(result);

}

// shuts down video display
void shutdown_display()
{

	printline("Shutting down video display...");

	if (g_console_initialized)
	{
		ConsoleShutdown();
		g_console_initialized = false;
	}

	g_game->video_shutdown();

	SDL_QuitSubSystem(SDL_INIT_VIDEO);

}

// redraws the proper display (Scoreboard, etc) on the screen, after first clearing the screen
// call this every time you want the display to return to normal
void display_repaint()
{
	SDL_FillRect(g_screen, NULL, 0); // erase screen
	SDL_Flip(g_screen);
	g_game->video_force_blit();
}

// loads all the .bmp's used by DAPHNE
// returns a 1 if they were all successfully loaded, or a 0 if they weren't
int load_bmps()
{

	int result = 1;	// assume success unless we hear otherwise
	int index = 0;
	char filename[81] = { 0 };

	for (; index < LED_RANGE; index++)
	{
		sprintf(filename, "pics/led%d.bmp", index);

		g_led_bmps[index] = load_one_bmp(filename);

		// If the bit map did not successfully load
		if (g_led_bmps[index] == 0)
		{
			char s[81];
			sprintf(s, "Could not load bitmap : %s", filename);
			printerror(s);
			result = 0;
		}
	}
	g_other_bmps[B_DL_PLAYER1] = load_one_bmp("pics/player1.bmp");
	g_other_bmps[B_DL_PLAYER2] = load_one_bmp("pics/player2.bmp");
	g_other_bmps[B_DL_LIVES] = load_one_bmp("pics/lives.bmp");
	g_other_bmps[B_DL_CREDITS] = load_one_bmp("pics/credits.bmp");
	g_other_bmps[B_DAPHNE_SAVEME] = load_one_bmp("pics/saveme.bmp");
	g_other_bmps[B_GAMENOWOOK] = load_one_bmp("pics/gamenowook.bmp");
    g_other_bmps[B_OVERLAY_LEDS] = load_one_bmp("pics/overlayleds.bmp");

	for (index = 0; index < B_EMPTY; index++)
	{
		if (g_other_bmps[index] == NULL)
		{
			printerror("Could not load an additional bitmap");
			result = 0;
		}
	}

	return(result);

}


SDL_Surface *load_one_bmp(char *filename)
{
	SDL_Surface *result = SDL_LoadBMP(filename);
	return(result);
}

// Draw's one of our LED's to the screen
// value contains the bitmap to draw (0-9 is valid)
// x and y contain the coordinates on the screen
// This function is called from scoreboard.cpp
// 1 is returned on success, 0 on failure
bool draw_led(int value, int x, int y)
{

	bool result = false;	// for now we don't do error checking
	SDL_Rect dest;

	if (g_ldp->is_blitting_allowed())
	{
		dest.x = (short) x;
		dest.y = (short) y;
		dest.w = (unsigned short) g_led_bmps[value]->w;
		dest.h = (unsigned short) g_led_bmps[value]->h;
		SDL_BlitSurface(g_led_bmps[value], NULL, g_screen, &dest);

		if (g_screen->flags & SDL_DOUBLEBUF)
		{
			SDL_Flip(g_screen);
		}
		else
		{
			SDL_UpdateRects(g_screen, 1, &dest);
		}
		result = true;
	}

	return(result);

}

// Draw overlay digits to the screen
void draw_overlay_leds(int values[], int num_digits, int start_x, int y, SDL_Surface *overlay)
{
	SDL_Rect src, dest;

	dest.x = start_x;
	dest.y = y;
	dest.w = OVERLAY_LED_WIDTH;
	dest.h = OVERLAY_LED_HEIGHT;

    src.y = 0;
    src.w = OVERLAY_LED_WIDTH;
    src.h = OVERLAY_LED_HEIGHT;
		
	/* Draw the digit(s) */
	for(int i = 0; i < num_digits; i++)
	{
		src.x = values[i] * OVERLAY_LED_WIDTH;
		SDL_BlitSurface(g_other_bmps[B_OVERLAY_LEDS], &src, overlay, &dest);
	    dest.x += OVERLAY_LED_WIDTH;
	}

    dest.x = start_x;
    dest.w = num_digits * OVERLAY_LED_WIDTH;
    SDL_UpdateRects(overlay, 1, &dest);
}

//  used to draw non LED stuff like scoreboard text
//  'which' corresponds to enumerated values
// Is unsafe because it does not check to see if blitting is allowed
// It also does not update the surface
void draw_othergfx_unsafe(int which, int x, int y)
{
	SDL_Surface *whichbmp = g_other_bmps[which];
	SDL_Rect dest;
	
	dest.x = (short) x;
	dest.y = (short) y;
	dest.w = (unsigned short) whichbmp->w;
	dest.h = (unsigned short) whichbmp->h;
	SDL_BlitSurface(whichbmp, NULL, g_screen, &dest);
}


//  used to draw non LED stuff like scoreboard text
//  'which' corresponds to enumerated values
bool draw_othergfx(int which, int x, int y)
{

	bool result = false;	// for now don't do any error checking
	SDL_Surface *whichbmp = g_other_bmps[which];
	SDL_Rect dest;
	
	if (g_ldp->is_blitting_allowed())
	{
		dest.x = (short) x;
		dest.y = (short) y;
		dest.w = (unsigned short) whichbmp->w;
		dest.h = (unsigned short) whichbmp->h;
		SDL_BlitSurface(whichbmp, NULL, g_screen, &dest);
		
		SDL_UpdateRects(g_screen, 1, &dest);
		if (g_screen->flags & SDL_DOUBLEBUF)
		{
			SDL_Flip(g_screen);
		}
		else
		{
			SDL_UpdateRects(g_screen, 1, &dest);
		}
		result = true;
	}

	return(result);

}

// de-allocates all of the .bmps that we have allocated
void free_bmps()
{

	int nuke_index = 0;

	// get rid of all the LED's
	for (; nuke_index < LED_RANGE; nuke_index++)
	{
		free_one_bmp(g_led_bmps[nuke_index]);
	}
	for (nuke_index = 0; nuke_index < B_EMPTY; nuke_index++)
	{
		// check to make sure it exists before we try to free
		if (g_other_bmps[nuke_index])
		{
			free_one_bmp(g_other_bmps[nuke_index]);
		}
	}
}

void free_one_bmp(SDL_Surface *candidate)
{

	SDL_FreeSurface(candidate);

}

//////////////////////////////////////////////////////////////////////////////////////////

SDL_Surface *get_screen()
{

	return g_screen;

}

int get_console_initialized()
{
	return g_console_initialized;
}

// sets our g_fullscreen bool (determines whether will be in fullscreen mode or not)
void set_fullscreen(bool value)
{
	g_fullscreen = value;
}

// returns video width
Uint16 get_video_width()
{
	return g_vid_width;
}

// sets g_vid_width
void set_video_width(Uint16 width)
{
	// protect against the width being too small
	if (width >= cg_normalwidths[0])
	{
		g_vid_width = width;
	}
	else
	{
		printline("ERROR : width requested is too small, ignoring");
	}
}

// returns video height
Uint16 get_video_height()
{
	return g_vid_height;
}

// sets g_vid_height
void set_video_height(Uint16 height)
{
	// protect against the height being too small
	if (height >= cg_normalheights[0])
	{
		g_vid_height = height;
	}
	else
	{
		printline("ERROR : height requested is too small, ignoring");
	}
}

///////////////////////////////////////////////////////////////////////////////////////////

// converts the SDL_Overlay to a BMP file
// ASSUMES OVERLAY IS ALREADY LOCKED!!!
void take_screenshot(SDL_Overlay *yuvimage)
{
	Uint8 *Y = yuvimage->pixels[0];
	Uint8 *V = yuvimage->pixels[1];
	Uint8 *U = yuvimage->pixels[2];
	int y_extra = yuvimage->pitches[0] - yuvimage->w;
	int v_extra = yuvimage->pitches[1] - (yuvimage->w / 2);
	int u_extra = yuvimage->pitches[2] - (yuvimage->w / 2);
	SDL_Color cur_color = { 0 };

	SDL_Surface *rgbimage = SDL_CreateRGBSurface(SDL_SWSURFACE, yuvimage->w, yuvimage->h, 32, 0xFF, 0xFF00, 0xFF0000, 0);
	if (rgbimage)
	{
		Uint32 *cur_pixel = (Uint32 *) rgbimage->pixels;

		// advance by 2 since the U and V channels are half-sized
		for (int row = 0; row < yuvimage->h; row += 2)
		{
			// advance by 2 since the U and V channels are half-sized
			for (int col = 0; col < yuvimage->w; col += 2)
			{
				// get a 2x2 block of pixels and store them due to YUV structure
				// upper left
				cur_color = yuv2rgb(*Y, *U, *V);
				cur_pixel[0] = SDL_MapRGB(rgbimage->format, cur_color.r, cur_color.g, cur_color.b);

				// upper right
				cur_color = yuv2rgb(*(Y+1), *U, *V);
				cur_pixel[1] = SDL_MapRGB(rgbimage->format, cur_color.r, cur_color.g, cur_color.b);

				// lower left
				cur_color = yuv2rgb(*(Y + yuvimage->pitches[0]), *U, *V);
				cur_pixel[0 + rgbimage->w] = SDL_MapRGB(rgbimage->format, cur_color.r, cur_color.g, cur_color.b);

				// lower right
				cur_color = yuv2rgb(*(Y + yuvimage->pitches[0] + 1), *U, *V);
				cur_pixel[1 + rgbimage->w] = SDL_MapRGB(rgbimage->format, cur_color.r, cur_color.g, cur_color.b);

				cur_pixel += 2;	// move two pixels over
				Y += 2;	// move two Y values over
				U++;	// but just 1 U and V
				V++;
			}
			Y += y_extra;
			V += v_extra;
			U += u_extra;
			cur_pixel += rgbimage->w;	// move down one row
			// WARNING: the above assumes that the pitch is the width * 4

			Y += yuvimage->pitches[0];	// move down one line
		}

		// now the RGB image has been created, time to save
		save_screenshot(rgbimage);
		SDL_FreeSurface(rgbimage);	// de-allocate surface, as we no longer need it

	} // end if RGB image
	else
	{
		printline("ERROR: Could not create RGB image to take screenshot with");
	}

	// NOW save the Y, U and V channels in separate files
	// Most people won't need or want these files, but I have use of them for testing
	Y = yuvimage->pixels[0];
	V = yuvimage->pixels[1];
	U = yuvimage->pixels[2];

	FILE *F = fopen("surface.Y", "wb");
	if (F)
	{
		for (int row = 0; row < yuvimage->h; row++)
		{
			fwrite(Y, 1, yuvimage->w, F);
			Y += yuvimage->pitches[0];
		}
		fclose(F);
	}

	F = fopen("surface.V", "wb");
	if (F)
	{
		for (int row = 0; row < (yuvimage->h / 2); row++)
		{
			fwrite(V, 1, yuvimage->w / 2, F);
			V += yuvimage->pitches[1];
		}
		fclose(F);
	}

	F = fopen("surface.U", "wb");
	if (F)
	{
		for (int row = 0; row < (yuvimage->h / 2); row++)
		{
			fwrite(U, 1, yuvimage->w / 2, F);
			U += yuvimage->pitches[2];
		}
		fclose(F);
	}

}

// saves an SDL surface to a .BMP file in the screenshots directory
void save_screenshot(SDL_Surface *shot)
{
	FILE *F = NULL;
	int screenshot_num = 0;
	char filename[81] = { 0 };

	// search for a filename that does not exist
	for (;;)
	{
		screenshot_num++;
		sprintf(filename, "screen%d.bmp", screenshot_num);
		F = fopen(filename, "rb");

		// if file does not exist, we'll save a screenshot to that filename
		if (!F)
		{
			break;
		}
		else
		{
			fclose(F);
		}
	}

	if (SDL_SaveBMP(shot, filename) == 0)
	{
		outstr("NOTE: Wrote screenshot to file ");
		printline(filename);
	}
	else
	{
		outstr("ERROR: Could not write screenshot to file ");
		printline(filename);
	}
}

// converts YUV to RGB
// Use this only when you don't care about speed =]
SDL_Color yuv2rgb(unsigned char y, unsigned char u, unsigned char v)
{
	SDL_Color result;

	int b = (int) (1.164*(y - 16)                   + 2.018*(u - 128));
	int g = (int) (1.164*(y - 16) - 0.813*(v - 128) - 0.391*(u - 128));
	int r = (int) (1.164*(y - 16) + 1.596*(v - 128));

	// clamp values
	if (b > 255) b = 255;
	if (b < 0) b = 0;
	if (g > 255) g = 255;
	if (g < 0) g = 0;
	if (r > 255) r = 255;
	if (r < 0) r = 0;
	
	result.r = (unsigned char) r;
	result.g = (unsigned char) g;
	result.b = (unsigned char) b;

	return(result);
}

void draw_string(char* t, int col, int row, SDL_Surface* overlay)
{
	SDL_Rect dest;

	dest.x = (short) ((col*6));
	dest.y = (short) ((row*13));
	dest.w = (unsigned short) (6 * strlen(t)); // width of rectangle area to draw (width of font * length of string)
	dest.h = 13;	// height of area (height of font)
		
	SDL_FillRect(overlay, &dest, 0); // erase anything at our destination before we print new text
	SDLDrawText(t,  overlay, FONT_SMALL, dest.x, dest.y);
	SDL_UpdateRects(overlay, 1, &dest);	
}

// toggles fullscreen mode
void vid_toggle_fullscreen()
{
	g_screen = SDL_SetVideoMode(g_screen->w,
		g_screen->h,
		g_screen->format->BitsPerPixel,
		g_screen->flags ^ SDL_FULLSCREEN);
}
