/*
 * vldp.h
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of VLDP, a virtual laserdisc player.
 *
 * VLDP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * VLDP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// API for the mpeg2 virtual laserdisc player--
// by Matt Ownby, July 24th, 2001

#ifndef VLDP_H
#define VLDP_H

#ifdef __cplusplus
extern "C" {
#endif

#include <SDL.h>

struct vldp_info
{
	double fps;	// the constant FPS of the opened video
	double ms_per_frame;	// how many milliseconds per frame (this is often more useful than the fps)
	Uint8 uses_fields;	// whether the video uses fields or not
	Uint32 w;	// width of the mpeg video
	Uint32 h;	// height of the mpeg video
};

enum
{
	STAT_ERROR, STAT_BUSY, STAT_STOPPED, STAT_PLAYING, STAT_PAUSED
};

// initializes the player, passing in the SDL surface that the video will be
// displayed on.
// set blank_during_searches if you want a black screen during searches like a real player does
// set blank_during_skips if you want a black screen during skips like super don does
// set both to 0 if you want the nicest performance (the black screen is not desireable IMO, but
// it is correct for searching)
// Returns 1 on success, or 0 on failure.
// Once the player has been initialized, nothing else should write to the
// SDL_Surface (absolutely nothing! hehe).
int vldp_init(SDL_Surface *screen);

// shuts down the player, de-allocating any memory that was allocated
void vldp_shutdown();

// Sets an optional callback which is allowed to modify the YUV image before
//  it is blitted to the screen.
// This is useful for drawing video overlay on top of the YUV image for
//  laserdisc games that support it.
// Setting the callback pointer to NULL disables it.
void vldp_set_yuv_callback(void (*callback)(SDL_Overlay *cur_overlay));

// opens (and parses if necessary) an mpeg1 or mpeg2 file specified by
// 'filename'.  This must be a demultiplexed video stream (audio not supported)
// Returns 1 on success or 0 on failure.
// This also returns immediately, but you have to wait for the current status to change to READY
// before you issue any other commands.  The only command you can issue if the status is not READY is the
// shutdown command.
int vldp_open_mpeg(char *filename);

// returns a struct full of info about the mpeg including its intended framerate, its resolution, etc
// returns NULL if there is an error
// NOTE : The mpeg must be opened for the result to have meaningful information.  Otherwise the result is random.
struct vldp_info *vldp_get_info();

// returns the current VLDP status. (see enum's for which statuses those can be)
int vldp_get_status();

// set searches to 1 if you want VLDP to make the laserdisc video black during searches
// set skips to 1 if you want VLDP to make the laserdisc video black during skips
void vldp_set_blanking(int searches, int skips);

// begins playing the mpeg, using 'timer' as a timer (use SDL_GetTicks() to produce a sensible timer value)
// returns 1 on success or 0 on error
int vldp_play(Uint32 timer);

// searches to, and _renders_ an mpeg frame (imitating the behavior of a laser
// disc player).  The first frame is frame 0 (the first frame on a laserdisc
// is frame 1).
// You can search to I, P or B frames.  The search is exact,
// not approximate.  It also should be VERY FAST (ie, search completes in < 1 second).  This is the one feature
// that no other mpeg player I have encountered has.
// returns 1 when the search is complete and the frame has been displayed or
// 0 if there was a problem.
int vldp_search(Uint16 frame);

// skips to the specified frame during playback as if the destination frame follows the current frame.
// timer indicates the new playback timer to be used once we resume playback
// returns 1 if successful or 0 if failure
int vldp_skip(Uint16 frame, Uint32 timer);

// pauses the playback of the video, showing a still frame.  If the video
// is already paused, this does nothing.
int vldp_pause();

// similar to pause except this function also steps forward 1 frame
// Notice there is no step_backward.  This is to alert the user the stepping backward is expensive and if you must use do it, use search instead.
// returns 1 on success
int vldp_step_forward();

// mimics the behavior of a laserdisc player by drawing a black screen,
// and resetting the current frame to the first frame.
int vldp_stop();

// returns the current mpeg frame # that is being displayed on the screen
// The first frame is frame # 0
Uint16 vldp_get_current_frame();

// Enables audio in either the left (0) or right (1) channel.  If audio is
// already enabled, this does nothing.
int vldp_enable_audio(int channel);

// Disables audio in either the left (0) or right (1) channel. If audio is
// already disabled, this does nothing.
int vldp_disable_audio(int channel);

#ifdef __cplusplus
}
#endif

#endif
