/*
 * vldp.c
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of VLDP, a virtual laserdisc player.
 *
 * VLDP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * VLDP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// see vldp.h for documentation on how to use the API

#include <stdio.h>
#include <string.h>
//#include <inttypes.h>
#include "vldp.h"
#include "vldp_common.h"

//////////////////////////////////////////////////////////////////////////////////////

int vldp_cmd(int cmd);
int vldp_wait_for_status(int stat);

//////////////////////////////////////////////////////////////////////////////////////

SDL_Thread *private_thread = NULL;

int p_searched_yet = 0;	// have we issued any search commands yet?
int p_initialized = 0;	// whether VLDP has been initialized

int g_blank_during_searches = 0;	// whether we should blank during searches
int g_blank_during_skips = 0;	// whether we should blank during skips

Uint8 g_req_cmdORcount = CMDORCOUNT_INITIAL;	// the current command parent thread requests of the child thread
unsigned int g_ack_count = ACK_COUNT_INITIAL;	// the result returned by the internal child thread
char g_req_file[STRSIZE];	// requested mpeg filename
Uint32 g_req_timer = 0;	// requests timer value to be used for mpeg playback
Uint16 g_req_frame = 0;		// requested frame to search to
Uint16 g_ack_frame = 0;		// actual current frame
int g_ack_status = 0;		// actual mpeg playing status (see enums)
struct vldp_info g_ack_info;	// contains info about the FPS, dimensions, etc of video

void (*g_yuv_callback)(SDL_Overlay *cur_overlay) = NULL;	// an optional callback that gets called right before we blit
	// this allows the user to add things to the YUV overlay (such as video overlay)

/////////////////////////////////////////////////////////////////////

int vldp_init(SDL_Surface *screen)
{
	p_initialized = 0;

	private_thread = SDL_CreateThread(idle_handler, screen);	// start our internal thread
	
	// if private thread was created successfully
	if (private_thread)
	{
		p_initialized = 1;
	}

	return p_initialized;
}

void vldp_shutdown()
{
	// only shutdown if we have previous initialized
	if (p_initialized)
	{
		vldp_cmd(VLDP_REQ_QUIT);
		SDL_WaitThread(private_thread, NULL);	// wait for private thread to terminate
	}
	p_initialized = 0;
}


void vldp_set_yuv_callback(void (*callback)(SDL_Overlay *cur_overlay))
{
	if (p_initialized)
	{
		g_yuv_callback = callback;
	}
}

// requests that we open an mpeg file
int vldp_open_mpeg(char *filename)
{
	int result = 2;	// assume we're busy so the loop below works the first time
	FILE *F = NULL;
	
	if (p_initialized)
	{
		F = fopen(filename, "rb");
		// if file exists, we can open it
		if (F)
		{
			fclose(F);
			strcpy(g_req_file, filename);
			vldp_cmd(VLDP_REQ_OPEN);
		
			// loop while the mpeg is parsing
			while (result == 2)
			{
				result = vldp_wait_for_status(STAT_STOPPED);	// wait until we get stopped or an error
				SDL_Delay(1);	// timing isn't critical here, so we can delay for 1 instead of 0
			}
		}
		else
		{
			fprintf(stderr, "VLDP ERROR : can't open file %s\n", filename);
		}
	}

	return result;
}

struct vldp_info *vldp_get_info()
{
	struct vldp_info *result = NULL;

	if (p_initialized)
	{
		result = &g_ack_info;
	}
	return result;
}

int vldp_get_status()
{
	return g_ack_status;
}

void vldp_set_blanking(int searches, int skips)
{
	g_blank_during_searches = searches;
	g_blank_during_skips = skips;
}

int vldp_play(Uint32 timer)
{
	int result = 0;

	if (p_initialized)
	{		
		// if we are playing without searching, seek to frame 0 to avoid the 'green effect'
		if (!p_searched_yet)
		{
			vldp_search(0);
		}

		g_req_timer = timer;
		result = vldp_cmd(VLDP_REQ_PLAY);
	}
	return(result);
}

int vldp_search(Uint16 frame)
{
	int result = 0;

	if (p_initialized)
	{
		p_searched_yet = 1;
		g_req_frame = frame;
		vldp_cmd(VLDP_REQ_SEARCH);
		result = vldp_wait_for_status(STAT_PAUSED);
	}
	return result;
}

int vldp_skip(Uint16 frame, Uint32 timer)
{
	int result = 0;

	// we can only skip if the mpeg is already playing	
	if (p_initialized && (g_ack_status == STAT_PLAYING))
	{
		g_req_frame = frame;
		g_req_timer = timer;
		result = vldp_cmd(VLDP_REQ_SKIP);
		if (!result) fprintf(stderr, "vldp_cmd rejected SKIP request!\n");
	}
	// remove me once this bug is discovered
	else
	{
		fprintf(stderr, "VLDP skip failed because one of these happened: \n");
		fprintf(stderr, "p_initialized is %d\n", p_initialized);
		fprintf(stderr, "g_ack_status == %d\n", g_ack_status);
	}
	return result;
}

int vldp_pause()
{
	int result = 0;
	if (p_initialized)
	{
		result = vldp_cmd(VLDP_REQ_PAUSE);
	}
	return result;
}

int vldp_step_forward()
{
	int result = 0;
		
	if (p_initialized)
	{
		result = vldp_cmd(VLDP_REQ_STEP_FORWARD);
	}
	return result;
}

int vldp_stop()
{	
	return 0;	// this is a useless function which I haven't finished yet anyway, so don't use it :)
}

Uint16 vldp_get_current_frame()
{
	Uint16 result = 0;

	if (p_initialized)
	{
		result = g_ack_frame;
	}

	return result;
}

///////////////////////////////////////////////////////////////////////////

// issues a command to the internal thread and returns 1 if the internal thread acknowledged our command
// or 0 if we timed out without getting a response
// NOTE : this does not mean that the internal thread has finished executing our requested command, only
// that the command has been received
int vldp_cmd(int cmd)
{
	int result = 0;
	Uint32 cur_time = SDL_GetTicks();
	Uint8 tmp = g_req_cmdORcount;	// we want to replace the real value atomically so we use a tmp variable first
	static unsigned int old_ack_count = ACK_COUNT_INITIAL;

	tmp++;	// increment the counter so child thread knows we're issuing a new command
	tmp &= 0xF;	// strip off old command
	tmp |= cmd;	// replace it with new command
	g_req_cmdORcount = tmp;	// here is the atomic replacement
	
	// loop until we timeout or get a response
	while (SDL_GetTicks() - cur_time < VLDP_TIMEOUT)
	{
		// if the count has changed, it means the other thread has acknowledged our new command
		if (g_ack_count != old_ack_count)
		{
			result = 1;
			old_ack_count = g_ack_count;	// prepare to receive the next command
			break;
		}
		SDL_Delay(0);	// switch to other thread
	}

	// if we weren't able to communicate, notify user
	if (!result)
	{
		fprintf(stderr, "VLDP error!  Timed out waiting for internal thread to accept command!\n");
	}
	
	return result;
}

// waits until the disc status is 'stat'
// if the stat we want is received, return 1 (success)
// if we time out, or if we get an error stat, return 0 (error)
// if we are legitimately busy, we return 2 (busy)
int vldp_wait_for_status(int stat)
{
	int result = 0;	// assume error unless we explicitly
	int done = 0;
	Uint32 cur_time = SDL_GetTicks();
	while (!done && (SDL_GetTicks() - cur_time < VLDP_TIMEOUT))
	{
		if (g_ack_status == stat)
		{
			done = 1;
			result = 1;
		}
		else if (g_ack_status == STAT_ERROR)
		{
			done = 1;
		}
		// else keep waiting
		SDL_Delay(0);	// switch to other thread, timing is critical so we delay for 0
	}

	// if we timed out but are busy, indicate that
	if (g_ack_status == STAT_BUSY)
	{
		result = 2;
	}
	
	// else if we timed out
	else if (SDL_GetTicks() - cur_time >= VLDP_TIMEOUT)
	{
		fprintf(stderr, "VLDP ERROR!!!!  Timed out with getting our expected response!\n");
	}

	return result;
}
