/*
 * video_out_null.c
 * Copyright (C) 2000-2002 Michel Lespinasse <walken@zoy.org>
 * Copyright (C) 1999-2000 Aaron Holtzman <aholtzma@ess.engr.uvic.ca>
 *
 * This file is part of mpeg2dec, a free MPEG-2 video stream decoder.
 * See http://libmpeg2.sourceforge.net/ for updates.
 *
 * mpeg2dec is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * mpeg2dec is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"

#include <stdlib.h>
#include <inttypes.h>

#include "video_out.h"
#include "convert.h"

// start MPO

#include <string.h>	// for memset
#include "../vldp/vldp_common.h"	// to get access to g_in_info struct and the yuv_buf struct
#include "../vldp/vldp_internal.h"	// for access to s_ variables from vldp_internal

#define YUV_BUF_COUNT 3	// libmpeg2 needs 3 buffers to do its thing ...
struct yuv_buf g_yuv_buf[YUV_BUF_COUNT];

////

static void null_draw_frame (vo_instance_t *instance, uint8_t * const * buf, void *id)
{
    Sint32 correct_elapsed_ms = 0;	// we want this signed since we compare against actual_elapsed_ms
    Sint32 actual_elapsed_ms = 0;	// we want this signed because it could be negative

	// if we don't need to skip any frames
	if (!(s_frames_to_skip | s_skip_all))
	{		
		int advanced = 0;	// whether we have advanced our frame counter yet

		// loop once, or more than once if we are paused
		do
		{
#ifndef VLDP_BENCHMARK
			// compute how much time ought to have elapsed based on our frame count
			correct_elapsed_ms = (Sint32) (s_frames_before_next_frame * g_out_info.ms_per_frame);
			actual_elapsed_ms = SDL_GetTicks() - s_timer;

			// if we are caught up enough that we don't need to skip any frames, then display the frame
			if (actual_elapsed_ms < (correct_elapsed_ms + (g_out_info.ms_per_frame * 2)))
			{
#endif
				// this is the potentially expensive callback that gets the hardware overlay
				// ready to be displayed, so we do this before we sleep
				// NOTE : if this callback fails, we don't want to display the frame due to double buffering considerations
				if (g_in_info->prepare_frame(&g_yuv_buf[(int) id]))
				{

#ifndef VLDP_BENCHMARK
				
					// stall if we are playing too quickly and if we don't have a command waiting for us
					while ((Sint32) (SDL_GetTicks() - s_timer) < correct_elapsed_ms)
					{
						if (ivldp_got_new_command())
						{
							break;
						}
	    				SDL_Delay(1);	// note, if this is set to 0, we don't get commands as quickly
					}

					// if a command comes in at the last second,
					// we don't want to render the next frame that we were going to because it could cause overrun
					// so we only display the frame if we haven't received a command
					if (!ivldp_got_new_command())
					{
#endif
						// draw the frame
						// we are using the pointer 'id' as an index, kind of risky, but convenient :)
						g_in_info->display_frame(&g_yuv_buf[(int) id]);
#ifndef VLDP_BENCHMARK
					} // end if we didn't get a new command to interrupt the frame being displayed
				} // end if the frame was prepared properly

				// else maybe we couldn't get a lock on the buffer fast enough, so we'll have to wait ...

			} // end if we don't drop any frames

			/*
			// else we're too far beyond so we're gonna have to drop some this frame to catch up (doh!)
			else
			{
				fprintf(stderr, "NOTE : dropped frame %u!  Expected %u but got %u\n",
					g_out_info.current_frame, correct_elapsed_ms, actual_elapsed_ms);
			}
			*/
#endif

			// if we have not advanced our frame counter yet, then do so
			// the reason this is necessary is because if we are paused, we will loop and we only want to
			// advance the frame counter the first time
			if (!advanced)
			{
				g_out_info.current_frame++;
				advanced = 1;
			}

			// now that the frame has either been displayed or skipped, we can update the counters to reflect
			s_frames_before_next_frame++;

			// if the frame is to be paused, then stall
			if (s_paused)
			{
				paused_handler();
			}
			// else if we are supposed to be playing
			else
			{
				play_handler();
			}
		  
		} while (s_paused && !s_skip_all && !s_step_forward);
		// loop while we are paused so video overlay gets redrawn

		s_step_forward = 0;	// clear this in case it was set (since we have now stepped forward)

	} // end if we don't have frames to skip
	
	// if we have frames to skip
	else
	{		
		// we could skip frames for another reason
		if (s_frames_to_skip > 0)
		{
			s_frames_to_skip--;	// we've skipped a frame, so decrease the count
		}
	}

#ifndef VLDP_BENCHMARK
	SDL_Delay(0);	// switch to another thread just in case
#endif
	
	// end MATT
}

static void null_setup_fbuf (vo_instance_t * _instance,
			    uint8_t ** buf, void ** id)
{
	static buffer_index = 0;
	*id = (int *) buffer_index;	// THIS IS A LITTLE TRICKY
	// We are setting an integer value to a pointer ...
	// Because it is convenient to let the pointer hold the value of this integer for us
	// Hopefully it doesn't cause any trouble later ;)

    buf[0] = g_yuv_buf[buffer_index].Y;
    buf[1] = g_yuv_buf[buffer_index].U;
    buf[2] = g_yuv_buf[buffer_index].V;
    
    buffer_index++;

	// we are operating under a (safe?) assumption that this function is called in sets of YUV_BUF_COUNT
	// so it should be safe to wraparound ... if our assumption is wrong, major havoc will ensure :)
	if (buffer_index >= YUV_BUF_COUNT)
	{
		buffer_index = 0;
	}

}

static int null_setup (vo_instance_t * instance, int width, int height,
		       vo_setup_result_t * result)
{
	int i = 0;
	
	g_in_info->report_mpeg_dimensions(width, height);	// alert parent-thread
	g_out_info.w = width;
	g_out_info.h = height;

	for (i = 0; i < YUV_BUF_COUNT; i++)
	{
		// allocate buffer according to size of picture	
		// We do not re-allocate these buffers if they have already been previous allocated, as a safety measure
		g_yuv_buf[i].Y_size = width * height;
		g_yuv_buf[i].UV_size = g_yuv_buf[i].Y_size >> 2;
		if (!g_yuv_buf[i].Y) g_yuv_buf[i].Y = malloc(g_yuv_buf[i].Y_size);
		if (!g_yuv_buf[i].U) g_yuv_buf[i].U = malloc(g_yuv_buf[i].UV_size);
		if (!g_yuv_buf[i].V) g_yuv_buf[i].V = malloc(g_yuv_buf[i].UV_size);
	}
	
    result->convert = NULL;
    return 0;
}

static void null_close (vo_instance_t *instance)
{
	int i = 0;
	
	for (i = 0; i < YUV_BUF_COUNT; i++)
	{
		free(g_yuv_buf[i].Y);
		g_yuv_buf[i].Y = NULL;
		free(g_yuv_buf[i].U);
		g_yuv_buf[i].U = NULL;
		free(g_yuv_buf[i].V);
		g_yuv_buf[i].V = NULL;
	}
}

vo_instance_t * vo_null_open ()
{
    vo_instance_t * instance;

    instance = (vo_instance_t *) malloc (sizeof (vo_instance_t));
    if (instance == NULL)
	return NULL;

    instance->setup = null_setup;	// MPO
    instance->setup_fbuf = null_setup_fbuf;	// MPO
    instance->set_fbuf = NULL;
    instance->start_fbuf = NULL;
    instance->draw = null_draw_frame;
    instance->discard = NULL;
    instance->close = null_close;	// MPO
	memset(g_yuv_buf, 0, sizeof(g_yuv_buf));	// MPO, fill this struct with 0's so that we can track whether mem has been allocated or not

    return instance;
}

// end MPO
