/*
 * vldp.h
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of VLDP, a virtual laserdisc player.
 *
 * VLDP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * VLDP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// API for the mpeg2 virtual laserdisc player
// by Matt Ownby, Mar 16th, 2003

#ifndef VLDP_H
#define VLDP_H

#ifdef __cplusplus
extern "C" {
#endif

#include <SDL.h>	// only used for threading

struct yuv_buf
{
	void *Y;	// Y channel
	void *U;	// U channel
	void *V;	// V channel
	unsigned int Y_size;	// size in bytes of Y
	unsigned int UV_size;	// size in bytes of U and V
};

// since this is C and not C++, we can't use booleans ...
#define VLDP_TRUE 1
#define VLDP_FALSE 0

// callback functions and state information provided to VLDP from the parent thread
struct vldp_in_info
{
	// VLDP calls this when it has a frame ready to draw, but before it has slept to maintain
	// a consistent framerate.  You should do all cpu-intensive cycles to prepare the frame
	// to be drawn here.  The remaining cycles you don't used will be used by VLDP to sleep
	// until it's time for the frame to be displayed.
	// This returns 1 if the frame was prepared successfully, or 0 on error
	int (*prepare_frame)(struct yuv_buf *buf);

	// VLDP calls this when it wants the frame that was earlier prepared to be displayed
	// ASAP
	void (*display_frame)(struct yuv_buf *buf);
	
	// VLDP calls this when it is doing the time consuming process of reporting an mpeg parse
	void (*report_parse_progress)(double percent_complete);

	// VLDP calls this to tell the parent-thread what the dimensions of the mpeg are (as soon we know ourselves!)
	// This comes before draw_frame to that draw_frame can be more optimized
	void (*report_mpeg_dimensions)(int width, int height);

	// VLDP calls this when it wants the parent-thread to render a blank frame
	void (*render_blank_frame)();

	///////////

	int blank_during_searches;	// if this is non-zero, VLDP will call render_blank_frame before every search
	int blank_during_skips;	// if this is non-zero, VLDP will call render_blank_frame before every skip
};

// functions and state information provided to the parent thread from VLDP
struct vldp_out_info
{
	// shuts down VLDP, de-allocates any memory that was allocated, etc ...
	void (*shutdown)();
	
	// All of the following functions return 1 on success, 0 on failure.
	
	// open an mpeg file and returns immediately.  File won't actually be open until
	// 'status' is equal to STAT_STOPPED.
	int (*open)(const char *filename);

	// like 'open' except it blocks until 'status' is equal to STAT_STOPPED (until all potential parsing has finished, etc)
	int (*open_and_block)(const char *filename);

	// plays the mpeg that has been previously open.  'timer' is the SDL_GetTicks() that
	// we should use for the beginning of the first frame that will be displayed
	int (*play)(Uint32 timer);
	
	// searches to a frame relative to the beginning of the mpeg (0 is the first frame)
	// returns immediately, but search is not complete until 'status' is STAT_PAUSED
	int (*search)(Uint16 frame);
	
	// like search except it blocks until the search is complete
	int (*search_and_block)(Uint16 frame);
	
	// skips to 'frame' and immediately begins playing.
	// 'timer' is the SDL_GetTicks() value relative to the beginning of 'frame'
	// the mpeg is required to be playing before skip is called, but this may be
	//  an artificial limitation designed to encourage proper laserdisc player usage
	int (*skip)(Uint16 frame, Uint32 timer);
	
	// pauses mpeg playback
	int (*pause)();
	
	// steps one frame forward while paused
	int (*step_forward)();
	
	// stops playback
	int (*stop)();

	////////////////////////////////////////////////////////////

	// State information for the parent thread's benefit
	double fps;	// the constant FPS of the opened video
	double ms_per_frame;	// how many milliseconds per frame (this is often more useful than the fps)
	Uint8 uses_fields;	// whether the video uses fields or not
	Uint32 w;	// width of the mpeg video
	Uint32 h;	// height of the mpeg video
	int status;	// the current status of the VLDP (see STAT_ enum's)
	Uint16 current_frame;	// the current frame of the opened mpeg that we are on
};

enum
{
	STAT_ERROR, STAT_BUSY, STAT_STOPPED, STAT_PLAYING, STAT_PAUSED
};

#ifdef WIN32
// this is the only function that gets exported by this DLL
__declspec(dllexport) const struct vldp_out_info *vldp_init(const struct vldp_in_info *in_info);
#else

// initializes the VLDP
// 'in_info' contains a few callback functions from parent thread that VLDP needs to use
// returns a pointer to output functions on success or NULL on failure
const struct vldp_out_info *vldp_init(const struct vldp_in_info *in_info);

#endif

#ifdef __cplusplus
}
#endif

#endif	// VLDP_H
