/*
 * vldp_internal.c
 *
 * Copyright (C) 2001 Matt Ownby
 *
 * This file is part of VLDP, a virtual laserdisc player.
 *
 * VLDP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * VLDP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// contains the function for the VLDP private thread

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
//#include <unistd.h>
//#include "inttypesreplace.h"
#include <SDL.h>

#include "vldp.h"
#include "vldp_common.h"
#include "vldp_internal.h"
#include "mpegscan.h"

#ifdef WIN32
#include "../vc++/inttypes.h"
#else
#include <inttypes.h>
#endif

#include "mpeg2.h"
#include "video_out.h"

/////////////////////////////

// NOTICE : these variables should only be used by the private thread !!!!!!!!!!!!

Uint8 s_old_req_cmdORcount = CMDORCOUNT_INITIAL;	// the last value of the command byte we received
int s_paused = 0;	// whether the video is to be paused
int s_step_forward = 0;	// whether to step 1 frame forward
int s_blanked = 0;	// whether the mpeg video is to be blanked
int s_frames_to_skip = 0;	// how many frames to skip before rendering the next frame (used for P and B frames seeking)
int s_skip_all = 0;	// if we are to skip any frame to be displayed (to avoid seeing frames we shouldn't)

Uint32 s_timer = 0;	// FPS timer used by the blitting code to run at the right speed
Uint32 s_frames_before_next_frame = 0;	// how many frames must have been displayed (relative to s_timer)
											// before we advance to the next frame

#define MAX_LDP_FRAMES 60000

static FILE *g_mpeg_handle = NULL;	// mpeg file we currently have open
static mpeg2dec_t *g_mpeg_data = NULL;	// structure for libmpeg2's state
static vo_instance_t *s_video_output = NULL;
static Uint32 g_frame_position[MAX_LDP_FRAMES] = { 0 };	// the file position of each I frame
static Uint16 g_totalframes = 0;	// total # of frames in the current mpeg

#define BUFFER_SIZE 262144
static Uint8 g_buffer[BUFFER_SIZE];	// buffer to hold mpeg2 file as we read it in

////////////////////////////////////////////////

// this is our video thread which gets called
int idle_handler(void *surface)
{
	int done = 0;

	s_video_output = (vo_instance_t *) vo_null_open();	// open 'null' driver (we just pass decoded frames to parent thread)
	if (s_video_output)
	{
		g_mpeg_data = mpeg2_init();
	}
	else
	{
		fprintf(stderr, "VLDP : Error opening LIBVO!\n");
		done = 1;
	}

	// unless we are drawing video to the screen, we just sit here
	// and listen for orders from the parent thread
	while (!done)
	{		
		// while we have received new commands to be processed
		// (so we don't go to sleep on skips)
		while (ivldp_got_new_command() && !done)
		{
			// examine the actual command (strip off the count)
			switch(g_req_cmdORcount & 0xF0)
			{
			case VLDP_REQ_QUIT:
				done = 1;
				break;
			case VLDP_REQ_OPEN:
				idle_handler_open();
				idle_handler_reset();	// is this reset necessary?
				break;
			case VLDP_REQ_PLAY:
				// shouldn't need to reset since we are not changing position in the mpeg file
				idle_handler_play();
				break;
			case VLDP_REQ_SEARCH:
				idle_handler_reset();
				idle_handler_search(0);
				break;
			case VLDP_REQ_SKIP:
				idle_handler_reset();
				idle_handler_search(1);
				break;
			case VLDP_REQ_PAUSE:	// pause command while we're already idle?  this is an error
			case VLDP_REQ_STOP:	// stop command while we're already idle? this is an error
				ivldp_ack_command();
				g_out_info.status = STAT_ERROR;
				break;
			default:
				fprintf(stderr, "VLDP WARNING : Idle handler received command which it is ignoring\n");
				break;
			} // end switch
			SDL_Delay(0);	// give other threads some breathing room (but not much hehe)
		} // end if we got a new command

		g_in_info->render_blank_frame();	// This makes sure that the video overlay gets drawn even if there is no video being played
		
//		SDL_Delay(1); // don't hog cpu
	} // end while we have not received a quit command
			
	// if we have a file open, close it
	if (g_mpeg_handle)
	{
		fclose(g_mpeg_handle);
		g_mpeg_handle = 0;
	}

	g_out_info.status = STAT_ERROR;
	mpeg2_close(g_mpeg_data);	// shutdown libmpeg2
	s_video_output->close(s_video_output);		// shutdown null driver

	ivldp_ack_command();	// acknowledge quit command

	return 0;
}

// returns 1 if there is a new command waiting for us or 0 otherwise
int ivldp_got_new_command()
{
	int result = 0;

	// if they are no longer equal	
	if (g_req_cmdORcount != s_old_req_cmdORcount)
	{
		result = 1;
	}
	
	return result;
}

// acknowledges a command sent by the parent thread
// NOTE : We don't check to see if parent thread got our acknowledgement because it creates too much latency
void ivldp_ack_command()
{
	s_old_req_cmdORcount = g_req_cmdORcount;
	g_ack_count++;	// here is where we acknowledge
}

// the handler we call if we're paused... called from video_out_sdl
// this function is called from within a while loop
void paused_handler()
{
	// the moment we render the still frame, we need to reset the FPS timer so we don't try to catch-up
	if (g_out_info.status != STAT_PAUSED)
	{
		g_out_info.status = STAT_PAUSED;
		// QUESTION : Do we really want to reset these variables?
		s_timer = SDL_GetTicks();	// since we have just rendered the frame we searched to, we refresh the timer
		s_frames_before_next_frame = 1;	// this gives us a little breathing room
	}

	// if we have a new command coming in	
	if (ivldp_got_new_command())
	{
		// strip off the count and examine the command
		switch (g_req_cmdORcount & 0xF0)
		{
		case VLDP_REQ_PLAY:
			ivldp_respond_req_play();
//			printf("Paused handler acknowledged play command at %u, %u ms late\n", SDL_GetTicks(), SDL_GetTicks() - s_timer);
			break;
		// if we get any of these commands, we have to skip the remaining buffered frames and reset
		case VLDP_REQ_STOP:
		case VLDP_REQ_QUIT:
		case VLDP_REQ_OPEN:
		case VLDP_REQ_SEARCH:
			s_skip_all = 1;
			// we don't acknowledge these requests here, we let the idle handler take care of them
			break;
		case VLDP_REQ_STEP_FORWARD:
			// if they've also requested a step forward
			ivldp_ack_command();
			s_step_forward = 1;
			break;
		default:	// else if we get a pause command or another command we don't know how to handle, just ignore it
			fprintf(stderr, "WARNING : pause handler received command %u that it is ignoring\n", g_req_cmdORcount);
			ivldp_ack_command();	// acknowledge the command
			break;
		} // end switch
	} // end if we have a new command coming in
}

// the handler we call if we're playing
// this handler is called from within a while loop
void play_handler()
{
	// if we've received a new incoming command
	if (ivldp_got_new_command())
	{
		// strip off count and examine command
		switch(g_req_cmdORcount & 0xF0)
		{
		case VLDP_REQ_NONE:	// no incoming command
			break;
		case VLDP_REQ_PAUSE:
		case VLDP_REQ_STEP_FORWARD:
			ivldp_ack_command();
			// if they've also requested a step forward
			if ((g_req_cmdORcount & 0xF0) == VLDP_REQ_STEP_FORWARD)
			{
				s_step_forward = 1;
			}
			s_paused = 1;
			s_blanked = 0;
			break;
		case VLDP_REQ_STOP:
		case VLDP_REQ_QUIT:
		case VLDP_REQ_OPEN:
		case VLDP_REQ_SEARCH:
		case VLDP_REQ_SKIP:
			s_skip_all = 1;	// bail out, handle these requests in the idle handler
			break;
		default:	// unknown or redundant command, just ignore
			ivldp_ack_command();
			fprintf(stderr, "WARNING : play handler received command which it is ignoring\n");
			break;
		}
	} // end if we got a new command
}

// resets libmpeg2 as if it were being run for the first time
// (needed when seeking within the file or opening a new file)
void idle_handler_reset()
{	
	// only reset if we have a file open
	if (g_mpeg_handle)
	{
		// reset libmpeg2 so it is prepared to begin reading from a new m2v file
		mpeg2_partial_init(g_mpeg_data);

		vldp_process_sequence_header();	// we need to process the sequence header before we can jump around the file for frames
	}
	else
	{
		fprintf(stderr, "VLDP ERROR : no mpeg file has been opened, can't reset\n");
	}
}

// sets the framerate inside our info structure based upon the framerate code received
void ivldp_set_framerate(Uint8 frame_rate_code)
{
	// now to compute the framerate

	switch (frame_rate_code)
	{
	case 1:
		g_out_info.fps = 23.976;
		break;
	case 2:
		g_out_info.fps = 24.0;
		break;
	case 3:
		g_out_info.fps = 25.0;
		break;
	case 4:
		g_out_info.fps = 29.97;
		break;
	case 5:
		g_out_info.fps = 30.0;
		break;
	case 6:
		g_out_info.fps = 50.0;
		break;
	case 7:
		g_out_info.fps = 59.94;
		break;
	case 8:
		g_out_info.fps = 60.0;
		break;
	default:
		// else we got an invalid frame rate code
		fprintf(stderr, "ERROR : Invalid frame rate code!\n");
		g_out_info.fps = 1.0;	// to avoid divide by 0 error
		break;
	} // end switch
	
	g_out_info.ms_per_frame = 1000.0 / g_out_info.fps;
	
}

/////////////////

// decode_mpeg2 function taken from mpeg2dec.c and optimized a bit
static void decode_mpeg2 (uint8_t * current, uint8_t * end)
{
    const mpeg2_info_t * info;
    int state;
    vo_setup_result_t setup_result;

    mpeg2_buffer (g_mpeg_data, current, end);

    info = mpeg2_info (g_mpeg_data);

	// loop until we return (state is -1)
	for (;;)
    {
		state = mpeg2_parse (g_mpeg_data);
		switch (state)
		{
		case -1:
		    return;
		case STATE_SEQUENCE:
		    /* might set nb fbuf, convert format, stride */
		    /* might set fbufs */
		    if (s_video_output->setup (s_video_output, info->sequence->width,
				       info->sequence->height, &setup_result))
			{
				fprintf (stderr, "display setup failed\n");	// this should never happen
		    }
	
		    if (setup_result.convert)
				mpeg2_convert (g_mpeg_data, setup_result.convert, NULL);
					    
		    // if this driver offers a setup_fbuf callback ...
//		    else if (s_video_output->setup_fbuf)
// we KNOW we offer a setup_fbuf function, so get rid of this conditional
		    {
				uint8_t * buf[3];
				void * id;
	
				s_video_output->setup_fbuf (s_video_output, buf, &id);
				mpeg2_set_buf (g_mpeg_data, buf, id);
				s_video_output->setup_fbuf (s_video_output, buf, &id);
				mpeg2_set_buf (g_mpeg_data, buf, id);
				s_video_output->setup_fbuf (s_video_output, buf, &id);
				mpeg2_set_buf (g_mpeg_data, buf, id);
		    }
		    break;
		case STATE_PICTURE:
		    /* might skip */
		    /* might set fbuf */

			// all possible stuff we could've done here was removed because the null driver
			// doesn't do any of it
		    
		    break;
		case STATE_PICTURE_2ND:
		    /* should not do anything */
		    break;
		case STATE_SLICE:
		case STATE_END:
		    /* draw current picture */
		    /* might free frame buffer */

			// if the init hasn't been called yet, this may fail so we have to put the conditional
		    if (info->display_fbuf)
		    {
				s_video_output->draw (s_video_output, info->display_fbuf->buf,
				      info->display_fbuf->id);
		    }
		    
		    break;
		} // end switch
    } // end endless for loop
}

/////////////////

// feeds libmpeg2 the beginning of the file up to the first Group of Picture
// It needs this info to get setup, and we probably don't want to start playing from the beginning of the file
void vldp_process_sequence_header()
{
	Uint32 val = 0;
	int index = 0;

#define BYTES_TO_READ 0x200

	rewind(g_mpeg_handle);	// start at the beginning
	fread(&g_buffer, 1, BYTES_TO_READ, g_mpeg_handle);	// assume that we must be the first frame in 0x200 bytes
		// if not, we'll have to increase the number
		
	// go until we have found the first frame or we run out of data
	while (val != 0x000001B8)
	{
		val = val << 8;
		val |= g_buffer[index];	// add newest byte to bottom of val
		index++;	// advance the end pointer
		if (index > BYTES_TO_READ)
		{
			fprintf(stderr, "VLDP : Could not find first frame in 0x%x bytes.  Modify source code to increase buffer!\n", BYTES_TO_READ);
			break;
		}
	}

	decode_mpeg2 (g_buffer, g_buffer + index - 4);	// decode the sequence header

}

// opens a new mpeg2 file
void idle_handler_open()
{
	char req_file[STRSIZE] = { 0 };
	
	strcpy(req_file, g_req_file);	// after we ack the command, this string could become clobbered at any time
	ivldp_ack_command();	// acknowledge open command

	g_out_info.status = STAT_BUSY;	// make us busy while opening the file

	// if we have previously opened an mpeg, we need to close it and reset
	if (g_mpeg_handle)
	{
		fclose(g_mpeg_handle);
		g_mpeg_handle = 0;

		g_in_info->render_blank_frame();
		g_in_info->render_blank_frame();
		// since the overlay is double buffered, we want to blank it 
		// twice before closing it.  This is to avoid a 'flicker effect' that we can
		// get when switching between mpeg files.


		// we need to close this surface, because the new mpeg may have a different resolution than the old one, and therefore,
		// the YUV buffer must be re-allocated
		s_video_output->close(s_video_output);
	}

	// check to make sure it's a video stream and also get framerate
	g_mpeg_handle = fopen(req_file, "rb");
	if (g_mpeg_handle)
	{
		Uint8 small_buf[8];
		fread(&small_buf, 1, 8, g_mpeg_handle);	// 1st 8 bytes reveals much
		
		// if we find the proper mpeg2 video header at the beginning of the file
		if (((small_buf[0] << 24) | (small_buf[1] << 16) | (small_buf[2] << 8) | small_buf[3]) == 0x000001B3)
		{
			g_out_info.w = (small_buf[4] << 4) | (small_buf[5] >> 4);	// get mpeg width
			g_out_info.h = ((small_buf[5] & 0x0F) << 8) | small_buf[6];	// get mpeg height
			ivldp_set_framerate(small_buf[7] & 0xF);	// set the framerate
			fclose(g_mpeg_handle);	// close file since the next function opens it again

			// load/parse all the frame locations in the file for super fast seeking
			if (ivldp_get_mpeg_frame_offsets(req_file))
			{
				g_mpeg_handle = fopen(req_file, "rb");	// open the new mpeg file
				g_out_info.status = STAT_STOPPED;	// now that the file is open, we're ready to play
			}
			else
			{
				fprintf(stderr, "VLDP PARSE ERROR : Is the video stream damaged?\n");
				fclose(g_mpeg_handle);
				g_mpeg_handle = NULL;
				g_out_info.status = STAT_ERROR;	// change from BUSY to ERROR
			}
		}
		
		// if the file had a bad header
		else
		{
			fprintf(stderr, "VLDP ERROR : Did not find expected header.  Is this mpeg stream demultiplexed??\n");
			fclose(g_mpeg_handle);
			g_mpeg_handle = NULL;
			g_out_info.status = STAT_ERROR;
		}
	} // end if file exists
	else
	{
		fprintf(stderr, "VLDP ERROR : Could not open file!\n");
		g_out_info.status = STAT_ERROR;
	}
#ifdef VLDP_DEBUG
	printf("vldp open returning ...\n");
#endif
}

// starts playing the mpeg from its current position
void idle_handler_play()
{
	// safety check: if we're not at the end-of-file on that mpeg2 stream ...
	if (!feof(g_mpeg_handle))
	{
		ivldp_respond_req_play();
		// when the frame is actually blitted is when we set the status to STAT_PLAYING
		ivldp_render();
	}
	
	// if we are at the EOF, we ought to return an error to the play command
	// to help people debug
	else
	{
		g_out_info.status = STAT_ERROR;
		ivldp_ack_command();
	}
}

// responds to play request
void ivldp_respond_req_play()
{
	s_timer = g_req_timer;
	s_frames_before_next_frame = 1;	// we want to render the currently shown frame for 1 frame before moving on
	ivldp_ack_command();	// acknowledge the play command
	g_out_info.status = STAT_PLAYING;	// we strive for instant response (and catch-up to maintain timing)
	s_paused = 0;	// we to not want to pause on 1 frame
	s_blanked = 0;	// we want to see the video
	s_frames_to_skip = 0;	// skip no frames, just play from current position
	// this value is reset again as soon as we confirm that we are playing
}

// displays 1 or more frames to the screen, according to the state variables.
// This function can be used to do both still frames and moving video.  Play and search both use this function.
void ivldp_render()
{
    Uint8 *end = NULL;
	int render_finished = 0;

#ifdef VLDP_BENCHMARK
	Uint32 render_start_time = SDL_GetTicks();	// keep track of when we started
	Sint16 render_start_frame = g_out_info.current_frame;	// keep track of the frame we started timing stuff on
	double total_seconds = 0.0;	// used to calculate FPS
	Sint16 total_frames = 0;	// used to calculate FPS
	FILE *F = fopen("benchmark.txt", "wt");	// create a little logfile for benchmark results
#endif

	// check to make sure a file has been opened
	if (!g_mpeg_handle)
	{
		render_finished = 1;
		fprintf(stderr, "VLDP RENDER ERROR : we tried to render an mpeg but none was open!\n");
		g_out_info.status = STAT_ERROR;
	}

	// while we're not finished playing and pausing		
    while (!render_finished)
    {
		end = g_buffer + fread (g_buffer, 1, BUFFER_SIZE, g_mpeg_handle);
		
		// safety check, they could be equal if we were already at EOF before we tried this
		if (g_buffer != end)
		{
			// read chunk of video stream
			decode_mpeg2 (g_buffer, end);	// display it to the screen
		}
		
		// if we've read to the end of the mpeg2 file, then we can't play anymore, so we pause on last frame
		if (end != g_buffer + BUFFER_SIZE)
		{
			g_out_info.status = STAT_PAUSED;
			render_finished = 1;
			s_skip_all = 0;	// in case we had to stop displaying frames, turn them back on now

			g_out_info.current_frame = 0;	// is this wise? :)
		}

		// if a new command is coming in, check to see if we need to stop
		if (ivldp_got_new_command())
		{
			// check to see if we need to suddenly abort the rendering process
			switch (g_req_cmdORcount & 0xF0)
			{
			case VLDP_REQ_QUIT:
			case VLDP_REQ_OPEN:
			case VLDP_REQ_SEARCH:
			case VLDP_REQ_STOP:
				g_out_info.status = STAT_BUSY;
				s_skip_all = 0;	// if we had to hide frames, re-enable them now (FIXME : Why did I do this? I can't remember)
				render_finished = 1;
				break;
			case VLDP_REQ_SKIP:
				// do not change the playing status
				s_skip_all = 0;
				render_finished = 1;
				break;
			} // end switch
		} // end if they got a new command
    } // end while

#ifdef VLDP_BENCHMARK
	fprintf(F, "Benchmarking result:\n");
	total_frames = g_out_info.current_frame - render_start_frame;
	total_seconds = (SDL_GetTicks() - render_start_time) / 1000.0;
	fprintf(F, "VLDP displayed %u frames (%d to %d) in %f seconds (%f FPS)\n",
		total_frames, render_start_frame, g_out_info.current_frame, total_seconds, total_frames / total_seconds);
	fclose(F);
#endif

}

// searches to any arbitrary frame, be it I, P, or B, and renders it
// if skip is set, it will do a laserdisc skip instead of a search (ie it will go a frame, resume playback,
// and not adjust any timers)
void idle_handler_search(int skip)
{
	Uint32 proposed_pos = 0;
	Uint16 req_frame = g_req_frame; // after we acknowledge the command, g_req_frame could become clobbered
	Uint16 actual_frame = g_req_frame;
	Uint32 skip_timer = g_req_timer;	// timer to be used if we skip
	int skipped_I = 0;

	ivldp_ack_command();	// acknowledge search/skip command
	
	// if we're doing a search, we want to change the status
	// but if we're doing a skip we don't want to change the status because it is
	// supposed to be instantaneous
	if (!skip)
	{
		g_out_info.status = STAT_BUSY;

		// if we are to blank during searches ...
		if (g_in_info->blank_during_searches)
		{
			g_in_info->render_blank_frame();
		}
	}

	// if this is a skip and we are to blank before skips
	else if (g_in_info->blank_during_skips)
	{
		g_in_info->render_blank_frame();
	}

	// do a bounds check
	if (req_frame < g_totalframes)
	{		
		proposed_pos = g_frame_position[req_frame];	// get the proposed position

		s_frames_to_skip = 0;	// the below problem is no longer a problem

		// loop until we find which position in the file to seek to
		for (;;)
		{
		  // if the frame we want is not an I frame, go backward until we find an I frame, and increase # of frames to skip forward
		  while ((proposed_pos == 0xFFFFFFFF) && (actual_frame > 0))
		  {
			s_frames_to_skip++;
			actual_frame--;
			proposed_pos = g_frame_position[actual_frame];
		  }
		  skipped_I++;

			// if we are only 2 frames away from an I frame, we will get a corrupted image and need to go back to
			// the I frame before this one
		  if ((skipped_I < 2) && (s_frames_to_skip < 3) && (actual_frame > 0))
		  {
		  	proposed_pos = 0xFFFFFFFF;
		  }
		  else
		  {
#ifdef VLDP_DEBUG
//				printf("We've decided on a position within the file.\n");
//				printf("skipped_I is %d\n", skipped_I);
//				printf("s_frames_to_skip is %d\n", s_frames_to_skip);
//				printf("actual_frame is %d\n", actual_frame);
#endif
		  	break;
		  }
		}

#ifdef VLDP_DEBUG
		printf("frames_to_skip is %d, skipped_I is %d\n", s_frames_to_skip, skipped_I);
		printf("position in mpeg2 stream we are seeking to : %x\n", proposed_pos);
#endif

		fseek(g_mpeg_handle, proposed_pos, SEEK_SET);	// go to the place in the stream where the I frame begins

		g_out_info.current_frame = req_frame - 1;	// this will be incremented in the player

		s_blanked = 0;	// we want to see the frame

		// if we're searching, not skipping, we want to pause when we render the frame, and we want to reset
		// all our timers		
		if (!skip)
		{
			s_paused = 1;	// we do want to pause on the frame we search to
			s_timer = SDL_GetTicks();	// reset timer so framerate is correct
			s_frames_before_next_frame = 0;	// display frame immediately
		}
		
		// if we are skipping we are actually in the middle of playback so we don't touch the timers
		else
		{
			s_paused = 0;
			s_timer = skip_timer;
			s_frames_before_next_frame = 0;
			// the timer is setup for when the new frame will be displayed
			// so this should be 0, not 1
		}
		ivldp_render();
	} // end if the bounds check passed
	else
	{
		fprintf(stderr, "SEARCH ERROR : frame %u was requested, but it is out of bounds\n", req_frame);
		g_out_info.status = STAT_ERROR;
	}
}



// parses an mpeg video stream to get its frame offsets, or if the parsing had taken place earlier
int ivldp_get_mpeg_frame_offsets(char *mpeg_name)
{
	char datafilename[320] = { 0 };
	int mpeg_datafile_good = 0;
	FILE *data_file = NULL;
	FILE *mpeg_file = NULL;
	int result = 1;
	Uint32 mpeg_size = 0;
	struct stat filestats;	// used to get the size of a file
	struct dat_header header;

	// GET LENGTH OF ACTUAL FILE
	mpeg_file = fopen(mpeg_name, "rb");
	if (mpeg_file)
	{
		fstat(fileno(mpeg_file), &filestats);	// get stats for file to get file length
		mpeg_size = filestats.st_size;
	}
	else
	{
		fprintf(stderr, "VLDP FATAL ERROR: video file could not be found!\n");
		return 0;
	}

	// change extension of file to be dat instead of (presumably) m2v
	strcpy(datafilename, mpeg_name);
	strcpy(&datafilename[strlen(mpeg_name)-3], "dat");

	// loop until we get a good datafile or until user quits
	while (!mpeg_datafile_good)
	{
		// keep trying to make the datafile until we succeed of the user quits
		while (!data_file)
		{
			data_file = fopen(datafilename, "rb");	// check to see if datafile exists

			// if file does not exist, we have to create it
			if (!data_file)
			{
				result = ivldp_parse_mpeg_frame_offsets(datafilename, mpeg_size, mpeg_file);
				data_file = NULL;	// need to set to NULL so we load the file on the next loop
			}
		} // end of inner while that tries to open the datafile

		// if mpeg parsing worked properly (or if it was not needed)
		if (result)
		{
			// now that file exists and we have it open, we have to read it

			fseek(data_file, 0L, SEEK_SET);
			fread(&header, sizeof(header), 1, data_file);	// read .DAT file header

			// if version, file size, or finished are wrong, the dat file is no good and has to be regenerated
			if ((header.length != mpeg_size) || (header.version != DAT_VERSION) || (header.finished != 1))
			{
//				printf("*** Alleged mpeg size is %u, actual size is %u\n", header.length, mpeg_size);
//				printf("Finished flag is %x\n", header.finished);
//				printf("DAT version is %x\n", header.version);
				printf("NOTICE : MPEG data file has to be created again!\n");
				fclose(data_file);
				data_file = NULL;
				unlink(datafilename);	// delete file and create it
			}
			else
			{
				g_out_info.uses_fields = header.uses_fields;
				mpeg_datafile_good = 1;	// escape the loop
			}
		} // end if mpeg parsing worked (or if it was not necessary)
	} // end while we don't have a good datafile
	
	// if we didn't exit the loop because someone quit, then we need to read the datafile
	// data_file is still open
	if (result && data_file)
	{
		g_totalframes = 0;

#ifdef VLDP_DEBUG
//		unlink("frame_report.txt");
#endif

		// read all the frame positions
		// if we don't read 4 bytes, it means we've hit the EOF and we're done
		while (fread(&g_frame_position[g_totalframes], 4, 1, data_file) == 1)
		{
#ifdef VLDP_DEBUG
//			FILE *tmp_F = fopen("frame_report.txt", "ab");
//			fprintf(tmp_F, "Frame %d has offset of %x\n", g_totalframes, g_frame_position[g_totalframes]);
//			fclose(tmp_F);
#endif
			g_totalframes++;
			
			// safety check, it is possible to make mpegs with too many frames to fit onto one CAV laserdisc
			// (in fact I did this, and it caused a lot of problems in the debug stages hehe)
			if (g_totalframes >= MAX_LDP_FRAMES)
			{
				fprintf(stderr, "ERROR : current mpeg has too many frames, VLDP will ignore any frames above %u\n", MAX_LDP_FRAMES);
				break;
			}
		}
#ifdef VLDP_DEBUG
		printf("*** g_totalframes is %u\n", g_totalframes);
		printf("And frame 0's offset is %x\n", g_frame_position[0]);
#endif
	}

	if (data_file)
	{
		fclose(data_file);
	}
	if (mpeg_file)
	{
		fclose(mpeg_file);
	}
	
	return result;
}


int ivldp_parse_mpeg_frame_offsets(char *datafilename, Uint32 mpeg_size, FILE *mpeg_file)
{
	int result = 1;
	FILE *data_file = fopen(datafilename, "wb");	// create file
	struct dat_header header;	// header to put inside .DAT file

	// if we could create the file successfully, then we need to populate it
	if (data_file)
	{
		Uint32 pos = 0;	// position in the file
		int count = 0;
		int parse_result = 0;

		header.version = DAT_VERSION;
		header.finished = 0;
		header.uses_fields = 0;
		header.length = mpeg_size;
		fwrite(&header, sizeof(header), 1, data_file);
		// first thing that goes in the file is the .DAT header
		// That way we can re-use the file another time with confidence that it's
		// the right one

		init_mpegscan();

		// keep reading the file while there is a file left to be read
		do
		{
#define PARSE_CHUNK 200000

			parse_result = parse_video_stream(mpeg_file, data_file, PARSE_CHUNK);
			pos += PARSE_CHUNK;

			// we want to give the user updates but don't want to flood them
			if (count > 10)
			{
				count = 0;
				
				// report progress to parent thread
				g_in_info->report_parse_progress((double) pos / mpeg_size);
			}
			count++;

		} while (parse_result == P_IN_PROGRESS);

		// if parse finished, then we have to update the header
		if (parse_result != P_ERROR)
		{
			header.finished = 1;
			header.uses_fields = 0;
			if (parse_result == P_FINISHED_FIELDS)
			{
				header.uses_fields = 1;
			}
			fseek(data_file, 0L, SEEK_SET);
			fwrite(&header, sizeof(header), 1, data_file);	// save changes
		}

		// we have to close data file because it's write-only
		// and it needs to be re-opened read-only
		fclose(data_file);
		data_file = NULL;

		// if the mpeg did not finish parsing gracefully, we've got problems
		// NOTE : I separated this from the other if above to guarantee that the file gets closed
		if (parse_result == P_ERROR)
		{
			fprintf(stderr, "There was an error parsing the MPEG file.\n");
			fprintf(stderr, "Either there is a bug in the parser or the MPEG file is corrupt.\n");
			fprintf(stderr, "OR the user aborted the decoding process :)\n");
			result = 0;
			unlink(datafilename);
		}
	} // end if we could create the file successfully

	// we couldn't create data file which means no write permission probably
	// this is probably a good time to shut VLDP down =]
	else
	{
		fprintf(stderr, "Could not create file %s\n", datafilename);
		fprintf(stderr, "This probably means you don't have permission to create the file\n");
		result = 0;
	}

	return result;
}
