// DaphneLoader C++ version (uses wxWindows for cross-platform compatibility)
// by Matt Ownby

#include <sys/stat.h>
#include "main.h"

#ifdef WIN32
#include <windows.h>
#else
#include <sys/types.h>	// for fork
#include <unistd.h>		// for fork
#endif

#if defined(__DARWIN__)
#include <mach-o/dyld.h>	// for _NSGetExecutablePath
#endif

///////////////////////////////////////////////////

ini g_ini;
MyFrame *g_mainframe = NULL;	// a pointer to our MyFrame instance in case we need to access its functions

///////////////////////////////////////////////////

BEGIN_EVENT_TABLE(MyFrame, wxFrame)
    EVT_MENU(ID_Quit, MyFrame::OnQuit)
    EVT_MENU(ID_About, MyFrame::OnAbout)
	EVT_MENU(ID_Start, MyFrame::OnStart)
	EVT_MENU(ID_Config, MyFrame::OnConfig)
	EVT_LISTBOX(ID_List_Select, MyFrame::OnSelected)
	EVT_BUTTON(ID_Config, MyFrame::OnConfig)	
	EVT_BUTTON(ID_Start, MyFrame::OnStart)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(MyPreConfig, wxDialog)
	EVT_BUTTON(ID_CFG_KEYS, MyPreConfig::OnConfigKeys)
	EVT_BUTTON(ID_CFG_BUTTONS, MyPreConfig::OnConfigButtons)
	EVT_BUTTON(ID_CFG_OK, MyPreConfig::OnOK)
	EVT_BUTTON(ID_CFG_CANCEL, MyPreConfig::OnCancel)
	EVT_BUTTON(ID_CFG_APPLY, MyPreConfig::OnApply)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(MyConfig, wxNotebook)
	EVT_NOTEBOOK_PAGE_CHANGED(ID_Notebook, MyConfig::OnTabChange)
	EVT_COMBOBOX(ID_PLAYER_TYPE, MyConfig::OnPlayerType)
	EVT_BUTTON(ID_BROWSE_FRAMEFILE, MyConfig::OnBrowseFramefile)
END_EVENT_TABLE()

IMPLEMENT_APP(MyApp)

#include <string.h>

// takes a pathname (relative or absolute) + a filename and removes the filename itself
// therefore it returns just the path (stored in 'path')
// !!! We assume that the 'path' buffer's size is >= 'file_with_path' size
// returns false if there is an error (such as the file having no path)
bool get_path_of_file(char *file_with_path, char *path)
{
	bool success = false;
	int index = strlen(file_with_path) - 1;	// start on last character

	// make sure the file_with_path is at least 2 characters long
	// because otherwise our proceeding calculations could cause a segfault
	if (index > 0)
	{
		// locate the preceeding / or \ character
		while ((index >= 0) && (file_with_path[index] != '/') && (file_with_path[index] != '\\'))
		{
			index--;
		}

		// if we found a leading / or \ character
		if (index >= 0)
		{
			strcpy(path, file_with_path);
			path[index+1] = 0;	// chop off filename from string (we already know
			success = true;
		}
	}

	return success;
}


bool MyApp::OnInit()
{
#if defined(__DARWIN__)
	char file[160], path[160];
	unsigned long l = sizeof(file);
	_NSGetExecutablePath(file, &l);
	get_path_of_file(file, path);	// isolate the path ...
	chdir(path);	// change to the directory where the loader is stored
	chdir("../../..");	// move out of application bundle schlop
//	printf("Changed to dir %s\n", path);
#endif
//	printf("loading init...\n");

	g_ini.load_ini();

    m_frame = new MyFrame( "Daphne Loader 1.0 (C++ version)", wxPoint(50,50), wxSize(420,285) );
    m_frame->Show(TRUE);
    SetTopWindow(m_frame);

	// hopefully this warning will deter some questions ...
	wxMessageBox("This program is VERY incomplete, almost all features are missing.\n"
			"It will have more features when someone decides to do the work to add them.",
			"WARNING!", wxOK | wxICON_ERROR, m_frame);

    return TRUE;
}

MyFrame::MyFrame(const wxString& title, const wxPoint& pos, const wxSize& size)
: wxFrame((wxFrame *)NULL, -1, title, pos, size),
	m_menuFile(NULL), m_menuHelp(NULL), m_menuBar(NULL),
	m_panel(NULL), m_listbox(NULL), m_start_button(NULL), m_config_button(NULL),
	m_canvas(NULL)
{
	g_mainframe = this;

	m_the_vars.vldp_framefile = "undefined";
	m_the_vars.ldp_type_index = 0;
	m_the_vars.ldp_driver = "noldp";	// safe default
	// FIXME : set more defaults later

	m_menuFile = new wxMenu();
//	m_menuFile->Append( ID_Error_Log, "View Daphne Error &Log" );
//    m_menuFile->AppendSeparator();
    m_menuFile->Append( ID_Quit, "E&xit" );

	m_menuGame = new wxMenu();
	m_menuGame->Append( ID_Start, "Start the game");
//	m_menuGame->Append( ID_Watch, "Watch the laserdisc associated with selected game");
	m_menuGame->AppendSeparator();
	m_menuGame->Append( ID_Config, "Configure DAPHNE options");

	m_menuHelp = new wxMenu();
//	m_menuHelp->Append( ID_Web_Site, "Daphne Web Site" );
//	m_menuHelp->Append( ID_Documentation, "Daphne Documentation" );
//	m_menuHelp->AppendSeparator();
	m_menuHelp->Append( ID_About, "About Daphne Loader" );

	m_menuBar = new wxMenuBar();
    m_menuBar->Append( m_menuFile, "&File" );
	m_menuBar->Append( m_menuGame, "&Game" );
	m_menuBar->Append( m_menuHelp, "&Help" );

    SetMenuBar( m_menuBar );

	// child controls
	m_panel = new wxPanel(this);
	m_listbox = new wxListBox(m_panel, ID_List_Select, wxPoint(5, 5), wxSize(235, 120));

	m_start_button = new wxButton(m_panel, ID_Start, "&Start!");
	m_config_button = new wxButton(m_panel, ID_Config, "&Configure");

	m_canvas = new MyCanvas(m_panel, wxPoint(-1,-1), wxSize(160, 120));

	m_daphne_logo = new MyCanvas(m_panel, wxPoint(-1, -1), wxSize(164, 52));	// fixme, find a way to fit around image
	m_daphne_logo->new_image("logo.jpg");

	// if init file can be loaded
	if (g_ini.ini_loaded())
	{
		if (g_ini.find_key("GameList"))
		{
			int game_count = atoi(g_ini.get_right_of_equals());
			int cur_game = 0;

			// add each game name to our listbox
			for (cur_game = 0; cur_game < game_count; cur_game++)
			{
				m_listbox->Append(g_ini.get_right_of_equals());
			}
			m_listbox->SetSelection(0);	// select first item
			wxCommandEvent tmp;
			OnSelected(tmp);	// force the graphic to display
		}
	}
	else
	{
		wxMessageBox("Could not open DaphneLoader .ini file!",
			"Fatal error!", wxOK | wxICON_ERROR, this);

		/*
		char s[81];
		getcwd(s, sizeof(s));
		wxMessageBox(s, s, wxOK | wxICON_ERROR, this);
		*/

		Close(TRUE);
	}

	// DO SIZERS NOW
	m_sizer_main = new wxBoxSizer(wxHORIZONTAL);
	m_sizer_right = new wxBoxSizer(wxVERTICAL);
	m_sizer_buttons = new wxBoxSizer(wxHORIZONTAL);
	
	// the button row ...
	m_sizer_buttons->Add(m_start_button, 0, 0, 0);
	m_sizer_buttons->Add(m_config_button, 0, 0, 0);

	// the right portion
	m_sizer_right->Add(m_canvas, 0, 0, 0);
	m_sizer_right->Add(0, 10, 0);	// spacer
	m_sizer_right->Add(m_sizer_buttons, 0, wxALIGN_CENTER_HORIZONTAL, 0);
	m_sizer_right->Add(0, 10, 0);	// spacer
	m_sizer_right->Add(m_daphne_logo, 0, wxALIGN_CENTER, 0);
	// add more stuff later like a little logo

	m_sizer_main->Add(m_listbox, 1, wxALL | wxEXPAND | wxGROW, 5);	// add the list of games here ...
	m_sizer_main->Add(m_sizer_right, 0, wxALL, 5);	// add our right panel

	m_panel->SetSizer(m_sizer_main);
	m_panel->Fit();
	this->Fit();

}

void MyFrame::OnQuit(wxCommandEvent& WXUNUSED(event))
{
    Close(TRUE);
}

void MyFrame::OnAbout(wxCommandEvent& WXUNUSED(event))
{
    wxMessageBox("Daphne Loader (C++ version) by Matt Ownby",
        "About Daphne Loader", wxOK | wxICON_INFORMATION, this);
}

void MyFrame::OnStart(wxCommandEvent &event)
{
	// make sure the queue is empty (it should be)
	m_cmd_line.clear();

#ifdef WIN32
	AddCmdLine("daphne.exe");	// name of the executable
#else
	AddCmdLine("daphne");	// name of the executable
#endif

	// verify that the EXE exists before we proceed any further ...
	FILE *F = fopen(m_cmd_line[0].c_str(), "rb");
	
	// if the file exists where we expect it ...
	if (F)
	{
		fclose(F);
	}

	// if the file does not exist ...
	else
	{
	    wxMessageBox("The DAPHNE executable was not found in the current directory.",
	        "Unable to start!", wxOK | wxICON_ERROR, this);
		return;
	}
	
	// attach game driver name to command line
	if (g_ini.find_key(m_listbox->GetStringSelection()))
	{
		AddCmdLine(g_ini.get_var_value("DriverName"));
	}

	AddCmdLine(m_the_vars.ldp_driver);

	// if we also need to include a framefile ...
	if (m_the_vars.ldp_driver == "vldp")
	{
		AddCmdLine("-framefile");

		// windows receives its command line as one big string, so it needs quotes surrounding
		// the framefile path (in case of spaces)		
#ifdef WIN32
		string inquotes = "\"";
		inquotes += m_the_vars.vldp_framefile + "\"";
		AddCmdLine(inquotes);
#else
		AddCmdLine(m_the_vars.vldp_framefile);
#endif
	}

	// windows and unix have different methods for launching a new process
#ifdef WIN32
	string cmd_line = "";

	// add command line to one big line win32 string ...
	for (int i = 0; i != m_cmd_line.size(); i++)
	{
		cmd_line += m_cmd_line[i];
		cmd_line += " ";
	}

	char cmd_line_buf[1024];	// because createprocess won't accept const char *
	strncpy(cmd_line_buf, cmd_line.c_str(), cmd_line.size());
	cmd_line_buf[cmd_line.size()] = 0;	// terminate string

	PROCESS_INFORMATION process_info;
	STARTUPINFO startup_info;
	memset(&startup_info, 0, sizeof(startup_info));
	startup_info.cb = sizeof(startup_info);
	BOOL result = CreateProcess("daphne.exe",
		cmd_line_buf, 
		NULL,	// process attributes
		NULL,	// thread attributes
		FALSE,	// inherit handles?
		0,	// creation flags
		NULL,	// new environment
		NULL,	// current directory (same as us)
		&startup_info,	// startup info
		&process_info);	// process information
#else
	const char **argv = (const char **) malloc(sizeof (const char *) * (m_cmd_line.size() + 1));
	// allocate memory to store command line, add an extra slot for NULL termination

	// add command line to one big line win32 string ...
	for (int i = 0; i != m_cmd_line.size(); i++)
	{
		argv[i] = m_cmd_line[i].c_str();
	}
	argv[m_cmd_line.size()] = NULL;	// final null terminating entry

	// fork and exec
	pid_t pid = fork();
	
	// if we are the child process
	if (pid == 0)
	{
		// if exec failed ...
		if (execv(argv[0], (char* const*) argv) < 0)
		{
			perror("Execv failed, error is");
			
			// get rid of wxWindows callbacks
			atexit(NULL);
//			on_exit(NULL, NULL);
			exit(1);	// critical shutdown ...
		}
	}

	// parent process
	else
	{
	}	

	free(argv);
#endif

}

void MyFrame::AddCmdLine(string s)
{
	m_cmd_line.push_back(s);	// add to the queue ...
}

void MyFrame::OnConfig(wxCommandEvent & WXUNUSED(event))
{
	// now create the config dialog box ...
	MyPreConfig pre_config(this);
	pre_config.ShowModal();	// show the modal dialog box ...
}

void MyFrame::OnSelected(wxCommandEvent &event)
{
		// if we can locate the game's data
		if (g_ini.find_key(m_listbox->GetStringSelection()))
		{
			const char *val = g_ini.get_var_value("ImageName");

			// if an image exists for this game
			if (val)
			{
				m_canvas->new_image(val);
			}

			// otherwise, display a blank image
			else
			{
				m_canvas->clear_image();
			}
		}

		m_canvas->Refresh();
}


BEGIN_EVENT_TABLE(MyCanvas, wxScrolledWindow)
    EVT_PAINT(MyCanvas::OnPaint)
END_EVENT_TABLE()

// Define a constructor for my canvas
MyCanvas::MyCanvas(wxWindow *parent, const wxPoint& pos, const wxSize& size):
 wxScrolledWindow(parent, -1, pos, size, wxSUNKEN_BORDER),
	 m_bitmap(NULL)
{
	// wxSUNKEN_BORDER, wxSIMPLE_BORDER
	m_image.AddHandler(new wxJPEGHandler);
	m_bitmap = new wxBitmap();
}

// tells the canvas to display a new image
void MyCanvas::new_image(const wxString &filename)
{
	m_bitmap->LoadFile("images/" + filename, wxBITMAP_TYPE_JPEG);
}

// erases bitmap
void MyCanvas::clear_image()
{
	m_bitmap->Create(1, 1);
}

// Define the repainting behaviour
void MyCanvas::OnPaint(wxPaintEvent& WXUNUSED(event))
{
	wxPaintDC dc(this);

	if ( m_bitmap && m_bitmap->Ok() )
	{
		wxMemoryDC memDC;

		memDC.SelectObject(* m_bitmap);

		// Normal, non-transparent blitting
		dc.Blit(0, 0, m_bitmap->GetWidth(), m_bitmap->GetHeight(), & memDC, 0, 0, wxCOPY, FALSE);

		memDC.SelectObject(wxNullBitmap);
	}
}

//////////////////////

MyPreConfig::MyPreConfig(wxWindow *parent)
	: wxDialog(parent, -1, wxString(_T("Config")))
{
	m_config = new MyConfig(this, ID_Notebook);
	m_config->CreateInitialPages();
	
	m_ok = new wxButton(this, ID_CFG_OK, "&OK");
	m_cancel = new wxButton(this, ID_CFG_CANCEL, "Cancel");
	m_apply = new wxButton(this, ID_CFG_APPLY, "Apply");
	
	// sizers
	m_sizer_main = new wxBoxSizer(wxVERTICAL);
	m_sizer_bottom_row = new wxBoxSizer(wxHORIZONTAL);
	
	int flags = wxALL;
	m_sizer_bottom_row->Add(m_ok, 0, flags, 4);
	m_sizer_bottom_row->Add(m_cancel, 0, flags, 4);
	m_sizer_bottom_row->Add(m_apply, 0, flags, 4);

    m_sizer_config = new wxNotebookSizer(m_config);
	m_sizer_main->Add(m_sizer_config, 1, wxEXPAND);
	m_sizer_main->Add(m_sizer_bottom_row, 0, wxEXPAND);

	wxSize real_min_size(0, 0);	// this'll eventually be the true minimum size

	for (int i = 0; i < 3; i++)
	{
		// display possibility 1
		m_config->display_dynamic_possibility(i);

		// compuate minimum size
		m_sizer_main->Layout();

		// update our real min size as we found bigger minimum sizes
		wxSize tmp = m_sizer_main->GetMinSize();
		if (tmp.GetWidth() > real_min_size.GetWidth()) real_min_size.SetWidth(tmp.GetWidth());
		if (tmp.GetHeight() > real_min_size.GetHeight()) real_min_size.SetHeight(tmp.GetHeight());
	}

	// we know truly know the minimum size so we can safely set it
	m_sizer_main->SetMinSize(real_min_size);
	
	this->SetSizer(m_sizer_main);
	this->Fit();
}

void MyPreConfig::OnConfigKeys(wxCommandEvent& event)
{
}

void MyPreConfig::OnConfigButtons(wxCommandEvent& event)
{
}

void MyPreConfig::OnOK(wxCommandEvent& event)
{
	m_config->apply_changes();
	EndModal(0);
}

void MyPreConfig::OnCancel(wxCommandEvent& event)
{
	EndModal(0);
}

void MyPreConfig::OnApply(wxCommandEvent& event)
{
	m_config->apply_changes();
}

MyConfig::MyConfig(wxWindow *parent, wxWindowID id,
	const wxPoint &pos, const wxSize &size, long style)
	: wxNotebook(parent, id, pos, size, style)
{
	// empty
}

// just a test to make sure that all the memory we were allocating was getting de-allocated properly
MyConfig::~MyConfig()
{
}

void MyConfig::CreateInitialPages()
{
	wxPanel *panel = NULL;
		
	panel = CreateGamePage();
	AddPage(panel, "Game", FALSE);
	
	panel = CreateLaserdiscPage();
	AddPage(panel, "Laserdisc", FALSE);
	
	panel = CreateVideoPage();
	AddPage(panel, "Video", FALSE);

	panel = CreateAudioPage();
	AddPage(panel, "Audio", FALSE);

	panel = CreateInputPage();
	AddPage(panel, "Input", FALSE);

	panel = CreateAdvancedPage();
	AddPage(panel, "Advanced", FALSE);
}

// if the laserdisc player type is selected ...
void MyConfig::OnPlayerType(wxCommandEvent &event)
{
	if (event.m_eventType == wxEVT_COMMAND_COMBOBOX_SELECTED)
	{
		HandlePlayerTypeSelection(event.m_commandString);
	}
}

// this isn't part of OnPlayerType because this function is also called when we first initialize the Laserdisc Page
void MyConfig::HandlePlayerTypeSelection(const char *player_name)
{
	// locate laserdisc player's info
	if (g_ini.find_key(player_name))
	{
		const char *val;
		val = g_ini.get_var_value("DriverName");
		if (val)
		{
			m_ldp_driver = val;
		}

		val = g_ini.get_var_value("Type");
		if (val)
		{
			m_ldp_type = val;
			ResetLaserdiscPageLayout();
		}
	}
}

void MyConfig::ResetLaserdiscPageLayout()
{

	// hide none message
	m_ldp_main_sizer->Show(m_none_sizer, false);

	m_vldp_box->Show(false);	// the box must be hidden separately because it is somewhat independent
	m_ldp_main_sizer->Show(m_vldp_sizer, false);

	m_ldp_box->Show(false);
	m_ldp_main_sizer->Show(m_ldp_sizer, false);

	// now make the selected one visible
	if (m_ldp_type == "default")
	{
		m_ldp_main_sizer->Show(m_none_sizer, true);
	}

	if (m_ldp_type == "virtual")
	{
		m_vldp_box->Show(true);
		m_ldp_main_sizer->Show(m_vldp_sizer, true);
	}

	if (m_ldp_type == "serial")
	{
		m_ldp_box->Show(true);
		m_ldp_main_sizer->Show(m_ldp_sizer, true);
	}

	m_centering_sizer->Layout();

}

// bring up a file browsing window so user can select their framefile
void MyConfig::OnBrowseFramefile(wxCommandEvent &event)
{
	wxFileDialog file_dialog(this, "Find your framefile",
		".",	// default directory
		"",	// default filename
		"*.txt",
		wxOPEN | wxHIDE_READONLY);

	int result = file_dialog.ShowModal();

	// if they pressed ok
	if (result == wxID_OK)
	{
		m_vldp_framefile->SetValue(file_dialog.GetPath());
	}
}

void MyConfig::OnTabChange(wxNotebookEvent &event)
{

	// if they click on the laserdisc tab, we need to make sure that it's up to date
	if (event.GetSelection() == 1)
	{
		HandlePlayerTypeSelection(m_player_type->GetStringSelection());
	}
	
	event.Skip();	// this seems to be required
}

// stores all widget variables into our variable struct because this class will be destroyed before the game is run
void MyConfig::apply_changes()
{
	struct the_vars *vars = g_mainframe->get_readwrite_vars();	// get write access to the structure ...

	vars->ldp_driver = m_ldp_driver;
	vars->ldp_type_index = m_player_type->GetSelection();
	vars->vldp_framefile = m_vldp_framefile->GetValue();
}

void MyConfig::display_dynamic_possibility(int which_one)
{
	switch (which_one)
	{
	case 0:
		m_ldp_type = "default";
		break;
	case 1:
		m_ldp_type = "virtual";
		break;
	case 2:
		m_ldp_type = "serial";
		break;
	default:
		assert(0);	// this should never happen
		break;
	}

	ResetLaserdiscPageLayout();
}

wxPanel *MyConfig::CreateGamePage()
{
	wxPanel *panel = new wxPanel(this);
	return panel;
}

wxPanel *MyConfig::CreateLaserdiscPage()
{
	m_the_vars = g_mainframe->get_readonly_vars();	// we only wait this class to be able to read those variables by default

    wxPanel *panel = new wxPanel(this);

	m_player_type = new wxComboBox(panel, ID_PLAYER_TYPE);
	m_player_type->SetToolTip("People who do not have a real laserdisc player will want to choose Virtual LDP.");

	wxStaticText *player_type_text = new wxStaticText(panel, -1, _T("Player Type:"));

	if (g_ini.find_key("LDPlayerList"))
	{
		int ldp_count = atoi(g_ini.get_right_of_equals());
		int cur_ldp = 0;

		// add each game name to our listbox
		for (cur_ldp = 0; cur_ldp < ldp_count; cur_ldp++)
		{
			m_player_type->Append(g_ini.get_right_of_equals());
		}
	}

	wxSize size = m_player_type->GetBestSize();
	size.SetWidth(size.GetWidth() + 30);	// add a bit of padding to the end
	m_player_type->SetSize(size);

	wxBoxSizer *expand_ldp_list = new wxBoxSizer(wxHORIZONTAL);
	expand_ldp_list->Add(m_player_type, 0, 0, 0);

	// START NONE
	m_none_text = new wxStaticText(panel, -1, _T("WARNING : This is probably not what you want!\n"
		"If you don't have a laserdisc player, choose Virtual LDP.\n"
		"Laserdisc player 'none' is for testing purposes only."));
	m_none_sizer = new wxBoxSizer(wxVERTICAL);
	m_none_sizer->Add(m_none_text, 0, 0, 0);
	// END NONE

	// START VLDP
	m_vldp_box = new wxStaticBox(panel, -1, "Virtual Laserdisc Player Options");
	m_vldp_sizer = new wxStaticBoxSizer(m_vldp_box, wxVERTICAL);

	m_vldp_text = new wxStaticText(panel, -1, _T("MPEG Framefile Location"));
	m_vldp_framefile = new wxTextCtrl(panel, -1, "", wxPoint(-1, -1), wxSize(350, -1));
	m_vldp_framefile->SetToolTip("The DAPHNE online documentation explains what a framefile is.");
	m_vldp_browse = new wxButton(panel, ID_BROWSE_FRAMEFILE, "Browse");
	m_vldp_blank_on_skips = new wxCheckBox(panel, -1, "Blank on skips");
	m_vldp_blank_on_skips->SetToolTip("This blanks the screen on every skip.\nThis is not authentic behavior, more of a gimmick.");
	m_vldp_blank_on_skips->Disable();
	m_vldp_blank_on_searches = new wxCheckBox(panel, -1, "Blank on searches");
	m_vldp_blank_on_searches->SetToolTip("This blanks the screen on every search.\nThis is authentic behavior.");
	m_vldp_blank_on_searches->Disable();
	m_disable_hwaccel = new wxCheckBox(panel, -1, "Disable MPEG hardware acceleration");
	m_disable_hwaccel->SetToolTip(_T("Disabling MPEG acceleration is safer but slower."));
	m_disable_hwaccel->Disable();

	m_vldp_sizer->Add(m_vldp_text, 0, wxTOP, 5);
	m_vldp_sizer->Add(m_vldp_framefile, 0, 0, 0);
	m_vldp_sizer->Add(m_vldp_browse, 0, wxTOP, 5);
	m_vldp_sizer->Add(0, 15, 0);	// spacer
	m_vldp_sizer->Add(m_disable_hwaccel, 0, wxTOP, 5);
	m_vldp_sizer->Add(m_vldp_blank_on_skips, 0, wxTOP, 5);
	m_vldp_sizer->Add(m_vldp_blank_on_searches, 0, wxTOP, 5);
	// END VLDP

	// START LDP
	m_ldp_box = new wxStaticBox(panel, -1, "LDP Options");
	m_ldp_sizer = new wxStaticBoxSizer(m_ldp_box, wxVERTICAL);

	wxStaticText *text = new wxStaticText(panel, -1, "Serial Port");
	wxString choices[] = 
	{
		"Port 1 (COM1)",
		"Port 2 (COM2)",
	};
	int choices_count = sizeof(choices) / sizeof(wxString);
	m_serial_port = new wxComboBox(panel, -1, choices[0], wxDefaultPosition, wxDefaultSize, choices_count, choices, wxCB_READONLY);
	m_serial_port->Disable();

	wxFlexGridSizer *flex_sizer = new wxFlexGridSizer(2, 2, 15, 5);
	flex_sizer->Add(text, 0, 0, 0);
	flex_sizer->Add(m_serial_port, 0, 0, 0);

	text = new wxStaticText(panel, -1, "Baud Rate");
	wxString choices2[] =
	{
		"9600",
		"4800",
	};
	choices_count = sizeof(choices2) / sizeof(wxString);
	m_baud_rate = new wxComboBox(panel, -1, choices2[0], wxDefaultPosition, wxDefaultSize, choices_count, choices2, wxCB_READONLY);
	m_baud_rate->Disable();

	flex_sizer->Add(text, 0, 0, 0);
	flex_sizer->Add(m_baud_rate, 0, 0, 0);
	m_ldp_sizer->Add(flex_sizer, 0, wxTOP, 15);

	// END LDP

	m_ldp_main_sizer = new wxBoxSizer(wxVERTICAL);
	m_ldp_main_sizer->Add(0, 10, 1);
	m_ldp_main_sizer->Add(player_type_text, 0, wxALIGN_CENTER_HORIZONTAL, 0);
	m_ldp_main_sizer->Add(expand_ldp_list, 0, wxALIGN_CENTER_HORIZONTAL, 0);
	m_ldp_main_sizer->Add(m_none_sizer, 0, wxTOP | wxALIGN_CENTER_HORIZONTAL, 10);
	m_ldp_main_sizer->Add(m_vldp_sizer, 0, wxTOP | wxALIGN_CENTER_HORIZONTAL, 10);
	m_ldp_main_sizer->Add(m_ldp_sizer, 0, wxTOP | wxALIGN_CENTER_HORIZONTAL, 10);

	m_centering_sizer = new wxBoxSizer(wxHORIZONTAL);
	m_centering_sizer->Add(0, 0, 1);
	m_centering_sizer->Add(m_ldp_main_sizer, 0, 0, 0);
	m_centering_sizer->Add(0, 0, 1);

	panel->SetSizer(m_centering_sizer);

	// restore vars here
	m_player_type->SetSelection(m_the_vars->ldp_type_index);	// choose element that was previously selected ...

	return panel;
}

wxPanel *MyConfig::CreateVideoPage()
{
	wxPanel *panel = new wxPanel(this);

	wxCheckBox *fullscreen = new wxCheckBox(panel, -1, "Run Fullscreen");
	fullscreen->Disable();

	wxStaticText *screen_size_text = new wxStaticText(panel, -1, "Screen Size");
	wxString choices[] = 
	{
		"(default)",
		"640x480",
		"720x480",
		"800x600",
		"1024x768",
		"1152x864",
		"1280x1024",
		"1600x1200"
	};

	int choices_count = sizeof(choices) / sizeof(wxString);
	m_screen_size = new wxComboBox(panel, -1, choices[0], wxDefaultPosition, wxDefaultSize, choices_count, choices, wxCB_READONLY);
	m_screen_size->Disable();

	wxBoxSizer *screen_size_sizer = new wxBoxSizer(wxHORIZONTAL);
	screen_size_sizer->Add(screen_size_text, 0, wxRIGHT, 5);
	screen_size_sizer->Add(m_screen_size, 0, 0, 0);
	
	wxBoxSizer *sizer = new wxBoxSizer(wxVERTICAL);
	sizer->Add(screen_size_sizer, 0, wxTOP, 0);
	sizer->Add(fullscreen, 0, wxTOP, 10);

	wxBoxSizer *center = new wxBoxSizer(wxHORIZONTAL);
	center->Add(0, 0, 1);
	center->Add(sizer, 0, wxALIGN_CENTER_VERTICAL, 0);
	center->Add(0, 0, 1);

	panel->SetSizer(center);

	return panel;
}

wxPanel *MyConfig::CreateAudioPage()
{
	wxPanel *panel = new wxPanel(this);

	m_disable_sound = new wxCheckBox(panel, -1, "Disable Sound");
	m_disable_sound->Disable();
	m_sound_buffer_modify = new wxCheckBox(panel, -1, "Sound Buffer Size");
	m_sound_buffer_modify->Disable();

	wxBoxSizer *sizer = new wxBoxSizer(wxVERTICAL);
	sizer->Add(m_disable_sound, 0, wxTOP, 0);
	sizer->Add(m_sound_buffer_modify, 0, wxTOP, 10);

	wxBoxSizer *center = new wxBoxSizer(wxHORIZONTAL);
	center->Add(0, 0, 1);
	center->Add(sizer, 0, wxALIGN_CENTER_VERTICAL, 0);
	center->Add(0, 0, 1);
	panel->SetSizer(center);

	return panel;
}

wxPanel *MyConfig::CreateInputPage()
{
	wxPanel *panel = new wxPanel(this);

	return panel;
}

wxPanel *MyConfig::CreateAdvancedPage()
{
	wxPanel *panel = new wxPanel(this);

	m_ld_latency_modify = new wxCheckBox(panel, -1, "LD latency adjust");
	m_ld_latency_modify->Disable();
	m_dont_report_stats = new wxCheckBox(panel, -1, "Don't Report Stats");
	m_dont_report_stats->Disable();
	m_disable_crc_check = new wxCheckBox(panel, -1, "Disable ROM CRC check");
	m_disable_crc_check->Disable();
	m_display_cmd_line = new wxCheckBox(panel, -1, "Display command line before launching");
	m_display_cmd_line->Disable();
	wxStaticText *custom_settings_text = new wxStaticText(panel, -1, "Custom Settings");
	m_custom_settings = new wxTextCtrl(panel, -1, "", wxDefaultPosition, wxSize(200, -1));
	m_custom_settings->Disable();

	wxBoxSizer *custom_settings_sizer = new wxBoxSizer(wxHORIZONTAL);
	custom_settings_sizer->Add(custom_settings_text, 0, wxRIGHT, 5);
	custom_settings_sizer->Add(m_custom_settings, 0, wxEXPAND, 0);

	wxBoxSizer *main_sizer = new wxBoxSizer(wxVERTICAL);
	int space = 10;
	main_sizer->Add(m_ld_latency_modify, 0, 0, 0);
	main_sizer->Add(m_dont_report_stats, 0, wxTOP, space);
	main_sizer->Add(m_disable_crc_check, 0, wxTOP, space);
	main_sizer->Add(m_display_cmd_line, 0, wxTOP, space);
	main_sizer->Add(custom_settings_sizer, 0, wxTOP, space);

	wxBoxSizer *centering_sizer = new wxBoxSizer(wxHORIZONTAL);
	centering_sizer->Add(0, 0, 1);
	centering_sizer->Add(main_sizer, 0, wxALIGN_CENTER_VERTICAL, 0);
	centering_sizer->Add(0, 0, 1);

	panel->SetSizer(centering_sizer);

	return panel;
}

//////////////////////

ini::ini()
: m_ini_loaded(false), m_buf(NULL), m_buf_size(0), m_buf_index(0)
{
}

ini::~ini()
{
	// de-allocate buffer if necessary
	if (m_buf)
	{
		delete [] m_buf;
		m_buf = NULL;
	}
}

void ini::load_ini()
{
	FILE *F;
	struct stat filestat;

	F = fopen("DaphneLoaderInfo.ini", "rb");

	// if we were able to open the file
	if (F)
	{
		fstat(fileno(F), &filestat);	// get size of file
		m_buf = new unsigned char[filestat.st_size];	// allocate memory to hold file

		// if allocation is successful
		if (m_buf)
		{
			// if we are able to read in the whole file, then consider the INI loaded
			if (fread(m_buf, 1, filestat.st_size, F) == (size_t) filestat.st_size)
			{
				m_ini_loaded = true;
				m_buf_size = filestat.st_size;
			}
		}

		fclose(F);
	}
}

// returns true if the .INI file was loaded successfully
bool ini::ini_loaded()
{
	return m_ini_loaded;
}

// returns true if the key was found and if the index has been moved to the beginning of the line
// directly below the key
// returns false if the key wasn't found (and the index has not been changed)
bool ini::find_key(const char *keyname)
{
	unsigned int index = 0;
	bool result = false;
	
	// go through entire buffer looking for match
	while (index < m_buf_size)
	{
		// if we found a match, check to make sure that it's surrounded in []'s
		if (strncmp(keyname, (const char *) &m_buf[index], strlen(keyname)) == 0)
		{
			if ((index > 0) && (m_buf[index-1] == '[') && (m_buf[index + strlen(keyname)] == ']'))
			{
				m_buf_index = index;
				go_to_next_line();
				result = true;
			}
		}

		index++;
	}

	return result;
}

// moves m_buf_index to the next line
void ini::go_to_next_line()
{
	// chop off anything before the newline characters
	while ((m_buf[m_buf_index] != 0xA) && (m_buf[m_buf_index] != 0xD))
	{
		m_buf_index++;
	}

	// now chop off the newline chars
	while ((m_buf[m_buf_index] == 0xA) || (m_buf[m_buf_index] == 0xD))
	{
		m_buf_index++;
	}

	// we're not at the beginning of the next line
}

// returns a pointer to a string containing everything to the left of an equals character
const char *ini::get_left_of_equals()
{
	static string result;
	int src_index = m_buf_index;

	result = "";
	while (m_buf[src_index] != '=')
	{
		result += m_buf[src_index];
		src_index++;
	}

	go_to_next_line();

	remove_trailing_whitespace(result);
	return result.c_str();
}

// returns a pointer to a string containing everything to the right of an equals character
const char *ini::get_right_of_equals()
{
	static string result;
	int src_index = m_buf_index;

	result = "";

	// go to the = sign
	while (m_buf[src_index] != '=')
	{
		src_index++;
	}
	src_index++;	// move one character passed the equals sign

	// fill up result until we get to the end of the line
	while ((m_buf[src_index] != 0xD) && (m_buf[src_index] != 0xA))
	{
		result += m_buf[src_index];
		src_index++;
	}

	go_to_next_line();

	remove_trailing_whitespace(result);
	return result.c_str();
}

// returns the value of a variable (ie if the line is "Blah=Gwerk" then it would return Gwerk
// if you searched for Blah)
// returns NULL if it can't find the variable, or if the variable is empty
// This does NOT change the current position within the .INI, so you can retrieve multiple vars without repositioning ...
const char *ini::get_var_value(const char *var)
{
	const char *return_val = NULL;
	static string result;
	unsigned int temp_buf_index = m_buf_index;	// this gets restored when we return ...

	// skip lines until we find the correct line
	// WARNING : this is dangerous if var does not exist at all
	while (strncmp((const char *) &m_buf[m_buf_index], var, strlen(var)) != 0)
	{
		go_to_next_line();
	}

	int src_index = m_buf_index, dst_index = 0;

	// go to the = sign
	while (m_buf[src_index] != '=')
	{
		src_index++;
	}
	src_index++;	// move one character passed the equals sign

	result = "";

	// fill up result until we get to the end of the line
	while ((m_buf[src_index] != 0xD) && (m_buf[src_index] != 0xA))
	{
		result += m_buf[src_index];
		src_index++;
	}

	go_to_next_line();

	// if we are returning any data
	if (result.length() > 0)
	{
		remove_trailing_whitespace(result);
		return_val = result.c_str();
	}

	m_buf_index = temp_buf_index;	// restore value ...

	return return_val;
}

// deletes trailing whitespace from a string
// This is so that string comparisons work correctly (the .INI file has at least one instance of trailing whitespace in it)
void ini::remove_trailing_whitespace(string &s)
{
	// while the end of the string is white space ...
	while ((s.length() > 0) && (isspace(s[s.length()-1])))
	{
		s.erase(s.length()-1, 1);	// erase last character from string ...
	}
}
